<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\City;
use App\Models\Catalogs\Country;
use App\Models\Catalogs\State;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class StateController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results = State::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.states.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $countries = Country::populateSelect()->get()->pluck('name_sat', 'id');
        return view('catalogs.states.create',compact('countries'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.states',
        ], [
            'code.required' => __('catalogs/state.error_code'),
            'code.unique' => __('catalogs/state.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $state = State::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/states');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\State  $state
     * @return \Illuminate\Http\Response
     */
    public function show(State $state)
    {
        return redirect('/catalogs/states');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\State  $state
     * @return \Illuminate\Http\Response
     */
    public function edit(State $state)
    {
        $countries = Country::populateSelect()->get()->pluck('name_sat', 'id');
        return view('catalogs.states.edit', compact('state','countries'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\State  $state
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, State $state)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.states')->ignore($state->id)
            ],
        ], [
            'code.required' => __('catalogs/state.error_code'),
            'code.unique' => __('catalogs/state.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $state->fill($request->only([
            'updated_uid',
            'name',
            'country_id',
            'sort_order',
            'status',
            'code',
        ]));

        //Guardar
        //Registro principal
        $state->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/states');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\State  $state
     * @return \Illuminate\Http\Response
     */
    public function destroy(State $state)
    {
        //Logica
        $state->updated_uid = \Auth::user()->id;
        $state->status = 0;
        $state->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/states');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
            'country_id' => 'required|integer',
        ], [
            'name.*' => __('catalogs/state.error_name'),
            'country_id.*' => __('catalogs/state.error_country_id'),
        ]);
    }

    /**
     * Lista de estados
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function getCities(Request $request)
    {
        // Variables
        $id = $request->id;
        $city_id = $request->city_id;

        //Logica
        if ($request->ajax() && !empty($id)) {
            $cities = City::where('state_id', $id)->populateSelect()->pluck('name', 'id');
            $html  = view('layouts.partials.commons.ajax_select')->with('results', $cities)->with('selected', $city_id)->render();
            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')],422);
    }
}
