<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\TariffFraction;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class TariffFractionController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results =TariffFraction::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.tariff_fractions.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.tariff_fractions.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.tariff_fractions',
        ], [
            'code.required' => __('catalogs/tariff_fraction.error_code'),
            'code.unique' => __('catalogs/tariff_fraction.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $tariff_fraction= TariffFraction::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/tariff-fractions');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\TariffFraction  $tariff_fraction
     * @return \Illuminate\Http\Response
     */
    public function show(TariffFraction $tariff_fraction)
    {
        return redirect('/catalogs/tariff-fractions');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\TariffFraction  $tariff_fraction
     * @return \Illuminate\Http\Response
     */
    public function edit(TariffFraction $tariff_fraction)
    {
        return view('catalogs.tariff_fractions.edit', compact('tariff_fraction'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\TariffFraction  $tariff_fraction
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, TariffFraction $tariff_fraction)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.tariff_fractions')->ignore($tariff_fraction->id)
            ],
        ], [
            'code.required' => __('catalogs/tariff_fraction.error_code'),
            'code.unique' => __('catalogs/tariff_fraction.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $tariff_fraction->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $tariff_fraction->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/tariff-fractions');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\TariffFraction  $tariff_fraction
     * @return \Illuminate\Http\Response
     */
    public function destroy(TariffFraction $tariff_fraction)
    {
        //Logica
        $tariff_fraction->updated_uid = \Auth::user()->id;
        $tariff_fraction->status = 0;
        $tariff_fraction->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/tariff-fractions');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/tariff_fraction.error_name'),
        ]);
    }

    /**
     * Autoacompletar select2 de productos sat
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function autocomplete(Request $request)
    {
        //Variables
        $term = $request->term;

        //Logica
        if ($request->ajax() && !empty($term)) {
            $tmp = TariffFraction::filter(['filter_search_select2' => $term])->active()->sortable('code')->limit(16)->get();
            $results = [];
            if ($tmp->isNotEmpty()) {
                foreach ($tmp as $result) {
                    $results[] = [
                        'id' => $result->id,
                        'text' => $result->description_select2,
                        'description' => $result->description_select2
                    ];
                }
            }
            return response()->json($results, 200);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Modal para buscar productos sat
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function modalSearch(Request $request)
    {
        //Variables
        $active_btn_select = (!empty($request->active_btn_select) ? $request->active_btn_select : '');
        $row_origin = (isset($request->row_origin) ? $request->row_origin : '');
        //En caso de estar vacio le asignamos un datos para mostrar los clientes con "a"
        $request->merge(['filter_search' => !empty($request->filter_search) ? $request->filter_search : '']);

        //Logica
        if ($request->ajax()) {
            $results = TariffFraction::filter($request->all())->active()->sortable('code')->limit(10)->get();

            //Busca cliente
            $html_tariff_fractions = view('layouts.partials.tariff_fractions.search', compact('results', 'active_btn_select','row_origin'))->render();

            //modal de buscar
            $html = view('layouts.partials.tariff_fractions.modal_search', compact('html_tariff_fractions', 'active_btn_select','row_origin'))->render();


            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Modal para buscar poductos sat
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function search(Request $request)
    {
        //Variables
        $active_btn_select = (!empty($request->active_btn_select) ? $request->active_btn_select : '');
        $row_origin = (isset($request->row_origin) ? $request->row_origin : '');
        //En caso de estar vacio le asignamos un datos para mostrar los clientes con "a"
        $request->merge(['filter_search' => !empty($request->filter_search) ? $request->filter_search : '']);

        //Logica
        if ($request->ajax()) {

            $results = TariffFraction::filter($request->all())->active()->sortable('code')->limit(10)->get();

            //Busca cliente
            $html = view('layouts.partials.tariff_fractions.search', compact('results', 'active_btn_select','row_origin'))->render();

            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }
}
