<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\TaxRegimen;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class TaxRegimenController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results =TaxRegimen::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.tax_regimens.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.tax_regimens.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.tax_regimens|max:3',
        ], [
            'code.required' => __('catalogs/tax_regimen.error_code'),
            'code.max' => __('catalogs/tax_regimen.error_code_max'),
            'code.unique' => __('catalogs/tax_regimen.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $tax_regimen= TaxRegimen::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/tax-regimens');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\TaxRegimen  $tax_regimen
     * @return \Illuminate\Http\Response
     */
    public function show(TaxRegimen $tax_regimen)
    {
        return redirect('/catalogs/tax-regimens');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\TaxRegimen  $tax_regimen
     * @return \Illuminate\Http\Response
     */
    public function edit(TaxRegimen $tax_regimen)
    {
        return view('catalogs.tax_regimens.edit', compact('tax_regimen'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\TaxRegimen  $tax_regimen
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, TaxRegimen $tax_regimen)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                'max:3',
                Rule::unique('tenant.tax_regimens')->ignore($tax_regimen->id)
            ],
        ], [
            'code.required' => __('catalogs/tax_regimen.error_code'),
            'code.max' => __('catalogs/tax_regimen.error_code_max'),
            'code.unique' => __('catalogs/tax_regimen.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $tax_regimen->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $tax_regimen->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/tax-regimens');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\TaxRegimen  $tax_regimen
     * @return \Illuminate\Http\Response
     */
    public function destroy(TaxRegimen $tax_regimen)
    {
        //Logica
        $tax_regimen->updated_uid = \Auth::user()->id;
        $tax_regimen->status = 0;
        $tax_regimen->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/tax-regimens');
    }

    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/tax_regimen.error_name'),
        ]);
    }
}
