<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\TrafficType;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class TrafficTypeController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results =TrafficType::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.traffic_types.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.traffic_types.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.traffic_types',
        ], [
            'code.required' => __('catalogs/traffic_type.error_code'),
            'code.unique' => __('catalogs/traffic_type.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $traffic_type= TrafficType::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/traffic-types');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\TrafficType  $traffic_type
     * @return \Illuminate\Http\Response
     */
    public function show(TrafficType $traffic_type)
    {
        return redirect('/catalogs/traffic-types');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\TrafficType  $traffic_type
     * @return \Illuminate\Http\Response
     */
    public function edit(TrafficType $traffic_type)
    {
        return view('catalogs.traffic_types.edit', compact('traffic_type'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\TrafficType  $traffic_type
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, TrafficType $traffic_type)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.traffic_types')->ignore($traffic_type->id)
            ],
        ], [
            'code.required' => __('catalogs/traffic_type.error_code'),
            'code.unique' => __('catalogs/traffic_type.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $traffic_type->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $traffic_type->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/traffic-types');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\TrafficType  $traffic_type
     * @return \Illuminate\Http\Response
     */
    public function destroy(TrafficType $traffic_type)
    {
        //Logica
        $traffic_type->updated_uid = \Auth::user()->id;
        $traffic_type->status = 0;
        $traffic_type->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/traffic-types');
    }

    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/traffic_type.error_name'),
        ]);
    }
}
