<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\TrailerType;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class TrailerTypeController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results =TrailerType::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.trailer_types.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.trailer_types.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.trailer_types',
        ], [
            'code.required' => __('catalogs/trailer_type.error_code'),
            'code.unique' => __('catalogs/trailer_type.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $trailer_type= TrailerType::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/trailer-types');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\TrailerType  $trailer_type
     * @return \Illuminate\Http\Response
     */
    public function show(TrailerType $trailer_type)
    {
        return redirect('/catalogs/trailer-types');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\TrailerType  $trailer_type
     * @return \Illuminate\Http\Response
     */
    public function edit(TrailerType $trailer_type)
    {
        return view('catalogs.trailer_types.edit', compact('trailer_type'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\TrailerType  $trailer_type
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, TrailerType $trailer_type)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.trailer_types')->ignore($trailer_type->id)
            ],
        ], [
            'code.required' => __('catalogs/trailer_type.error_code'),
            'code.unique' => __('catalogs/trailer_type.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $trailer_type->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $trailer_type->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/trailer-types');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\TrailerType  $trailer_type
     * @return \Illuminate\Http\Response
     */
    public function destroy(TrailerType $trailer_type)
    {
        //Logica
        $trailer_type->updated_uid = \Auth::user()->id;
        $trailer_type->status = 0;
        $trailer_type->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/trailer-types');
    }

    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/trailer_type.error_name'),
        ]);
    }
}
