<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\TransferReason;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class TransferReasonController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results =TransferReason::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.transfer_reasons.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.transfer_reasons.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.transfer_reasons',
        ], [
            'code.required' => __('catalogs/transfer_reason.error_code'),
            'code.unique' => __('catalogs/transfer_reason.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $transfer_reason= TransferReason::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/transfer-reasons');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\TransferReason  $transfer_reason
     * @return \Illuminate\Http\Response
     */
    public function show(TransferReason $transfer_reason)
    {
        return redirect('/catalogs/transfer-reasons');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\TransferReason  $transfer_reason
     * @return \Illuminate\Http\Response
     */
    public function edit(TransferReason $transfer_reason)
    {
        return view('catalogs.transfer_reasons.edit', compact('transfer_reason'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\TransferReason  $transfer_reason
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, TransferReason $transfer_reason)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.transfer_reasons')->ignore($transfer_reason->id)
            ],
        ], [
            'code.required' => __('catalogs/transfer_reason.error_code'),
            'code.unique' => __('catalogs/transfer_reason.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $transfer_reason->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $transfer_reason->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/transfer-reasons');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\TransferReason  $transfer_reason
     * @return \Illuminate\Http\Response
     */
    public function destroy(TransferReason $transfer_reason)
    {
        //Logica
        $transfer_reason->updated_uid = \Auth::user()->id;
        $transfer_reason->status = 0;
        $transfer_reason->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/transfer-reasons');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/transfer_reason.error_name'),
        ]);
    }
}
