<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\TransportAerial;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class TransportAerialController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results =TransportAerial::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.transport_aerials.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.transport_aerials.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.transport_aerials',
        ], [
            'code.required' => __('catalogs/transport_aerial.error_code'),
            'code.unique' => __('catalogs/transport_aerial.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $transport_aerial= TransportAerial::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/transport-aerials');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\TransportAerial  $transport_aerial
     * @return \Illuminate\Http\Response
     */
    public function show(TransportAerial $transport_aerial)
    {
        return redirect('/catalogs/transport-aerials');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\TransportAerial  $transport_aerial
     * @return \Illuminate\Http\Response
     */
    public function edit(TransportAerial $transport_aerial)
    {
        return view('catalogs.transport_aerials.edit', compact('transport_aerial'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\TransportAerial  $transport_aerial
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, TransportAerial $transport_aerial)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.transport_aerials')->ignore($transport_aerial->id)
            ],
        ], [
            'code.required' => __('catalogs/transport_aerial.error_code'),
            'code.unique' => __('catalogs/transport_aerial.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $transport_aerial->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
            'nationality',
            'designator',
        ]));

        //Guardar
        //Registro principal
        $transport_aerial->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/transport-aerials');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\TransportAerial  $transport_aerial
     * @return \Illuminate\Http\Response
     */
    public function destroy(TransportAerial $transport_aerial)
    {
        //Logica
        $transport_aerial->updated_uid = \Auth::user()->id;
        $transport_aerial->status = 0;
        $transport_aerial->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/transport-aerials');
    }

    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/transport_aerial.error_name'),
        ]);
    }
}
