<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\TransportFigure;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class TransportFigureController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results =TransportFigure::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.transport_figures.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.transport_figures.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.transport_figures',
        ], [
            'code.required' => __('catalogs/transport_figure.error_code'),
            'code.unique' => __('catalogs/transport_figure.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $transport_figure= TransportFigure::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/transport-figures');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\TransportFigure  $transport_figure
     * @return \Illuminate\Http\Response
     */
    public function show(TransportFigure $transport_figure)
    {
        return redirect('/catalogs/transport-figures');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\TransportFigure  $transport_figure
     * @return \Illuminate\Http\Response
     */
    public function edit(TransportFigure $transport_figure)
    {
        return view('catalogs.transport_figures.edit', compact('transport_figure'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\TransportFigure  $transport_figure
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, TransportFigure $transport_figure)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.transport_figures')->ignore($transport_figure->id)
            ],
        ], [
            'code.required' => __('catalogs/transport_figure.error_code'),
            'code.unique' => __('catalogs/transport_figure.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $transport_figure->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $transport_figure->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/transport-figures');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\TransportFigure  $transport_figure
     * @return \Illuminate\Http\Response
     */
    public function destroy(TransportFigure $transport_figure)
    {
        //Logica
        $transport_figure->updated_uid = \Auth::user()->id;
        $transport_figure->status = 0;
        $transport_figure->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/transport-figures');
    }

    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/transport_figure.error_name'),
        ]);
    }
}
