<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\TransportPart;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class TransportPartController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results =TransportPart::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.transport_parts.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.transport_parts.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.transport_parts',
        ], [
            'code.required' => __('catalogs/transport_part.error_code'),
            'code.unique' => __('catalogs/transport_part.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $transport_part= TransportPart::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/transport-parts');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\TransportPart  $transport_part
     * @return \Illuminate\Http\Response
     */
    public function show(TransportPart $transport_part)
    {
        return redirect('/catalogs/transport-parts');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\TransportPart  $transport_part
     * @return \Illuminate\Http\Response
     */
    public function edit(TransportPart $transport_part)
    {
        return view('catalogs.transport_parts.edit', compact('transport_part'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\TransportPart  $transport_part
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, TransportPart $transport_part)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.transport_parts')->ignore($transport_part->id)
            ],
        ], [
            'code.required' => __('catalogs/transport_part.error_code'),
            'code.unique' => __('catalogs/transport_part.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $transport_part->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $transport_part->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/transport-parts');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\TransportPart  $transport_part
     * @return \Illuminate\Http\Response
     */
    public function destroy(TransportPart $transport_part)
    {
        //Logica
        $transport_part->updated_uid = \Auth::user()->id;
        $transport_part->status = 0;
        $transport_part->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/transport-parts');
    }

    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/transport_part.error_name'),
        ]);
    }
}
