<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\UnitMeasure;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class UnitMeasureController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results = UnitMeasure::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.unit_measures.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.unit_measures.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.unit_measures|max:3',
        ], [
            'code.required' => __('catalogs/unit_measure.error_code'),
            'code.max' => __('catalogs/unit_measure.error_code_max'),
            'code.unique' => __('catalogs/unit_measure.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['decimal_place' => (int)$request->decimal_place]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $unit_measure = UnitMeasure::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/unit-measures');
    }

    /**
     * Display the specified resource.
     *
     * @param UnitMeasure $unit_measure
     * @return \Illuminate\Http\Response
     */
    public function show(UnitMeasure $unit_measure)
    {
        return redirect('/catalogs/unit-measures');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param UnitMeasure $unit_measure
     * @return \Illuminate\Http\Response
     */
    public function edit(UnitMeasure $unit_measure)
    {
        return view('catalogs.unit_measures.edit', compact('unit_measure'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @param UnitMeasure $unit_measure
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, UnitMeasure $unit_measure)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                'max:3',
                Rule::unique('tenant.unit_measures')->ignore($unit_measure->id)
            ],
        ], [
            'code.required' => __('catalogs/unit_measure.error_code'),
            'code.max' => __('catalogs/unit_measure.error_code_max'),
            'code.unique' => __('catalogs/unit_measure.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['decimal_place' => (int)$request->decimal_place]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $unit_measure->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'decimal_place',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $unit_measure->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/unit-measures');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param UnitMeasure $unit_measure
     * @return \Illuminate\Http\Response
     */
    public function destroy(UnitMeasure $unit_measure)
    {
        //Logica
        $unit_measure->updated_uid = \Auth::user()->id;
        $unit_measure->status = 0;
        $unit_measure->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/unit-measures');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/unit_measure.error_name'),
        ]);
    }

    /**
     * Autoacompletar select2 de unidades de medida
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function autocomplete(Request $request)
    {
        //Variables
        $term = $request->term;

        //Logica
        if ($request->ajax() && !empty($term)) {
            $tmp = UnitMeasure::filter(['filter_search_select2' => $term])->active()->sortable('name')->limit(16)->get();
            $results = [];
            if ($tmp->isNotEmpty()) {
                foreach ($tmp as $result) {
                    $results[] = [
                        'id' => $result->id,
                        'text' => $result->description_select2,
                        'description' => $result->description_select2
                    ];
                }
            }
            return response()->json($results, 200);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Modal para buscar unidades de medida
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function modalSearch(Request $request)
    {
        //Variables
        $active_btn_select = (!empty($request->active_btn_select) ? $request->active_btn_select : '');
        $row_origin = (isset($request->row_origin) ? $request->row_origin : '');
        $precise_code = (isset($request->precise_code) ? $request->precise_code : '');
        //En caso de estar vacio le asignamos un datos para mostrar los clientes con "a"
        if(!empty($precise_code)){
            $request->merge(['filter_code' => !empty($request->filter_search) ? $request->filter_search : '']);
            $request->merge(['filter_search' => '']);
        }else{
            $request->merge(['filter_search' => !empty($request->filter_search) ? $request->filter_search : '']);
        }

        //Logica
        if ($request->ajax()) {
            $results = UnitMeasure::filter($request->all())->active()->sortable('name')->limit(10)->get();

            //Busca cliente
            $html_unit_measures = view('layouts.partials.unit_measures.search', compact('results', 'active_btn_select','row_origin'))->render();

            //modal de buscar
            $html = view('layouts.partials.unit_measures.modal_search', compact('html_unit_measures', 'active_btn_select','row_origin'))->render();


            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Modal para buscar unidades de medida
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function search(Request $request)
    {
        //Variables
        $active_btn_select = (!empty($request->active_btn_select) ? $request->active_btn_select : '');
        $row_origin = (isset($request->row_origin) ? $request->row_origin : '');
        $precise_code = (isset($request->precise_code) ? $request->precise_code : '');
        //En caso de estar vacio le asignamos un datos para mostrar los clientes con "a"
        if(!empty($precise_code)){
            $request->merge(['filter_code' => !empty($request->filter_search) ? $request->filter_search : '']);
            $request->merge(['filter_search' => '']);
        }else{
            $request->merge(['filter_search' => !empty($request->filter_search) ? $request->filter_search : '']);
        }

        //Logica
        if ($request->ajax()) {

            $results = UnitMeasure::filter($request->all())->active()->sortable('name')->limit(10)->get();

            //Busca cliente
            $html = view('layouts.partials.unit_measures.search', compact('results', 'active_btn_select','row_origin'))->render();

            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }
}
