<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\UomCustom;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class UomCustomController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results =UomCustom::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.uom_customs.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.uom_customs.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.uom_customs',
        ], [
            'code.required' => __('catalogs/uom_custom.error_code'),
            'code.unique' => __('catalogs/uom_custom.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $uom_custom= UomCustom::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/uom-customs');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\UomCustom  $uom_custom
     * @return \Illuminate\Http\Response
     */
    public function show(UomCustom $uom_custom)
    {
        return redirect('/catalogs/uom-customs');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\UomCustom  $uom_custom
     * @return \Illuminate\Http\Response
     */
    public function edit(UomCustom $uom_custom)
    {
        return view('catalogs.uom_customs.edit', compact('uom_custom'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\UomCustom  $uom_custom
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, UomCustom $uom_custom)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.uom_customs')->ignore($uom_custom->id)
            ],
        ], [
            'code.required' => __('catalogs/uom_custom.error_code'),
            'code.unique' => __('catalogs/uom_custom.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $uom_custom->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $uom_custom->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/uom-customs');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\UomCustom  $uom_custom
     * @return \Illuminate\Http\Response
     */
    public function destroy(UomCustom $uom_custom)
    {
        //Logica
        $uom_custom->updated_uid = \Auth::user()->id;
        $uom_custom->status = 0;
        $uom_custom->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/uom-customs');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/uom_custom.error_name'),
        ]);
    }


    /**
     * Autoacompletar select2 de productos sat
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function autocomplete(Request $request)
    {
        //Variables
        $term = $request->term;

        //Logica
        if ($request->ajax() && !empty($term)) {
            $tmp = UomCustom::filter(['filter_search_select2' => $term])->active()->sortable('code')->limit(16)->get();
            $results = [];
            if ($tmp->isNotEmpty()) {
                foreach ($tmp as $result) {
                    $results[] = [
                        'id' => $result->id,
                        'text' => $result->description_select2,
                        'description' => $result->description_select2
                    ];
                }
            }
            return response()->json($results, 200);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Modal para buscar productos sat
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function modalSearch(Request $request)
    {
        //Variables
        $active_btn_select = (!empty($request->active_btn_select) ? $request->active_btn_select : '');
        $row_origin = (isset($request->row_origin) ? $request->row_origin : '');
        //En caso de estar vacio le asignamos un datos para mostrar los clientes con "a"
        $request->merge(['filter_search' => !empty($request->filter_search) ? $request->filter_search : '']);

        //Logica
        if ($request->ajax()) {
            $results = UomCustom::filter($request->all())->active()->sortable('code')->limit(10)->get();

            //Busca cliente
            $html_uom_customs = view('layouts.partials.uom_customs.search', compact('results', 'active_btn_select','row_origin'))->render();

            //modal de buscar
            $html = view('layouts.partials.uom_customs.modal_search', compact('html_uom_customs', 'active_btn_select','row_origin'))->render();


            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Modal para buscar poductos sat
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function search(Request $request)
    {
        //Variables
        $active_btn_select = (!empty($request->active_btn_select) ? $request->active_btn_select : '');
        $row_origin = (isset($request->row_origin) ? $request->row_origin : '');
        //En caso de estar vacio le asignamos un datos para mostrar los clientes con "a"
        $request->merge(['filter_search' => !empty($request->filter_search) ? $request->filter_search : '']);

        //Logica
        if ($request->ajax()) {

            $results = UomCustom::filter($request->all())->active()->sortable('code')->limit(10)->get();

            //Busca cliente
            $html = view('layouts.partials.uom_customs.search', compact('results', 'active_btn_select','row_origin'))->render();

            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }
}
