<?php

namespace App\Http\Controllers\Sales;

use App\Exports\CustomersExport;
use App\Helpers\Helper;
use App\Models\Catalogs\Bank;
use App\Models\Catalogs\CfdiUse;
use App\Models\Catalogs\Country;
use App\Models\Catalogs\Currency;
use App\Models\Catalogs\IeduNivelEducativo;
use App\Models\Catalogs\PaymentMethod;
use App\Models\Catalogs\PaymentTerm;
use App\Models\Catalogs\PaymentWay;
use App\Models\Sales\Customer;
use App\Models\Sales\CustomerBankAccount;
use App\Models\Sales\CustomerIedu;
use App\Models\Sales\Salesperson;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\ConnectException;
use GuzzleHttp\Exception\ServerException;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;
use GuzzleHttp\Psr7;
use GuzzleHttp\Exception\RequestException;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\CustomerTemplateImportExport;
use App\Imports\CustomerTemplateBeforeImportImport;
use App\Imports\CustomerTemplateImportImport;
use App\Models\Catalogs\TaxRegimen;

class CustomerController extends Controller
{
    private $import_results = [];

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {

        $this->import_results [] = (object)[
            'name' => 'Juan Perez Cruz',
            'taxid' => 'XAXX010101000',
            'tax_regimen' => 'Incorporación Fiscal',
            'payment_term' => 'Contado',
            'payment_way' => 'Transferencia electrónica de fondos',
            'payment_method' => 'PUE',
            'cfdi_use' => 'Gastos en general',
            'salesperson' => 'Angel Hernandez Cruz',
            'email' => 'corre@gmail.com',
            'phone' => '4421096112',
            'address_1' => 'Av. Juarez',
            'address_2' => '10',
            'address_3' => '5A',
            'address_4' => 'Centro',
            'country' => 'México',
            'state' => 'Queretaro',
            'city' => 'Queretaro',
            'postcode' => '76000',
        ];
        $this->import_results [] = (object)[
            'name' => 'Yamiss Soft SA de CV',
            'taxid' => 'XAXX010101000',
            'tax_regimen' => 'General de Ley Personas Morales',
            'payment_term' => 'Contado',
            'payment_way' => 'Efectivo',
            'payment_method' => 'PPD',
            'cfdi_use' => 'Gastos en general',
            'salesperson' => '',
            'email' => '',
            'phone' => '4428712354',
            'address_1' => 'Av. Morelos',
            'address_2' => '13',
            'address_3' => '',
            'address_4' => 'Centro',
            'country' => 'México',
            'state' => 'Hidalgo',
            'city' => 'Actopan',
            'postcode' => '',
        ];
        $this->import_results [] = (object)[
            'name' => 'Maria Lugo Juarez',
            'taxid' => 'XAXX010101000',
            'tax_regimen' => '',
            'payment_term' => 'Contado',
            'payment_way' => '',
            'payment_method' => '',
            'cfdi_use' => '',
            'salesperson' => '',
            'email' => '',
            'phone' => '',
            'address_1' => '',
            'address_2' => '',
            'address_3' => '',
            'address_4' => '',
            'country' => 'México',
            'state' => '',
            'city' => '',
            'postcode' => '',
        ];
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);
        if (!isset($request->filter_status) && $request->filter_status != 'all') {
            $request->request->add([
                'filter_status' => '1'
            ]);
        }

        //Consulta
        $results = Customer::filter($request->all());
        if(isset($request->filter_status) && $request->filter_status != 'all'){
            $results = $results->where('customers.status','=',(int)$request->filter_status);
        }
        $results = $results->with('paymentTerm')
            ->with('country')
            ->sortable('name')
            ->paginate($limit);

        //Vista
        return view('sales.customers.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $payment_terms = PaymentTerm::populateSelect()->pluck('name', 'id');
        $payment_ways = PaymentWay::populateSelect()->get()->pluck('name_sat', 'id');
        $payment_methods = PaymentMethod::populateSelect()->get()->pluck('name_sat', 'id');
        $cfdi_uses = CfdiUse::populateSelect()->get()->pluck('name_sat', 'id');
        $salespersons = Salesperson::populateSelect()->pluck('name', 'id');
        $countries = Country::populateSelect()->get()->pluck('name_sat', 'id');
        $banks = Bank::populateSelect()->pluck('name', 'id');
        $currencies = Currency::populateSelect()->get()->pluck('name_sat', 'id');
        //Complemento IEDU
        $iedu_nivel_educativos = IeduNivelEducativo::populateSelect()->pluck('name', 'id');
        $tax_regimens = TaxRegimen::populateSelect()->get()->pluck('name_sat', 'id');
        return view('sales.customers.create',
            compact('payment_terms', 'payment_ways', 'payment_methods', 'cfdi_uses', 'salespersons', 'countries',
                'banks', 'currencies', 'iedu_nivel_educativos','tax_regimens'));
    }

    /**
     * Modal para alta de clientes
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function modalCreate(Request $request)
    {
        //Variables


        //Logica
        if ($request->ajax()) {

            $payment_terms = PaymentTerm::populateSelect()->pluck('name', 'id');
            $payment_ways = PaymentWay::populateSelect()->get()->pluck('name_sat', 'id');
            $payment_methods = PaymentMethod::populateSelect()->get()->pluck('name_sat', 'id');
            $cfdi_uses = CfdiUse::populateSelect()->get()->pluck('name_sat', 'id');
            $countries = Country::populateSelect()->get()->pluck('name_sat', 'id');
            $tax_regimens = TaxRegimen::populateSelect()->get()->pluck('name_sat', 'id');

            //modal de buscar
            $html = view('layouts.partials.customers.modal_create',
                compact('payment_terms', 'payment_ways', 'payment_methods', 'cfdi_uses', 'countries','tax_regimens'))->render();


            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     * @throws ValidationException
     */
    public function store(Request $request)
    {
        //Validacion
        $request->merge(['name' => trim(str_replace('  ', ' ', $request->name))]);
        $request->merge(['taxid' => strtoupper(trim($request->taxid))]);
        $this->validation($request);

        //Logica
        $company = Helper::firstCompany(); //Empresa
        if(!empty(setting('customers_per_taxid'))){
            $company = Helper::defaultCompany(); //Empresa
        }

        //Validaciones manuales
        $validator = \Validator::make([], []);
        //Validamos que RFC no este registrado previamente
        if ($request->taxid != 'XAXX010101000' && $request->taxid != 'XEXX010101000') {
            $customer_taxid = Customer::where('taxid', '=', $request->taxid)->where('company_id', '=', $company->id)->first();
            if (!empty($customer_taxid)) {
                $validator->after(function ($validator) {
                    $validator->errors()->add('taxid', __('sales/customer.error_taxid_unique'));
                });
            }
        }
        //Validacion de RFC en la lista de LCO insigna
        try{
            if(!empty(setting('validate_taxid_insigna')) && $request->taxid != 'XAXX010101000' && $request->taxid != 'XEXX010101000'){
                if(!Helper::validateInsignaLcoRfc($request->taxid)){
                    $validator->after(function ($validator) {
                        $validator->errors()->add('taxid', __('sales/customer.error_taxid_lco'));
                    });
                }
            }
        } catch (\Exception $e) {
            if ($request->ajax()) {
                return response()->json(['error' => $e->getMessage()], $e->getCode());
            }
            flash($e->getMessage())->error();
            return back()->withInput();
        }
        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        \DB::connection('tenant')->beginTransaction();
        try {

            //Logica
            $request->merge(['created_uid' => \Auth::user()->id]);
            $request->merge(['updated_uid' => \Auth::user()->id]);
            $request->merge(['status' => 1]); //Por default activo
            $request->merge(['company_id' => $company->id]);

            //Guardar
            //Registro principal
            $customer = Customer::create($request->input());

            //Cuentas bancarias
            if (!empty($request->item_bank_account)) {
                foreach ($request->item_bank_account as $key => $result) {
                    $customer_bank_account = CustomerBankAccount::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'customer_id' => $customer->id,
                        'name' => $result['name'],
                        'bank_id' => $result['bank_id'],
                        'currency_id' => $result['currency_id'],
                        'account_number' => $result['account_number'],
                        'sort_order' => $key,
                        'status' => 1,
                    ]);
                }
            }

            //Complemento IEDU
            if (!empty($request->item_iedu)) {
                foreach ($request->item_iedu as $key => $result) {
                    $customer_iedu = CustomerIedu::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'customer_id' => $customer->id,
                        'name' => $result['nombre_alumno'],
                        'nombre_alumno' => $result['nombre_alumno'],
                        'curp' => $result['curp'],
                        'iedu_nivel_educativo_id' => $result['iedu_nivel_educativo_id'],
                        'sort_order' => $key,
                        'status' => 1,
                    ]);
                }
            }

            \DB::connection('tenant')->commit();

            //Si la llamada es por ajax
            if ($request->ajax()) {
                return response()->json($customer, 200);
            }

            //Mensaje
            flash(__('general.text_form_success_add'))->success();

            //Redireccion
            return redirect('/sales/customers');
        } catch (\Exception $e) {
            \DB::connection('tenant')->rollback();
            if ($request->ajax()) {
                return response()->json(['error' => $e->getMessage()], $e->getCode());
            }
            flash($e->getMessage())->error();
            return back()->withInput();
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Sales\Customer $customer
     * @return \Illuminate\Http\Response
     */
    public function show(Customer $customer)
    {
        return redirect('/sales/customers');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Sales\Customer $customer
     * @return \Illuminate\Http\Response
     */
    public function edit(Customer $customer)
    {
        $payment_terms = PaymentTerm::populateSelect()->pluck('name', 'id');
        $payment_ways = PaymentWay::populateSelect()->get()->pluck('name_sat', 'id');
        $payment_methods = PaymentMethod::populateSelect()->get()->pluck('name_sat', 'id');
        $cfdi_uses = CfdiUse::populateSelect()->get()->pluck('name_sat', 'id');
        $salespersons = Salesperson::populateSelect()->pluck('name', 'id');
        $countries = Country::populateSelect()->get()->pluck('name_sat', 'id');
        $banks = Bank::populateSelect()->pluck('name', 'id');
        $currencies = Currency::populateSelect()->get()->pluck('name_sat', 'id');
        //Complemento IEDU
        $iedu_nivel_educativos = IeduNivelEducativo::populateSelect()->pluck('name', 'id');
        $tax_regimens = TaxRegimen::populateSelect()->get()->pluck('name_sat', 'id');
        return view('sales.customers.edit',
            compact('customer', 'payment_terms', 'payment_ways', 'payment_methods', 'cfdi_uses', 'salespersons',
                'countries', 'banks', 'currencies', 'iedu_nivel_educativos','tax_regimens'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param Request $request
     * @param Customer $customer
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     * @throws ValidationException
     */
    public function update(Request $request, Customer $customer)
    {
        //Validacion
        $request->merge(['name' => trim(str_replace('  ', ' ', $request->name))]);
        $request->merge(['taxid' => strtoupper(trim($request->taxid))]);
        $this->validation($request);

        //Validaciones manuales
        $validator = \Validator::make([], []);
        //Validamos que RFC no ste registrado previamente
        if ($request->taxid != 'XAXX010101000' && $request->taxid != 'XEXX010101000') {
            $customer_taxid = Customer::where('id', '<>', $customer->id)->where('taxid', '=', $request->taxid)->where('company_id', '=', $customer->company_id)->first();
            if (!empty($customer_taxid)) {
                $validator->after(function ($validator) {
                    $validator->errors()->add('taxid', __('sales/customer.error_taxid_unique'));
                });
            }
        }
        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        \DB::connection('tenant')->beginTransaction();
        try {

            //Logica
            $request->merge(['updated_uid' => \Auth::user()->id]);
            $request->merge(['status' => !empty($request->status) ? 1 : 0]);
            $customer->fill($request->only([
                'updated_uid',
                'name',
                'taxid',
                'numid',
                'payment_term_id',
                'payment_way_id',
                'payment_method_id',
                'cfdi_use_id',
                'salesperson_id',
                'email',
                'phone',
                'phone_mobile',
                'address_1',
                'address_2',
                'address_3',
                'address_4',
                'address_5',
                'address_6',
                'city_id',
                'state_id',
                'country_id',
                'postcode',
                'comment',
                'sort_order',
                'status',
                'tax_regimen_id'
            ]));

            //Guardar
            //Registro principal
            $customer->save();

            //Cuentas bancarias
            //Elimina
            if (!empty($request->delete_item_bank_account)) {
                foreach ($request->delete_item_bank_account as $key => $result) {
                    //Actualizar status
                    $customer_bank_account = CustomerBankAccount::findOrFail($result);
                    $customer_bank_account->updated_uid = \Auth::user()->id;
                    $customer_bank_account->status = 0;
                    $customer_bank_account->save();
                }
            }
            //Guarda
            if (!empty($request->item_bank_account)) {
                foreach ($request->item_bank_account as $key => $result) {
                    //Datos
                    $data = [
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'customer_id' => $customer->id,
                        'name' => $result['name'],
                        'bank_id' => $result['bank_id'],
                        'currency_id' => $result['currency_id'],
                        'account_number' => $result['account_number'],
                        'sort_order' => $key,
                        'status' => 1,
                    ];
                    //Valida si es registro nuevo o actualizacion
                    if (!empty($result['id'])) {
                        $customer_bank_account = CustomerBankAccount::findOrFail($result['id']);
                        $customer_bank_account->fill(array_only($data, [
                            'updated_uid',
                            'customer_id',
                            'name',
                            'bank_id',
                            'currency_id',
                            'account_number',
                            'sort_order',
                        ]));
                        $customer_bank_account->save();
                    } else {
                        $customer_bank_account = CustomerBankAccount::create($data);
                    }
                }
            }

            //Complemento IEDU
            //Elimina
            if (!empty($request->delete_item_iedu)) {
                foreach ($request->delete_item_iedu as $key => $result) {
                    //Actualizar status
                    $customer_iedu = CustomerIedu::findOrFail($result);
                    $customer_iedu->updated_uid = \Auth::user()->id;
                    $customer_iedu->status = 0;
                    $customer_iedu->save();
                }
            }
            //Guarda
            if (!empty($request->item_iedu)) {
                foreach ($request->item_iedu as $key => $result) {
                    //Datos
                    $data = [
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'customer_id' => $customer->id,
                        'name' => $result['nombre_alumno'],
                        'nombre_alumno' => $result['nombre_alumno'],
                        'curp' => $result['curp'],
                        'iedu_nivel_educativo_id' => $result['iedu_nivel_educativo_id'],
                        'sort_order' => $key,
                        'status' => 1,
                    ];
                    //Valida si es registro nuevo o actualizacion
                    if (!empty($result['id'])) {
                        $customer_iedu = CustomerIedu::findOrFail($result['id']);
                        $customer_iedu->fill(array_only($data, [
                            'updated_uid',
                            'customer_id',
                            'name',
                            'nombre_alumno',
                            'curp',
                            'iedu_nivel_educativo_id',
                            'sort_order',
                        ]));
                        $customer_iedu->save();
                    } else {
                        $customer_iedu = CustomerIedu::create($data);
                    }
                }
            }

            \DB::connection('tenant')->commit();

            //Mensaje
            flash(__('general.text_form_success_edit'))->success();

            //Redireccion
            return redirect('/sales/customers');
        } catch (\Exception $e) {
            \DB::connection('tenant')->rollback();
            flash($e->getMessage())->error();
            return back()->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Sales\Customer $customer
     * @return \Illuminate\Http\Response
     */
    public function destroy(Customer $customer)
    {
        //Logica
        $customer->updated_uid = \Auth::user()->id;
        $customer->status = 0;
        $customer->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/sales/customers');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     * @throws ValidationException
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
            'taxid' => [
                'required',
                'regex:/^[A-Z&Ñ]{3,4}[0-9]{2}(0[1-9]|1[012])(0[1-9]|[12][0-9]|3[01])[A-Z0-9]{2}[0-9A]$/i'
            ],
            //'numid' => 'nullable|required_if:taxid,XEXX010101000',
            'payment_term_id' => 'required|integer',
            //'email' => 'nullable|email',
            'country_id' => 'required|integer',
            'item_bank_account.*.name' => 'required',
            'item_bank_account.*.bank_id' => 'required',
            'item_bank_account.*.currency_id' => 'required',
            'item_bank_account.*.account_number' => [
                'required',
                'regex:/^([0-9]{10,11}|[0-9]{15,16}|[0-9]{18})$/'
            ],
            'item_iedu.*.nombre_alumno' => 'required',
            'item_iedu.*.curp' => [
                'required',
                'regex:/^[A-Z][A,E,I,O,U,X][A-Z]{2}[0-9]{2}[0-1][0-9][0-3][0-9][M,H][A-Z]{2}[B,C,D,F,G,H,J,K,L,M,N,Ñ,P,Q,R,S,T,V,W,X,Y,Z]{3}[0-9,A-Z][0-9]$/'
            ],
            'item_iedu.*.iedu_nivel_educativo_id' => 'required',
        ], [
            'name.*' => __('sales/customer.error_name'),
            'taxid.required' => __('sales/customer.error_taxid'),
            'taxid.regex' => __('sales/customer.error_taxid_format'),
            //'numid.*' => __('sales/customer.error_numid'),
            'payment_term_id.*' => __('sales/customer.error_payment_term_id'),
            //'email.email' => __('sales/customer.error_email_format'),
            'country_id.*' => __('sales/customer.error_country_id'),
            'item_bank_account.*.name.*' => __('sales/customer.error_bank_account_name'),
            'item_bank_account.*.bank_id.*' => __('sales/customer.error_bank_account_bank_id'),
            'item_bank_account.*.currency_id.*' => __('sales/customer.error_bank_account_currency_id'),
            'item_bank_account.*.account_number.required' => __('sales/customer.error_bank_account_account_number'),
            'item_bank_account.*.account_number.regex' => __('sales/customer.error_bank_account_account_number_format'),
            'item_iedu.*.nombre_alumno.*' => __('sales/customer.error_iedu_nombre_alumno'),
            'item_iedu.*.curp.required' => __('sales/customer.error_iedu_curp'),
            'item_iedu.*.curp.regex' => __('sales/customer.error_iedu_curp_format'),
            'item_iedu.*.iedu_nivel_educativo_id.*' => __('sales/customer.error_iedu_iedu_nivel_educativo_id'),
        ]);
        //Validaciones manuales
        $validator = \Validator::make([], []);

        //Valida para clientes extranjeros que seleccionen un pais que no sea mexico
        $country = Country::findOrFail($request->country_id);
        if ($request->taxid == 'XEXX010101000' && $country->code == 'MEX') {
            $validator->after(function ($validator) {
                $validator->errors()->add('country_id', __('sales/customer.error_country_id_not_mex'));
            });
        }

        if(setting('cfdi_version') == 'cfdi40'){
            if (empty($request->tax_regimen_id)) {
                $validator->after(function ($validator) {
                    $validator->errors()->add('tax_regimen_id', __('sales/customer.error_tax_regimen_id'));
                });
            }
            if (empty($request->postcode)) {
                $validator->after(function ($validator) {
                    $validator->errors()->add('postcode', __('sales/customer.error_postcode'));
                });
            }
        }

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }
    }

    /**
     * Obtener registro
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCustomer(Request $request)
    {
        //Variables
        $id = $request->id;

        //Logica
        if ($request->ajax() && !empty($id)) {
            $customer = Customer::findOrFail($id);
            $customer->html_customer_iedus = view('layouts.partials.commons.ajax_select_customer_iedu')->with('results',
                $customer->customerActiveIedus->pluck('name', 'id'))->with('selected', null)->render();
            return response()->json($customer, 200);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Lista de cuentas bancarias
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function getCustomerBankAccounts(Request $request)
    {
        // Variables
        $id = $request->id;
        $customer_bank_account_id = $request->customer_bank_account_id;

        //Logica
        if ($request->ajax() && !empty($id)) {
            $customer_bank_accounts = CustomerBankAccount::where('customer_id', $id)->populateSelect()->pluck('name',
                'id');
            $html = view('layouts.partials.commons.ajax_select')->with('results',
                $customer_bank_accounts)->with('selected', $customer_bank_account_id)->render();
            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Autoacompletar select2 de clientes
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function autocomplete(Request $request)
    {
        //Variables
        $term = $request->term;

        //Logica
        if ($request->ajax() && !empty($term)) {
            $tmp = Customer::filter(['filter_search_select2' => $term])->active()->sortable('name')->limit(16)->get();
            $results = [];
            if ($tmp->isNotEmpty()) {
                foreach ($tmp as $result) {
                    $results[] = [
                        'id' => $result->id,
                        'text' => $result->text_select2,
                        'description' => $result->description_select2
                    ];
                }
            }
            return response()->json($results, 200);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Modal para buscar clientes
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function modalSearch(Request $request)
    {
        //Variables
        $active_btn_select = (!empty($request->active_btn_select) ? $request->active_btn_select : '');
        //En caso de estar vacio le asignamos un datos para mostrar los clientes con "a"
        $request->merge(['filter_search' => !empty($request->filter_search) ? $request->filter_search : '']);

        //Logica
        if ($request->ajax()) {
            $results = Customer::filter($request->all())->active()->sortable('name')->limit(10)->get();

            //Busca cliente
            $html_customers = view('layouts.partials.customers.search',
                compact('results', 'active_btn_select'))->render();

            //modal de buscar
            $html = view('layouts.partials.customers.modal_search',
                compact('html_customers', 'active_btn_select'))->render();


            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Modal para buscar clientes
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function search(Request $request)
    {
        //Variables
        $active_btn_select = (!empty($request->active_btn_select) ? $request->active_btn_select : '');
        //En caso de estar vacio le asignamos un datos para mostrar los clientes con "a"
        $request->merge(['filter_search' => !empty($request->filter_search) ? $request->filter_search : '']);

        //Logica
        if ($request->ajax()) {

            $results = Customer::filter($request->all())->active()->sortable('name')->limit(10)->get();

            //Busca cliente
            $html = view('layouts.partials.customers.search',
                compact('results', 'active_btn_select'))->render();

            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Exportar datos a excel
     *
     * @param Request $request
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function exportToExcel(Request $request)
    {
        while (ob_get_level()) ob_end_clean();
        ob_start();

        return Excel::download(new CustomersExport($request),
            __('sales/customer.document_title') . '-' . config('app.name') . '.xlsx');
    }

    /**
     * Importar viviendas desde archivo
     *
     * @return \Illuminate\Http\Response
     */
    public function createImport()
    {
        //Datos
        $salespersons = Salesperson::populateSelect()->pluck('name', 'id');
        $payment_terms = PaymentTerm::populateSelect()->pluck('name', 'id');
        $payment_ways = PaymentWay::populateSelect()->get()->pluck('name_sat', 'id');
        $payment_methods = PaymentMethod::populateSelect()->get()->pluck('name_sat', 'id');
        $cfdi_uses = CfdiUse::populateSelect()->get()->pluck('name_sat', 'id');
        $countries = Country::populateSelect()->get()->pluck('name_sat', 'id');
        $import_results = $this->import_results;
        $tax_regimens = TaxRegimen::populateSelect()->get()->pluck('name_sat', 'id');

        //Vista
        return view('sales.customers.import',compact('salespersons','payment_terms','payment_ways','payment_methods','cfdi_uses','import_results','countries', 'tax_regimens'));
    }

    /**
     * Descargar plantilla
     *
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function downloadTemplateImport(Request $request)
    {
        //Datos
        $import_results = $this->import_results;
        if(!empty($request->update)){
            $import_results = [];
            $results = Customer::filter([])->sortable('name')->get();
            foreach($results as $result){
                $import_results[] = (object)[
                    'name' => $result->name,
                    'taxid' => $result->taxid,
                    'tax_regimen' => $result->taxRegimen->name ?? '',
                    'payment_term' => $result->paymentTerm->name ?? '',
                    'payment_way' => $result->paymentWay->name ?? '',
                    'payment_method' => $result->paymentMethod->code ?? '',
                    'cfdi_use' => $result->cfdiUse->name ?? '',
                    'salesperson' => $result->salesperson->name ?? '',
                    'email' => $result->email,
                    'phone' => $result->phone,
                    'address_1' => $result->address_1,
                    'address_2' => $result->address_2,
                    'address_3' => $result->address_3,
                    'address_4' => $result->address_4,
                    'country' => $result->country->name ?? '',
                    'state' => $result->state->name ?? '',
                    'city' => $result->city->name ?? '',
                    'postcode' => $result->postcode,
                ];
            }

        }

        //Descargar archivo
        while (ob_get_level()) ob_end_clean();
        ob_start();

        return Excel::download(new CustomerTemplateImportExport($request, $import_results),
            __('sales/customer.text_template_import') . '-' . config('app.name') . '.xlsx',\Maatwebsite\Excel\Excel::XLSX);
    }

    /**
     * Obtener informacion antes de procesar el archivo
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function beforeImport(Request $request)
    {
        //Validaciones
        //Validacion de archivos por extension
        if ($request->hasFile('file_customers_import')) {
            $request->merge(['file_customers_import_ext' => request()->file('file_customers_import')->getClientOriginalExtension()]);
        }
        $validator = \Validator::make($request->all(), [
            'file_customers_import' => 'required|max:2048',
            'file_customers_import_ext' => 'nullable|in:xls,xlsx'
        ], [
            'file_customers_import.*' => __('sales/customer.error_file_customers_import'),
            'file_customers_import_ext.*' => __('sales/customer.error_file_customers_import'),
        ]);
        //Validaciones manuales
        //$validator = \Validator::make([], []);

        //Errores
        if ($validator->fails()) {
            $errors = '';
            foreach ($validator->errors()->all() as $message) {
                $errors = $message;
            }
            return response()->json(['error' => $errors], 422);
        }

        try {
            //Lógica

            //Importar
            $import = new CustomerTemplateBeforeImportImport;
            Excel::import($import, request()->file('file_customers_import'));

            //Mensaje
            return response()->json([
                'success' => 'ok',
                'total_import_customers' => sprintf(__('sales/customer.help_import_sweet_alert_1'),Helper::numberFormat($import->total_customers)),
            ]);
        }catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 422);
        }
    }

    /**
     * Importar viviendas desde archivo
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     * @throws \Illuminate\Validation\ValidationException
     */
    public function storeImport(Request $request)
    {
        //Validacion
        //Validacion de archivos por extension
        if ($request->hasFile('file_customers_import')) {
            $request->merge(['file_customers_import_ext' => request()->file('file_customers_import')->getClientOriginalExtension()]);
        }
        $this->validate($request, [
            'file_customers_import' => 'required||max:2048',
            'file_customers_import_ext' => 'nullable|in:xls,xlsx'
        ], [
            'file_customers_import.*' => __('sales/customer.error_file_customers_import'),
            'file_customers_import_ext.*' => __('sales/customer.error_file_customers_import'),
        ]);

        try {
            //Lógica

            //Importar
            Excel::import(new CustomerTemplateImportImport(), request()->file('file_customers_import'));

            //Mensaje
            flash(__('general.text_form_success_import'))->success();

            //Redireccion
            return redirect()->route('customers.index');

        } catch (\Illuminate\Validation\ValidationException $e ) {
            return back()->withErrors($e->errors());
        } catch (\Exception $e) {
            flash($e->getMessage())->error();
            return back()->withInput();
        }
    }
}
