<?php

namespace App\Http\Controllers\Sales;

use App\Helpers\Helper;
use App\Models\Base\Pac;
use App\Helpers\PacHelper;
use App\Models\Base\Folio;
use App\Helpers\BaseHelper;
use Illuminate\Support\Str;
use App\Models\Catalogs\Tax;
use Illuminate\Http\Request;
use App\Helpers\Cfdi33Helper;
use App\Models\Sales\Customer;
use App\Models\Catalogs\CfdiUse;
use App\Models\Catalogs\Product;
use App\Models\Catalogs\Project;
use App\Mail\SendCustomerInvoice;
use App\Models\Base\BranchOffice;
use App\Models\Base\CfdiDownload;
use App\Models\Catalogs\Currency;
use App\Models\Sales\Salesperson;
use App\Models\Sales\CustomerIedu;
use App\Models\Catalogs\PaymentWay;
use App\Models\Catalogs\SatProduct;
use App\Models\Catalogs\TaxRegimen;
use App\Http\Controllers\Controller;
use App\Models\Catalogs\PaymentTerm;
use App\Models\Catalogs\UnitMeasure;
use Maatwebsite\Excel\Facades\Excel;
use App\Models\Catalogs\CfdiRelation;
use App\Models\Sales\CustomerInvoice;
use Illuminate\Support\Facades\Crypt;
use App\Models\Catalogs\PaymentMethod;
use App\Exports\CustomerInvoicesExport;
use App\Models\Sales\CustomerQuotation;
use App\Models\Sales\CustomerRemission;
use App\Models\Sales\CustomerInvoiceIne;
use App\Models\Sales\CustomerInvoiceTax;
use App\Models\Sales\CustomerInvoiceCfdi;
use App\Models\Sales\CustomerInvoiceLine;
use App\Models\Catalogs\ReasonCancellation;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use App\Models\Sales\CustomerInvoiceRelation;
use App\Models\Sales\CustomerInvoiceTaxLegend;
use Illuminate\Validation\ValidationException;
use App\Models\Sales\CustomerInvoiceComplement;
use App\Models\Sales\CustomerInvoiceLineComplement;
use App\Exports\CustomerInvoiceTemplateImportExport;
use App\Imports\CustomerInvoiceTemplateImportImport;
use App\Imports\CustomerInvoiceTemplateBeforeImportImport;

class CustomerInvoiceController extends Controller
{
    private $list_status = [];
    private $document_type_code = 'customer.invoice';

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->list_status = [
            CustomerInvoice::DRAFT => __('sales/customer_invoice.text_status_draft'),
            CustomerInvoice::OPEN => __('sales/customer_invoice.text_status_open'),
            CustomerInvoice::TO_PAY => __('sales/customer_invoice.text_status_to_pay'),
            CustomerInvoice::PAID => __('sales/customer_invoice.text_status_paid'),
            CustomerInvoice::CANCEL => __('sales/customer_invoice.text_status_cancel'),
            CustomerInvoice::CANCEL_PER_AUTHORIZED => __('sales/customer_invoice.text_status_cancel_per_authorized'),
        ];
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);
        if(!empty($request->download_xmlpdf)){
            $request->request->add([
                'page' => 1
            ]);
            $limit = 1000000;
        }
        $salespersons = Salesperson::populateSelect()->pluck('name', 'id');
        $branch_offices = BranchOffice::populateSelect()->pluck('name', 'id');
        $list_status = $this->list_status;
        if (empty($request->filter_date_from)) {
            $request->request->add([
                'filter_date_from' => Helper::date(\Date::parse('first day of this month'))
            ]);
        }
        if (empty($request->filter_date_to)) {
            $request->request->add([
                'filter_date_to' => Helper::date(\Date::parse('last day of this month'))
            ]);
        }
        $request->request->add(['filter_document_type_code' => $this->document_type_code]); //Filtra tipo de documento

        //Consulta
        $results = CustomerInvoice::filter($request->all())
            ->with('customerInvoiceCfdi')
            ->with('customer')
            ->with('customer.blackListSat')
            ->with('salesperson')
            ->with('currency')
            ->sortable(['date' => 'desc'])->paginate($limit);

        //Descar zip de XML´s y PDF´s
        if(!empty($request->download_xmlpdf)){
            if($results->isNotEmpty()) {
                $company = Helper::defaultCompany();
                $file_zip_name = __('sales/customer_invoice.document_title_download_xmlpdf').'_' . $company->taxid .'_'. str_replace(['-','/'],['',''],$request->filter_date_from).'_'. str_replace(['-','/'],['',''],$request->filter_date_to) .'_'.random_int(10,99).'.zip';
                if(\Storage::exists('temp/' . $file_zip_name)) {
                    @\Storage::delete('temp/' . $file_zip_name);
                }
                $zip =\Zipper::make(\Storage::path('temp/' . $file_zip_name));

                foreach ($results as $result) {
                    if (!empty($result->customerInvoiceCfdi->cfdi_version) && !empty($result->customerInvoiceCfdi->uuid)) {

                        //Ruta y validacion del XML
                        $path_xml = Helper::setDirectory(CustomerInvoice::PATH_XML_FILES_CI, $result->company_id) . '/';
                        $file_xml_pac = $path_xml . $result->customerInvoiceCfdi->file_xml_pac;
                        if (!empty($result->customerInvoiceCfdi->file_xml_pac) && \Storage::exists($file_xml_pac)) {
                            $zip->add(\Storage::path($file_xml_pac), str_replace('/','',$result->name) . '.xml');
                        }

                        //Ruta y validacion del pdf
                        $pdf = $this->print($result, false, true);
                        $file_pdf_name = Str::random().'.pdf';
                        //Guardamos en directorio temporal
                        \Storage::put('temp/' . $file_pdf_name, $pdf);
                        $zip->add(\Storage::path('temp/' . $file_pdf_name), str_replace('/','',$result->name) . '.pdf');

                    }
                }
                $zip->close();
                if(\Storage::exists('temp/' . $file_zip_name)) {
                    while (ob_get_level()) ob_end_clean();
                    ob_start();

                    return response()->download(\Storage::path('temp/' . $file_zip_name));
                }else{
                    flash(__('base/customer_invoice.error_download_xmlpdf'))->warning();
                }
            }
        }

        //Folios pendientes de pago
        if(!empty(setting('folios_to_pay'))){
            $folios = Folio::perActive()->orderByDesc('created_at')->get();
            if($folios->isNotEmpty()){
                $msg = '';
                foreach($folios as $folio){
                    $msg .= "<br/>" . $folio->product .' (' . money($folio->amount_total, 'MXN', true)->format() . ')';
                }
                flash(sprintf(__('general.text_alert_per_active_folios'),$msg))->error();
            }
        }

        $download_xml_pdf_id = null;
        if(!empty($request->download_xml_pdf_id)){
            $download_xml_pdf_id = $request->download_xml_pdf_id;
        }

        //Vista
        return view('sales.customer_invoices.index',
            compact('results', 'salespersons', 'branch_offices', 'list_status', 'download_xml_pdf_id'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        $salespersons = Salesperson::populateSelect()->pluck('name', 'id');
        $branch_offices = BranchOffice::populateSelect()->pluck('name', 'id');
        $currencies = Currency::populateSelect()->get()->pluck('name_sat', 'id');
        $payment_terms = PaymentTerm::populateSelect()->pluck('name', 'id');
        $payment_ways = PaymentWay::populateSelect()->get()->pluck('name_sat', 'id');
        $payment_methods = PaymentMethod::populateSelect()->get()->pluck('name_sat', 'id');
        $cfdi_uses = CfdiUse::populateSelect()->get()->pluck('name_sat', 'id');
        $taxes = Tax::populateSelect()->pluck('name', 'id');
        $cfdi_relations = CfdiRelation::populateSelect()->get()->pluck('name_sat', 'id');
        $projects = Project::populateSelect()->pluck('name', 'id');
        $company = Helper::defaultCompany(); //Empresa
        if(!empty($company->tax_regimen_id2)){
            $tax_regimens = TaxRegimen::populateSelect()->whereIn('id',[$company->tax_regimen_id, $company->tax_regimen_id2])->get()->pluck('name_sat', 'id');
        }else{
            $tax_regimens = collect([]);
        }
        $tax_regimen_customers = TaxRegimen::populateSelect()->get()->pluck('name_sat', 'id');

        $customer_quotation = null;
        if(!empty($request->cq_id)){
            $customer_quotation = CustomerQuotation::findOrFail($request->cq_id);
        }

        $customer_remission = null;
        if(!empty($request->cr_id)){
            $customer_remission = CustomerRemission::findOrFail($request->cr_id);
            $customer_quotation = $customer_remission;
        }

        $duplicate_ci = null;
        $duplicate_ci_complement = null;
        if($request->duplicate_id){
            $duplicate_ci = CustomerInvoice::findOrFail($request->duplicate_id);
            $duplicate_ci_complement = $duplicate_ci->customerInvoiceComplement;
        }

        $ine_process_types = __('general.text_ine_process_types');
        $ine_committee_types = __('general.text_ine_committee_types');
        $ine_ambits = __('general.text_ine_ambits');
        $ine_entities = __('general.text_ine_entities');
        $global_periodicities = __('general.text_global_periodicities');
        $global_months = __('general.text_global_months');
        $global_years = [date('Y') => date('Y'), (date('Y') - 1) => (date('Y') - 1)];
        $discount_types = __('general.text_discount_types');
        $tax_objects = __('general.text_tax_objects');

        return view('sales.customer_invoices.create',
            compact('salespersons', 'branch_offices', 'currencies', 'payment_ways', 'payment_terms', 'payment_methods',
                'cfdi_uses', 'taxes', 'cfdi_relations','customer_quotation', 'customer_remission','duplicate_ci','projects','tax_regimens','tax_regimen_customers', 'duplicate_ci_complement', 'ine_process_types', 'ine_committee_types', 'ine_ambits', 'ine_entities','global_periodicities','global_months', 'global_years', 'discount_types', 'tax_objects'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);

        //Almacenamiento de archivos
        $customer_invoice = null;

        \DB::connection('tenant')->beginTransaction();
        try {
            //Logica
            $company = Helper::defaultCompany(); //Empresa
            $request->merge(['created_uid' => \Auth::user()->id]);
            $request->merge(['updated_uid' => \Auth::user()->id]);
            $request->merge(['status' => isset($request->pre_invoice) ? CustomerInvoice::DRAFT : CustomerInvoice::OPEN]);
            $request->merge(['company_id' => $company->id]);
            //Ajusta fecha y genera fecha de vencimiento
            $date = Helper::createDateTime($request->date);
            $request->merge(['date' => Helper::dateTimeToSql($date)]);
            $date_due = $date;
            $date_due_fix = $request->date_due;//Fix valida si la fecha de vencimiento esta vacia en caso de error
            if (!empty($request->date_due)) {
                $date_due = Helper::createDate($request->date_due);
            } else {
                $payment_term = PaymentTerm::findOrFail($request->payment_term_id);
                $date_due = $payment_term->days > 0 ? $date->copy()->addDays($payment_term->days) : $date->copy();
            }
            $request->merge(['date_due' => Helper::dateToSql($date_due)]);

            //Obtiene folio
            $request->merge(['name' => Str::random(40)]);

            //Valida que tenga folios disponibles
            if(BaseHelper::getAvailableFolios()<=0 && !isset($request->pre_invoice)){
                throw new \Exception(__('general.error_available_folios'));
            }

            //Guardar
            //Registro principal
            $customer_invoice = CustomerInvoice::create($request->input());
            $request->merge(['customer_invoice_id' => $customer_invoice->id]);
            $customer_invoice_complement = CustomerInvoiceComplement::create($request->input());

            //
            if($customer_invoice->customer->taxid != 'XAXX010101000'){
                $customer_invoice->global_periodicity = null;
                $customer_invoice->global_month = null;
                $customer_invoice->global_year = null;
                $customer_invoice->save();
            }


            //Registro de lineas
            $amount_discount = 0;
            $amount_untaxed = 0;
            $amount_tax = 0;
            $amount_tax_ret = 0;
            $amount_total = 0;
            $balance = 0;
            $taxes = array();
            //Lineas
            if (!empty($request->item)) {
                foreach ($request->item as $key => $item) {
                    //Logica
                    $item_quantity = (double)$item['quantity'];
                    $item_price_unit = (double)$item['price_unit'];
                    $item_quota_ieps = (double)$item['quota_ieps'];
                    //Ajuste para cuando tiene IVA incluido
                    if (!empty($item['taxes']) && !empty($item['includes_iva'])) {
                        foreach ($item['taxes'] as $tax_id) {
                            if (!empty($tax_id)) {
                                $tax = Tax::findOrFail($tax_id);
                                if($tax->code == '002' && $tax->rate>0) {
                                    if ($tax->factor == 'Tasa') {
                                        $item_price_unit = $item_price_unit / (1+($tax->rate/100));
                                    } elseif ($tax->factor == 'Cuota') {
                                        $item_price_unit -= $tax->rate;
                                    }
                                    $item_price_unit = round($item_price_unit, \App\Helpers\Helper::companyProductPriceDecimalPlace());
                                    $input_items[$key]['price_unit'] = $item_price_unit;
                                }
                            }
                            break;
                        }
                    }
                    $item_discount = round((double)$item['discount'], 2);
                    $item_price_reduce = $item['discount_type'] == 'A' ? $item_price_unit : ($item_price_unit * (100 - $item_discount) / 100);
                    $item_amount_untaxed = round($item_quantity * $item_price_reduce, 2);
                    if($item_quota_ieps > 0){
                        $item_amount_untaxed_per_taxes = round($item_quantity * (($item_price_unit * (100 - $item_discount) / 100) - $item_quota_ieps), 2);
                    }else{
                        $item_amount_untaxed_per_taxes = $item_amount_untaxed;
                    }
                    if($item['discount_type'] == 'A'){
                        $item_amount_untaxed -= $item_discount;
                        $item_amount_untaxed_per_taxes -= $item_discount;
                    }
                    $item_amount_discount = $item['discount_type'] == 'A' ? $item_discount : round($item_quantity * $item_price_unit, 2) - $item_amount_untaxed;
                    $item_amount_tax = 0;
                    $item_amount_tax_ret = 0;
                    if (!empty($item['taxes'])) {
                        foreach ($item['taxes'] as $tax_id) {
                            if (!empty($tax_id)) {
                                $tax = Tax::findOrFail($tax_id);
                                $tmp = 0;
                                if ($tax->factor == 'Tasa') {
                                    $tmp = ($item_amount_untaxed_per_taxes + (!empty($item['fix_amount_untaxed']) && !empty($tax->code) ? 0.01 : 0 )) * $tax->rate / 100;
                                } elseif ($tax->factor == 'Cuota') {
                                    $tmp = $tax->rate;
                                }
                                $tmp = round($tmp, 2);
                                if ($tax->type == 'R') { //Retenciones
                                    $item_amount_tax_ret += $tmp;
                                } else { //Traslados
                                    $item_amount_tax += $tmp;
                                }

                                //Sumatoria de impuestos
                                $taxes[$tax_id] = array(
                                    'amount_base' => $item_amount_untaxed_per_taxes + (isset($taxes[$tax_id]['amount_base']) ? $taxes[$tax_id]['amount_base'] : 0),
                                    'amount_tax' => $tmp + (isset($taxes[$tax_id]['amount_tax']) ? $taxes[$tax_id]['amount_tax'] : 0),
                                );
                            }
                        }
                    }
                    $item_amount_total = $item_amount_untaxed + $item_amount_tax + $item_amount_tax_ret;
                    //Sumatoria totales
                    $amount_discount += $item_amount_discount;
                    $amount_untaxed += $item_amount_untaxed;
                    $amount_tax += $item_amount_tax;
                    $amount_tax_ret += $item_amount_tax_ret;
                    $amount_total += $item_amount_total;

                    //Guardar linea
                    $customer_invoice_line = CustomerInvoiceLine::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'customer_invoice_id' => $customer_invoice->id,
                        'name' => $item['name'],
                        'product_id' => $item['product_id'],
                        'sat_product_id' => $item['sat_product_id'],
                        'unit_measure_id' => $item['unit_measure_id'],
                        'quantity' => $item_quantity,
                        'price_unit' => $item_price_unit,
                        'discount' => $item_discount,
                        'price_reduce' => $item_price_reduce,
                        'amount_discount' => $item_amount_discount,
                        'amount_untaxed' => $item_amount_untaxed,
                        'amount_tax' => $item_amount_tax,
                        'amount_tax_ret' => $item_amount_tax_ret,
                        'amount_total' => $item_amount_total,
                        'sort_order' => $key,
                        'status' => 1,
                        'quota_ieps' => $item_quota_ieps,
                        'tax_object' => !empty($item['tax_object']) ? $item['tax_object'] : (empty($item['taxes']) ? '01' : '02'),
                        'fix_amount_untaxed' => !empty($item['fix_amount_untaxed']) ? 1 : null,
                        'discount_type' => $item['discount_type'],
                    ]);

                    //Guardar impuestos por linea
                    if (!empty($item['taxes'])) {
                        $customer_invoice_line->taxes()->sync($item['taxes']);
                    } else {
                        $customer_invoice_line->taxes()->sync([]);
                    }

                    //Datos de complementos IEDU
                    if(!empty($item['customer_iedu_id'])){
                        $customer_iedu = CustomerIedu::findOrFail($item['customer_iedu_id']);
                        $item['iedu_nombre_alumno'] = $customer_iedu->nombre_alumno;
                        $item['iedu_curp'] = $customer_iedu->curp;
                        $item['iedu_nivel_educativo'] = $customer_iedu->ieduNivelEducativo->code;
                        $item['iedu_aut_rvoe'] = $customer_iedu->ieduNivelEducativo->aut_rvoe;
                    }
                    $customer_invoice_line_complement = CustomerInvoiceLineComplement::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'customer_invoice_line_id' => $customer_invoice_line->id,
                        'name' => $item['name'],
                        'customer_iedu_id' => !empty($item['customer_iedu_id']) ? $item['customer_iedu_id'] : null,
                        'iedu_nombre_alumno' => !empty($item['iedu_nombre_alumno']) ? $item['iedu_nombre_alumno'] : '',
                        'iedu_curp' => !empty($item['iedu_curp']) ? $item['iedu_curp'] : '',
                        'iedu_nivel_educativo' => !empty($item['iedu_nivel_educativo']) ? $item['iedu_nivel_educativo'] : '',
                        'iedu_aut_rvoe' => !empty($item['iedu_aut_rvoe']) ? $item['iedu_aut_rvoe'] : '',
                        'iedu_rfc_pago' => !empty($item['iedu_rfc_pago']) ? $item['iedu_rfc_pago'] : '',
                        'numero_pedimento' => !empty($item['numero_pedimento']) ? $item['numero_pedimento'] : '',
                        'sort_order' => $key,
                        'status' => 1,
                    ]);
                }
            }

            //Resumen de impuestos
            if (!empty($taxes)) {
                $i = 0;
                foreach ($taxes as $tax_id => $result) {
                    $tax = Tax::findOrFail($tax_id);
                    $customer_invoice_tax = CustomerInvoiceTax::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'customer_invoice_id' => $customer_invoice->id,
                        'name' => $tax->name,
                        'tax_id' => $tax_id,
                        'amount_base' => $result['amount_base'],
                        'amount_tax' => $result['amount_tax'],
                        'sort_order' => $i,
                        'status' => 1,
                    ]);
                    $i++;
                }
            }

            //Cfdi relacionados
            if (!empty($request->item_relation)) {
                foreach ($request->item_relation as $key => $result) {
                    //Guardar
                    $customer_invoice_relation = CustomerInvoiceRelation::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'customer_invoice_id' => $customer_invoice->id,
                        'relation_id' => !empty($result['relation_id']) ? $result['relation_id'] : null,
                        'sort_order' => $key,
                        'status' => 1,
                        'uuid_related' => $result['uuid_related'],
                    ]);
                }
            }

            //Complementos
            //Leyendas fiscales
            if (!empty($request->complement_tax_legend) && !empty($request->item_tax_legend)) {
                foreach ($request->item_tax_legend as $key => $result) {
                    //Guardar
                    if(!empty($result['texto_leyenda'])){
                        $customer_invoice_tax_legend = CustomerInvoiceTaxLegend::create([
                            'created_uid' => \Auth::user()->id,
                            'updated_uid' => \Auth::user()->id,
                            'customer_invoice_id' => $customer_invoice->id,
                            'disposicion_fiscal' => $result['disposicion_fiscal'],
                            'norma' => $result['norma'],
                            'texto_leyenda' => $result['texto_leyenda'],
                            'sort_order' => $key,
                            'status' => 1,
                        ]);
                    }
                }
            }

            //INE
            if (!empty($request->complement_ine) && !empty($request->item_ine)) {
                foreach ($request->item_ine as $key => $result) {
                    //Guardar
                    if(!empty($result['ine_entity'])){
                        $customer_invoice_ine = CustomerInvoiceIne::create([
                            'created_uid' => \Auth::user()->id,
                            'updated_uid' => \Auth::user()->id,
                            'customer_invoice_id' => $customer_invoice->id,
                            'ine_entity' => $result['ine_entity'],
                            'ambit' => $result['ambit'],
                            'id_accounting' => $result['id_accounting'],
                            'sort_order' => $key,
                            'status' => 1,
                        ]);
                    }
                }
            }

            //Valida que tenga exista la clase de facturacion
            $class_cfdi = setting('cfdi_version');
            if (empty($class_cfdi)) {
                throw new \Exception(__('general.error_cfdi_version'));
            }
            if (!method_exists($this, $class_cfdi)) {
                throw new \Exception(__('general.error_cfdi_class_exists'));
            }

            //Registros de cfdi
            $customer_invoice_cfdi = CustomerInvoiceCfdi::create([
                'created_uid' => \Auth::user()->id,
                'updated_uid' => \Auth::user()->id,
                'customer_invoice_id' => $customer_invoice->id,
                'name' => $customer_invoice->name,
                'cfdi_version' => $class_cfdi,
                'status' => 1,
            ]);

            //Actualiza registro principal con totales
            $customer_invoice->amount_discount = $amount_discount;
            $customer_invoice->amount_untaxed = $amount_untaxed;
            $customer_invoice->amount_tax = $amount_tax;
            $customer_invoice->amount_tax_ret = $amount_tax_ret;
            $customer_invoice->amount_total = $amount_total;
            $customer_invoice->balance = $amount_total;
            $customer_invoice->update();

            //Obtiene folio
            $document_type = Helper::getNextDocumentTypeByCode($this->document_type_code,$company->id,isset($request->pre_invoice) ? true : false, $customer_invoice->branch_office_id);
            $customer_invoice->document_type_id = $document_type['id'];
            $customer_invoice->name = $document_type['name'];
            $customer_invoice->serie = $document_type['serie'];
            $customer_invoice->folio = $document_type['folio'];
            $customer_invoice->update();

            //Solo guarda los datos sin timbrar en caso de pre-factura
            if(isset($request->pre_invoice)) {
                //Mensaje
                flash(__('general.text_success_pre_customer_invoice'))->success();

            }else{ //Crear CFDI
                //Valida Empresa y PAC para timbrado
                PacHelper::validateSatActions($company);

                //Crear XML y timbra
                $tmp = $this->$class_cfdi($customer_invoice);

                //Guardar registros de CFDI
                $customer_invoice_cfdi->fill(array_only($tmp, [
                    'pac_id',
                    'cfdi_version',
                    'uuid',
                    'date',
                    'file_xml',
                    'file_xml_pac',
                ]));
                $customer_invoice_cfdi->save();

                //Disminuye folios
                BaseHelper::decrementFolios();

                //Actualiza cotizacion en caso de estar relacionada
                if(!empty($customer_invoice->customer_quotation_id)) {
                    $customer_quotation = CustomerQuotation::findOrFail($customer_invoice->customer_quotation_id);
                    $customer_quotation->status = CustomerQuotation::BILLED;
                    $customer_quotation->save();
                }
                if(!empty($customer_invoice->customer_remission_id)) {
                    $customer_remission = CustomerRemission::findOrFail($customer_invoice->customer_remission_id);
                    $customer_remission->status = CustomerRemission::BILLED;
                    $customer_remission->save();
                }

                //Mensaje
                flash(__('general.text_success_customer_invoice_cfdi'))->success();
            }

            \DB::connection('tenant')->commit();

            if(!isset($request->pre_invoice)) {
                $this->saveCfdiDownloads($customer_invoice, $customer_invoice_cfdi);
            }

        } catch (\Exception $e) {
            //Fix fechas
            $request->merge([
                'date' => Helper::convertSqlToDateTime($request->date),
            ]);
            if (!empty($date_due_fix)) {
                $request->merge([
                    'date_due' => Helper::convertSqlToDate($request->date_due),
                ]);
            }else{
                $request->merge([
                    'date_due' => '',
                ]);
            }

            \DB::connection('tenant')->rollback();
            $company = Helper::defaultCompany(); //Empresa
            \Log::error('(' . $company->taxid . ') ' . $e->getMessage());
            flash((isset($request->pre_invoice) ? '' : __('general.error_cfdi_pac').'<br>').$e->getMessage())->error();
            return back()->withInput();
        }

        //Almacenamiento dropbox
        self::dropboxBackup($customer_invoice);

        //Envio auomatico de correo
        if(isset($request->invoice_and_send)) {
            try{
                $this->autoSendMail($customer_invoice);
            } catch (\Exception $e) {
                flash($e->getMessage())->error();
            }
        }

        //Redireccion
        return redirect()->route('customer-invoices.index',['download_xml_pdf_id' => !empty($request->download_xml_pdf) ? $customer_invoice->id : 0]);
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Sales\CustomerInvoice $customer_invoice
     * @return \Illuminate\Http\Response
     */
    public function show(CustomerInvoice $customer_invoice)
    {
        //Datos
        $data = [];

        //Si tiene CFDI obtiene la informacion de los nodos
        if(!empty($customer_invoice->customerInvoiceCfdi->file_xml_pac) && !empty($customer_invoice->customerInvoiceCfdi->uuid)){

            $path_xml = Helper::setDirectory(CustomerInvoice::PATH_XML_FILES_CI, $customer_invoice->company_id) . '/';
            $file_xml_pac = $path_xml . $customer_invoice->customerInvoiceCfdi->file_xml_pac;

            //Valida que el archivo exista
            if (!empty($customer_invoice->customerInvoiceCfdi->file_xml_pac) && \Storage::exists($file_xml_pac)) {
                $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($file_xml_pac));
                $data = Cfdi33Helper::getQuickArrayCfdi($cfdi);

                //Genera codigo QR
                $image = QrCode::format('png')->size(150)->margin(0)->generate($data['qr_cadena']);
                $data['qr'] = 'data:image/png;base64,' . base64_encode($image);

                //Regimen fiscal
                $data['tax_regimen'] = TaxRegimen::where('code', $data['cfdi33']->emisor['RegimenFiscal'])->first()->name_sat;
                $data['tax_regimen_customer'] = !empty($data['cfdi33']->receptor['RegimenFiscalReceptor']) ? TaxRegimen::where('code', $data['cfdi33']->receptor['RegimenFiscalReceptor'])->first()->name_sat : '';
            }
        }

        return view('sales.customer_invoices.show', compact('customer_invoice','data'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Sales\CustomerInvoice $customer_invoice
     * @return \Illuminate\Http\Response
     */
    public function edit(CustomerInvoice $customer_invoice)
    {
        $salespersons = Salesperson::populateSelect()->pluck('name', 'id');
        $branch_offices = BranchOffice::populateSelect()->pluck('name', 'id');
        $currencies = Currency::populateSelect()->get()->pluck('name_sat', 'id');
        $payment_terms = PaymentTerm::populateSelect()->pluck('name', 'id');
        $payment_ways = PaymentWay::populateSelect()->get()->pluck('name_sat', 'id');
        $payment_methods = PaymentMethod::populateSelect()->get()->pluck('name_sat', 'id');
        $cfdi_uses = CfdiUse::populateSelect()->get()->pluck('name_sat', 'id');
        $taxes = Tax::populateSelect()->pluck('name', 'id');
        $cfdi_relations = CfdiRelation::populateSelect()->get()->pluck('name_sat', 'id');
        $projects = Project::populateSelect()->pluck('name', 'id');
        $company = $customer_invoice->company; //Empresa
        if(!empty($company->tax_regimen_id2)){
            $tax_regimens = TaxRegimen::populateSelect()->whereIn('id',[$company->tax_regimen_id, $company->tax_regimen_id2])->get()->pluck('name_sat', 'id');
        }else{
            $tax_regimens = collect([]);
        }
        $tax_regimen_customers = TaxRegimen::populateSelect()->get()->pluck('name_sat', 'id');

        $customer_invoice_complement = CustomerInvoiceComplement::where('customer_invoice_id', $customer_invoice->id)->first();

        $ine_process_types = __('general.text_ine_process_types');
        $ine_committee_types = __('general.text_ine_committee_types');
        $ine_ambits = __('general.text_ine_ambits');
        $ine_entities = __('general.text_ine_entities');

        $global_periodicities = __('general.text_global_periodicities');
        $global_months = __('general.text_global_months');
        $global_years = [date('Y') => date('Y'), (date('Y') - 1) => (date('Y') - 1)];
        $discount_types = __('general.text_discount_types');
        $tax_objects = __('general.text_tax_objects');

        return view('sales.customer_invoices.edit',
            compact('customer_invoice','salespersons', 'branch_offices', 'currencies', 'payment_ways', 'payment_terms', 'payment_methods',
                'cfdi_uses', 'taxes', 'cfdi_relations','projects','tax_regimens','tax_regimen_customers', 'customer_invoice_complement', 'ine_process_types', 'ine_committee_types', 'ine_ambits', 'ine_entities', 'global_periodicities', 'global_months', 'global_years', 'discount_types', 'tax_objects'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @param  \App\Models\Sales\CustomerInvoice $customer_invoice
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, CustomerInvoice $customer_invoice)
    {
        //Validacion
        $this->validation($request);

        \DB::connection('tenant')->beginTransaction();
        try {

            //Logica
            $request->merge(['created_uid' => \Auth::user()->id]);
            $request->merge(['updated_uid' => \Auth::user()->id]);
            $request->merge(['status' => isset($request->pre_invoice) ? CustomerInvoice::DRAFT : CustomerInvoice::OPEN]);
            //Ajusta fecha y genera fecha de vencimiento
            $date = Helper::createDateTime($request->date);
            $request->merge(['date' => Helper::dateTimeToSql($date)]);
            $date_due = $date;
            $date_due_fix = $request->date_due;//Fix valida si la fecha de vencimiento esta vacia en caso de error
            if (!empty($request->date_due)) {
                $date_due = Helper::createDate($request->date_due);
            } else {
                $payment_term = PaymentTerm::findOrFail($request->payment_term_id);
                $date_due = $payment_term->days > 0 ? $date->copy()->addDays($payment_term->days) : $date->copy();
            }
            $request->merge(['date_due' => Helper::dateToSql($date_due)]);
            $customer_invoice->fill($request->only([
                'updated_uid',
                'date',
                'date_due',
                'reference',
                'customer_id',
                'branch_office_id',
                'payment_term_id',
                'payment_way_id',
                'payment_method_id',
                'cfdi_use_id',
                'salesperson_id',
                'currency_id',
                'currency_value',
                'cfdi_relation_id',
                'comment',
                'status',
                'title',
                'project_id',
                'tax_regimen_id',
                'tax_regimen_customer_id',
                'global_periodicity',
                'global_month',
                'global_year',
                'addenda'
            ]));

            //Guardar
            //Registro principal
            $customer_invoice->save();

            //
            if($customer_invoice->customer->taxid != 'XAXX010101000'){
                $customer_invoice->global_periodicity = null;
                $customer_invoice->global_month = null;
                $customer_invoice->global_year = null;
                $customer_invoice->save();
            }

            $customer_invoice_complement = CustomerInvoiceComplement::where('customer_invoice_id',$customer_invoice->id)->first();
            if(!empty($customer_invoice_complement)){
                $customer_invoice_complement->fill($request->only([
                    'updated_uid',
                    'ine_process_type',
                    'ine_committee_type',
                    'ine_id_accounting',
                ]));

                $customer_invoice_complement->save();
            }

            //Actualiza folios
            if(!isset($request->pre_invoice)) {
                //Valida que tenga folios disponibles
                if(BaseHelper::getAvailableFolios()<=0){
                    throw new \Exception(__('general.error_available_folios'));
                }
            }

            //Elimina partidas
            if (!empty($request->delete_item)) {
                foreach ($request->delete_item as $key => $result) {
                    //Actualizar status
                    $customer_invoice_line = CustomerInvoiceLine::findOrFail($result);
                    $customer_invoice_line->updated_uid = \Auth::user()->id;
                    $customer_invoice_line->status = 0;
                    $customer_invoice_line->save();
                }
            }
            //Registro de lineas
            $amount_discount = 0;
            $amount_untaxed = 0;
            $amount_tax = 0;
            $amount_tax_ret = 0;
            $amount_total = 0;
            $balance = 0;
            $taxes = array();
            //Lineas
            if (!empty($request->item)) {
                foreach ($request->item as $key => $item) {
                    //Logica
                    $item_quantity = (double)$item['quantity'];
                    $item_price_unit = (double)$item['price_unit'];
                    $item_quota_ieps = (double)$item['quota_ieps'];
                    //Ajuste para cuando tiene IVA incluido
                    if (!empty($item['taxes']) && !empty($item['includes_iva'])) {
                        foreach ($item['taxes'] as $tax_id) {
                            if (!empty($tax_id)) {
                                $tax = Tax::findOrFail($tax_id);
                                if($tax->code == '002' && $tax->rate>0) {
                                    if ($tax->factor == 'Tasa') {
                                        $item_price_unit = $item_price_unit / (1+($tax->rate/100));
                                    } elseif ($tax->factor == 'Cuota') {
                                        $item_price_unit -= $tax->rate;
                                    }
                                    $item_price_unit = round($item_price_unit, \App\Helpers\Helper::companyProductPriceDecimalPlace());
                                    $input_items[$key]['price_unit'] = $item_price_unit;
                                }
                            }
                            break;
                        }
                    }
                    $item_discount = (double)$item['discount'];
                    $item_price_reduce = $item['discount_type'] == 'A' ? $item_price_unit : ($item_price_unit * (100 - $item_discount) / 100);
                    $item_amount_untaxed = round($item_quantity * $item_price_reduce, 2);
                    if($item_quota_ieps > 0){
                        $item_amount_untaxed_per_taxes = round($item_quantity * (($item_price_unit * (100 - $item_discount) / 100) - $item_quota_ieps), 2);
                    }else{
                        $item_amount_untaxed_per_taxes = $item_amount_untaxed;
                    }
                    if($item['discount_type'] == 'A'){
                        $item_amount_untaxed -= $item_discount;
                        $item_amount_untaxed_per_taxes -= $item_discount;
                    }
                    $item_amount_discount = $item['discount_type'] == 'A' ? $item_discount : round($item_quantity * $item_price_unit, 2) - $item_amount_untaxed;
                    $item_amount_tax = 0;
                    $item_amount_tax_ret = 0;
                    if (!empty($item['taxes'])) {
                        foreach ($item['taxes'] as $tax_id) {
                            if (!empty($tax_id)) {
                                $tax = Tax::findOrFail($tax_id);
                                $tmp = 0;
                                if ($tax->factor == 'Tasa') {
                                    $tmp = ($item_amount_untaxed_per_taxes + (!empty($item['fix_amount_untaxed']) && !empty($tax->code) ? 0.01 : 0 )) * $tax->rate / 100;
                                } elseif ($tax->factor == 'Cuota') {
                                    $tmp = $tax->rate;
                                }
                                $tmp = round($tmp, 2);
                                if ($tax->type == 'R') { //Retenciones
                                    $item_amount_tax_ret += $tmp;
                                } else { //Traslados
                                    $item_amount_tax += $tmp;
                                }

                                //Sumatoria de impuestos
                                $taxes[$tax_id] = array(
                                    'amount_base' => $item_amount_untaxed_per_taxes + (isset($taxes[$tax_id]['amount_base']) ? $taxes[$tax_id]['amount_base'] : 0),
                                    'amount_tax' => $tmp + (isset($taxes[$tax_id]['amount_tax']) ? $taxes[$tax_id]['amount_tax'] : 0),
                                );
                            }
                        }
                    }
                    $item_amount_total = $item_amount_untaxed + $item_amount_tax + $item_amount_tax_ret;
                    //Sumatoria totales
                    $amount_discount += $item_amount_discount;
                    $amount_untaxed += $item_amount_untaxed;
                    $amount_tax += $item_amount_tax;
                    $amount_tax_ret += $item_amount_tax_ret;
                    $amount_total += $item_amount_total;

                    //Guardar linea
                    $data = [
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'customer_invoice_id' => $customer_invoice->id,
                        'name' => $item['name'],
                        'product_id' => $item['product_id'],
                        'sat_product_id' => $item['sat_product_id'],
                        'unit_measure_id' => $item['unit_measure_id'],
                        'quantity' => $item_quantity,
                        'price_unit' => $item_price_unit,
                        'discount' => $item_discount,
                        'price_reduce' => $item_price_reduce,
                        'amount_discount' => $item_amount_discount,
                        'amount_untaxed' => $item_amount_untaxed,
                        'amount_tax' => $item_amount_tax,
                        'amount_tax_ret' => $item_amount_tax_ret,
                        'amount_total' => $item_amount_total,
                        'sort_order' => $key,
                        'status' => 1,
                        'quota_ieps' => $item_quota_ieps,
                        'tax_object' => !empty($item['tax_object']) ? $item['tax_object'] : (empty($item['taxes']) ? '01' : '02'),
                        'fix_amount_untaxed' => !empty($item['fix_amount_untaxed']) ? 1 : null,
                        'discount_type' => $item['discount_type'],
                    ];
                    if (!empty($item['id'])) {
                        $customer_invoice_line = CustomerInvoiceLine::findOrFail($item['id']);
                        $customer_invoice_line->fill(array_only($data, [
                            'updated_uid',
                            'name',
                            'product_id',
                            'sat_product_id',
                            'unit_measure_id',
                            'quantity',
                            'price_unit',
                            'discount',
                            'price_reduce',
                            'amount_discount',
                            'amount_untaxed',
                            'amount_tax',
                            'amount_tax_ret',
                            'amount_total',
                            'sort_order',
                            'quota_ieps',
                            'tax_object',
                            'fix_amount_untaxed',
                            'discount_type'
                        ]));
                        $customer_invoice_line->save();
                    }else{
                        $customer_invoice_line = CustomerInvoiceLine::create($data);
                    }

                    //Guardar impuestos por linea
                    if (!empty($item['taxes'])) {
                        $customer_invoice_line->taxes()->sync($item['taxes']);
                    } else {
                        $customer_invoice_line->taxes()->sync([]);
                    }

                    //Datos de complementos IEDU
                    if(!empty($item['customer_iedu_id'])){
                        $customer_iedu = CustomerIedu::findOrFail($item['customer_iedu_id']);
                        $item['iedu_nombre_alumno'] = $customer_iedu->nombre_alumno;
                        $item['iedu_curp'] = $customer_iedu->curp;
                        $item['iedu_nivel_educativo'] = $customer_iedu->ieduNivelEducativo->code;
                        $item['iedu_aut_rvoe'] = $customer_iedu->ieduNivelEducativo->aut_rvoe;
                    }
                    $customer_invoice_line_complement = CustomerInvoiceLineComplement::where('customer_invoice_line_id','=',$customer_invoice_line->id)->first();
                    $data_complement = [
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'customer_invoice_line_id' => $customer_invoice_line->id,
                        'name' => $item['name'],
                        'customer_iedu_id' => !empty($item['customer_iedu_id']) ? $item['customer_iedu_id'] : null,
                        'iedu_nombre_alumno' => !empty($item['iedu_nombre_alumno']) ? $item['iedu_nombre_alumno'] : '',
                        'iedu_curp' => !empty($item['iedu_curp']) ? $item['iedu_curp'] : '',
                        'iedu_nivel_educativo' => !empty($item['iedu_nivel_educativo']) ? $item['iedu_nivel_educativo'] : '',
                        'iedu_aut_rvoe' => !empty($item['iedu_aut_rvoe']) ? $item['iedu_aut_rvoe'] : '',
                        'iedu_rfc_pago' => !empty($item['iedu_rfc_pago']) ? $item['iedu_rfc_pago'] : '',
                        'numero_pedimento' => !empty($item['numero_pedimento']) ? $item['numero_pedimento'] : '',
                        'sort_order' => $key,
                        'status' => 1,
                    ];
                    if(!empty($customer_invoice_line_complement)){
                        $customer_invoice_line_complement->fill(array_only($data_complement, [
                            'updated_uid',
                            'name',
                            'customer_iedu_id',
                            'iedu_nombre_alumno',
                            'iedu_curp',
                            'iedu_nivel_educativo',
                            'iedu_aut_rvoe',
                            'iedu_rfc_pago',
                            'numero_pedimento',
                            'sort_order',
                        ]));
                        $customer_invoice_line_complement->save();
                    }else{
                        $customer_invoice_line_complement = CustomerInvoiceLineComplement::create($data_complement);
                    }
                }
            }

            //Resumen de impuestos
            CustomerInvoiceTax::where('customer_invoice_id','=',$customer_invoice->id)->delete(); //Borra todo e inserta nuevamente
            if (!empty($taxes)) {
                $i = 0;
                foreach ($taxes as $tax_id => $result) {
                    $tax = Tax::findOrFail($tax_id);
                    $customer_invoice_tax = CustomerInvoiceTax::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'customer_invoice_id' => $customer_invoice->id,
                        'name' => $tax->name,
                        'tax_id' => $tax_id,
                        'amount_base' => $result['amount_base'],
                        'amount_tax' => $result['amount_tax'],
                        'sort_order' => $i,
                        'status' => 1,
                    ]);
                    $i++;
                }
            }

            //Cfdi relacionados
            CustomerInvoiceRelation::where('customer_invoice_id','=',$customer_invoice->id)->delete(); //Borra todo e inserta nuevamente
            if (!empty($request->item_relation)) {
                foreach ($request->item_relation as $key => $result) {
                    //Guardar
                    $customer_invoice_relation = CustomerInvoiceRelation::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'customer_invoice_id' => $customer_invoice->id,
                        'relation_id' => !empty($result['relation_id']) ? $result['relation_id'] : null,
                        'sort_order' => $key,
                        'status' => 1,
                        'uuid_related' => $result['uuid_related'],
                    ]);
                }
            }

            //Complementos
            //Leyendas fiscales
            CustomerInvoiceTaxLegend::where('customer_invoice_id','=',$customer_invoice->id)->delete(); //Borra todo e inserta nuevamente
            if (!empty($request->complement_tax_legend) && !empty($request->item_tax_legend)) {
                foreach ($request->item_tax_legend as $key => $result) {
                    //Guardar
                    if(!empty($result['texto_leyenda'])){
                        $customer_invoice_tax_legend = CustomerInvoiceTaxLegend::create([
                            'created_uid' => \Auth::user()->id,
                            'updated_uid' => \Auth::user()->id,
                            'customer_invoice_id' => $customer_invoice->id,
                            'disposicion_fiscal' => $result['disposicion_fiscal'],
                            'norma' => $result['norma'],
                            'texto_leyenda' => $result['texto_leyenda'],
                            'sort_order' => $key,
                            'status' => 1,
                        ]);
                    }
                }
            }

            //Ine
            CustomerInvoiceIne::where('customer_invoice_id','=',$customer_invoice->id)->delete(); //Borra todo e inserta nuevamente
            if (!empty($request->complement_ine) && !empty($request->item_ine)) {
                foreach ($request->item_ine as $key => $result) {
                    //Guardar
                    if(!empty($result['ine_entity'])){
                        $customer_invoice_ine = CustomerInvoiceIne::create([
                            'created_uid' => \Auth::user()->id,
                            'updated_uid' => \Auth::user()->id,
                            'customer_invoice_id' => $customer_invoice->id,
                            'ine_entity' => $result['ine_entity'],
                            'ambit' => $result['ambit'],
                            'id_accounting' => $result['id_accounting'],
                            'sort_order' => $key,
                            'status' => 1,
                        ]);
                    }
                }
            }

            //Valida que tenga exista la clase de facturacion
            $class_cfdi = setting('cfdi_version');
            if (empty($class_cfdi)) {
                throw new \Exception(__('general.error_cfdi_version'));
            }
            if (!method_exists($this, $class_cfdi)) {
                throw new \Exception(__('general.error_cfdi_class_exists'));
            }

            //Registros de cfdi
            CustomerInvoiceCfdi::where('customer_invoice_id','=',$customer_invoice->id)->delete(); //Borra todo e inserta nuevamente
            $customer_invoice_cfdi = CustomerInvoiceCfdi::create([
                'created_uid' => \Auth::user()->id,
                'updated_uid' => \Auth::user()->id,
                'customer_invoice_id' => $customer_invoice->id,
                'name' => $customer_invoice->name,
                'cfdi_version' => $class_cfdi,
                'status' => 1,
            ]);

            //Actualiza registro principal con totales
            $customer_invoice->amount_discount = $amount_discount;
            $customer_invoice->amount_untaxed = $amount_untaxed;
            $customer_invoice->amount_tax = $amount_tax;
            $customer_invoice->amount_tax_ret = $amount_tax_ret;
            $customer_invoice->amount_total = $amount_total;
            $customer_invoice->balance = $amount_total;
            $customer_invoice->update();

            if(!isset($request->pre_invoice)) {
                //Obtiene folio
                $document_type = Helper::getNextDocumentTypeByCode($this->document_type_code, $customer_invoice->company->id, false, $customer_invoice->branch_office_id);
                $customer_invoice->draft = $customer_invoice->name;
                $customer_invoice->name = $document_type['name'];
                $customer_invoice->serie = $document_type['serie'];
                $customer_invoice->folio = $document_type['folio'];
                $customer_invoice->save();
            }

            //Solo guarda los datos sin timbrar en caso de pre-factura
            if(isset($request->pre_invoice)) {
                //Mensaje
                flash(__('general.text_success_pre_customer_invoice'))->success();

            }else{ //Crear CFDI
                //Valida Empresa y PAC para timbrado
                PacHelper::validateSatActions($customer_invoice->company);

                //Crear XML y timbra
                $tmp = $this->$class_cfdi($customer_invoice);

                //Guardar registros de CFDI
                $customer_invoice_cfdi->fill(array_only($tmp, [
                    'pac_id',
                    'cfdi_version',
                    'uuid',
                    'date',
                    'file_xml',
                    'file_xml_pac',
                ]));
                $customer_invoice_cfdi->save();

                //Disminuye folios
                BaseHelper::decrementFolios();

                //Actualiza cotizacion en caso de estar relacionada
                if(!empty($customer_invoice->customer_quotation_id)) {
                    $customer_quotation = CustomerQuotation::findOrFail($customer_invoice->customer_quotation_id);
                    $customer_quotation->status = CustomerQuotation::BILLED;
                    $customer_quotation->save();
                }

                if(!empty($customer_invoice->customer_remission_id)) {
                    $customer_remission = CustomerRemission::findOrFail($customer_invoice->customer_remission_id);
                    $customer_remission->status = CustomerRemission::BILLED;
                    $customer_remission->save();
                }

                //Mensaje
                flash(__('general.text_success_customer_invoice_cfdi'))->success();
            }

            \DB::connection('tenant')->commit();

            if(!isset($request->pre_invoice)) {
                $this->saveCfdiDownloads($customer_invoice, $customer_invoice_cfdi);
            }

        } catch (\Exception $e) {
            //Fix fechas
            $request->merge([
                'date' => Helper::convertSqlToDateTime($request->date),
            ]);
            if (!empty($date_due_fix)) {
                $request->merge([
                    'date_due' => Helper::convertSqlToDate($request->date_due),
                ]);
            }else{
                $request->merge([
                    'date_due' => '',
                ]);
            }

            \DB::connection('tenant')->rollback();
            \Log::error('(' . $customer_invoice->company->taxid . ') ' . $e->getMessage());
            flash((isset($request->pre_invoice) ? '' : __('general.error_cfdi_pac').'<br>').$e->getMessage())->error();
            return back()->withInput();
        }

        //Almacenamiento dropbox
        self::dropboxBackup($customer_invoice);

        //Envio auomatico de correo
        if(isset($request->invoice_and_send)) {
            try{
                $this->autoSendMail($customer_invoice);
            } catch (\Exception $e) {
                flash($e->getMessage())->error();
            }
        }

        //Redireccion
        return redirect()->route('customer-invoices.index',['download_xml_pdf_id' => !empty($request->download_xml_pdf) ? $customer_invoice->id : 0]);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Sales\CustomerInvoice $customer_invoice
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, CustomerInvoice $customer_invoice)
    {
        \DB::connection('tenant')->beginTransaction();
        try {
            //Logica
            if ((int)$customer_invoice->status != CustomerInvoice::CANCEL && $customer_invoice->balance >= $customer_invoice->amount_total) {
                //Actualiza status
                $customer_invoice->updated_uid = \Auth::user()->id;
                $customer_invoice->status = CustomerInvoice::CANCEL;
                //Por autorizar cuando se manda la autorizacion al buzon tributario del SAT
                if($request->cancelable == 2){
                    $customer_invoice->status = CustomerInvoice::CANCEL_PER_AUTHORIZED;
                }
                $customer_invoice->save();

                //Actualiza status CFDI
                $customer_invoice->customerInvoiceCfdi->status = 0;
                $customer_invoice->customerInvoiceCfdi->reason_cancellation_id = $request->reason_cancellation_id;
                $customer_invoice->customerInvoiceCfdi->reason_cancellation_uuid = $request->reason_cancellation_uuid;
                $customer_invoice->customerInvoiceCfdi->save();

                if(!empty($customer_invoice->customer_quotation_id)){
                    $customer_invoice->customerQuotation->status = CustomerQuotation::OPEN;
                    $customer_invoice->customerQuotation->save();
                }
                if(!empty($customer_invoice->customer_remission_id)){
                    $customer_invoice->customerRemission->status = CustomerRemission::OPEN;
                    $customer_invoice->customerRemission->save();
                }

                //Cancelacion del timbre fiscal
                if (!empty($customer_invoice->customerInvoiceCfdi->cfdi_version) && !empty($customer_invoice->customerInvoiceCfdi->uuid)) {
                    //Valida Empresa y PAC para cancelar timbrado
                    PacHelper::validateSatCancelActions($customer_invoice->company,$customer_invoice->customerInvoiceCfdi->pac);

                    //Obtener el sellos del CFDI
                    $path_xml = Helper::setDirectory(CustomerInvoice::PATH_XML_FILES_CI, $customer_invoice->company_id) . '/';
                    $file_xml_pac = $path_xml . $customer_invoice->customerInvoiceCfdi->file_xml_pac;
                    $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($file_xml_pac));
                    $data = Cfdi33Helper::getQuickArrayCfdi($cfdi);

                    //Arreglo temporal para actualizar Customer Invoice CFDI
                    $tmp = [
                        'cancel_date' => Helper::dateTimeToSql(\Date::now()),
                        'cancel_response' => '',
                        'cancel_state' => $request->cancel_state,
                        'rfcR' => $data['cfdi33']->Receptor['Rfc'] ?? $customer_invoice->customer->taxid,
                        'uuid' => $customer_invoice->customerInvoiceCfdi->uuid,
                        'total' => Helper::numberFormat($customer_invoice->amount_total,
                            $customer_invoice->currency->decimal_place, false),
                        'cfdi_type' => $customer_invoice->documentType->cfdiType->code,
                        'cfdi_version' => 'cfdi3.3',
                        'fe' => substr($data['cfdi33']->complemento->timbreFiscalDigital['SelloCFD'],-8), //Los últimos 8 caracteres del Atributo Sello del CFDI
                        'file_xml_pac' => $file_xml_pac,
                        'reason_cancellation_code' => $customer_invoice->customerInvoiceCfdi->reasonCancellation->code ?? '',
                        'reason_cancellation_uuid' => $customer_invoice->customerInvoiceCfdi->reason_cancellation_uuid ?? '',
                    ];

                    //Cancelar Timbrado de XML
                    $class_pac = $customer_invoice->customerInvoiceCfdi->pac->code . 'Cancel';
                    $tmp = PacHelper::$class_pac($tmp,$customer_invoice->company,$customer_invoice->customerInvoiceCfdi->pac);

                    //Guardar registros de CFDI
                    $customer_invoice->customerInvoiceCfdi->fill(array_only($tmp,[
                        'cancel_date',
                        'cancel_response',
                        'cancel_state',
                    ]));
                    $customer_invoice->customerInvoiceCfdi->save();

                    $cfdi_download = CfdiDownload::where('uuid', $customer_invoice->customerInvoiceCfdi->uuid)->where('type', 2)->first();
                    if(!empty($cfdi_download)){
                        $cfdi_download->status = 'Cancelado';
                        $cfdi_download->save();
                    }

                    //Disminuye folios
                    BaseHelper::decrementFolios();

                }
            }
            \DB::connection('tenant')->commit();

            //Mensaje
            flash(__('general.text_form_success_cancel'))->success();


        } catch (\Exception $e) {
            \DB::connection('tenant')->rollback();
            \Log::error('(' . $customer_invoice->company->taxid . ') ' . $e->getMessage());
            flash($e->getMessage())->error();
            return redirect('/sales/customer-invoices');
        }

        //Almacenamiento dropbox
        if ((int)$customer_invoice->status == CustomerInvoice::CANCEL) {
            self::dropboxBackup($customer_invoice,false,true);
        }

        //Redireccion
        return redirect('/sales/customer-invoices');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {

        $this->validate($request, [
            'customer_id' => 'required|integer',
            'branch_office_id' => 'required|integer',
            'date' => 'required|date|date_format:"'.setting('datetime_format', 'd-m-Y H:i:s').'"',
            'date_due' => 'nullable|date|date_format:"'.setting('date_format', 'd-m-Y').'"|after_or_equal:' . Helper::date(Helper::createDateTime($request->date)),
            'currency_id' => 'required|integer',
            'currency_value' => 'required|numeric|min:0.1',
            'payment_term_id' => 'required|integer',
            'payment_way_id' => 'required|integer',
            'payment_method_id' => 'required|integer',
            'cfdi_use_id' => 'required|integer',
            'cfdi_relation_id' => 'nullable|integer|required_with:item_relation',
            'item' => 'required',
            'item.*.name' => 'required',
            'item.*.unit_measure_id' => 'required',
            'item.*.sat_product_id' => 'required',
            'item.*.quantity' => 'required|numeric|min:0.00001',
            'item.*.price_unit' => 'required|numeric|min:0.00001',
            'item.*.discount' => 'nullable|numeric|min:0',
            'item_relation.*.uuid_related' => 'required',

        ], [
            'customer_id.*' => __('sales/customer_invoice.error_customer_id'),
            'branch_office_id.*' => __('sales/customer_invoice.error_branch_office_id'),
            'date.required' => __('sales/customer_invoice.error_date'),
            'date.date' => __('sales/customer_invoice.error_date_format'),
            'date.date_format' => __('sales/customer_invoice.error_date_format'),
            'date_due.date' => __('sales/customer_invoice.error_date_due_format'),
            'date_due.date_format' => __('sales/customer_invoice.error_date_due_format'),
            'date_due.after_or_equal' => __('sales/customer_invoice.error_date_due_after'),
            'currency_id.*' => __('sales/customer_invoice.error_currency_id'),
            'currency_value.*' => __('sales/customer_invoice.error_currency_value'),
            'payment_term_id.*' => __('sales/customer_invoice.error_payment_term_id'),
            'payment_way_id.*' => __('sales/customer_invoice.error_payment_way_id'),
            'payment_method_id.*' => __('sales/customer_invoice.error_payment_method_id'),
            'cfdi_use_id.*' => __('sales/customer_invoice.error_cfdi_use_id'),
            'cfdi_relation_id.*' => __('sales/customer_invoice.error_cfdi_relation_id'),
            'item.required' => __('sales/customer_invoice.error_item'),
            'item.*.name.*' => __('sales/customer_invoice.error_line_name'),
            'item.*.unit_measure_id.*' => __('sales/customer_invoice.error_line_unit_measure_id'),
            'item.*.sat_product_id.*' => __('sales/customer_invoice.error_line_sat_product_id'),
            'item.*.quantity.*' => __('sales/customer_invoice.error_line_quantity'),
            'item.*.price_unit.*' => __('sales/customer_invoice.error_line_price_unit'),
            'item.*.discount.*' => __('sales/customer_invoice.error_line_discount'),
            'item_relation.*.uuid_related.*' => __('sales/customer_invoice.error_relation_uuid_related'),
        ]);

        //Validacion complemento IEDU
        if(\App\Helpers\Helper::companyComplementoCfdiIedu()){
            $this->validate($request, [
                'item.*.customer_iedu_id' => 'required',
                'item.*.iedu_rfc_pago' => ['nullable', 'regex:/^[A-Z&Ñ]{3,4}[0-9]{2}(0[1-9]|1[012])(0[1-9]|[12][0-9]|3[01])[A-Z0-9]{2}[0-9A]$/i'],
            ], [
                'item.*.customer_iedu_id.*' => __('sales/customer_invoice.error_line_iedu_customer_iedu_id'),
                'item.*.iedu_rfc_pago.*' => __('sales/customer_invoice.error_line_iedu_rfc_pago_format'),
            ]);
        }

        //Validaciones manuales
        $validator = \Validator::make([], []);
        if(setting('cfdi_version') == 'cfdi40'){
            if (empty($request->tax_regimen_customer_id)) {
                $validator->after(function ($validator) {
                    $validator->errors()->add('tax_regimen_customer_id', __('sales/customer_invoice.error_tax_regimen_customer_id'));
                });
            }
            $customer = Customer::find($request->customer_id);
            if (empty($customer->postcode) && !in_array($customer->taxid, ['XAXX010101000', 'XEXX010101000'])) {
                $validator->after(function ($validator) {
                    $validator->errors()->add('customer_id', __('sales/customer_invoice.error_postcode_customer'));
                });
            }
        }

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

    }

    /**
     * Crear XML y enviar a timbrar CFDI 3.3
     *
     * @param CustomerInvoice $customer_invoice
     * @return array|\CfdiUtils\Elements\Cfdi33\Concepto|float|int
     * @throws \Exception
     */
    public function cfdi33(CustomerInvoice $customer_invoice)
    {
        $current_entity_loader = @libxml_disable_entity_loader(false);
        try {
            //Logica
            $pac = Pac::findOrFail(setting('default_pac_id')); //PAC
            $customer_invoice_complement = $customer_invoice->customerInvoiceComplement;

            //Arreglo CFDI 3.3
            $cfdi33 = [];
            if (!empty($customer_invoice->serie)) {
                $cfdi33['Serie'] = $customer_invoice->serie;
            }
            $cfdi33['Folio'] = $customer_invoice->folio;
            $cfdi33['Fecha'] = \Date::parse($customer_invoice->date)->format('Y-m-d\TH:i:s');
            //$cfdi33['Sello']
            $cfdi33['FormaPago'] = $customer_invoice->paymentWay->code;
            $cfdi33['NoCertificado'] = $customer_invoice->company->certificate_number;
            //$cfdi33['Certificado']
            $cfdi33['CondicionesDePago'] = $customer_invoice->paymentTerm->name;
            $cfdi33['SubTotal'] = Helper::numberFormat($customer_invoice->amount_untaxed + $customer_invoice->amount_discount,
                $customer_invoice->currency->decimal_place, false);
            if($customer_invoice->amount_discount>0) {
                $cfdi33['Descuento'] = Helper::numberFormat($customer_invoice->amount_discount,
                    $customer_invoice->currency->decimal_place, false);
            }
            $cfdi33['Moneda'] = $customer_invoice->currency->code;
            if ($customer_invoice->currency->code != 'MXN') {
                $cfdi33['TipoCambio'] = Helper::numberFormat($customer_invoice->currency_value, 4, false);
            }
            $cfdi33['Total'] = Helper::numberFormat($customer_invoice->amount_total,
                $customer_invoice->currency->decimal_place, false);
            $cfdi33['TipoDeComprobante'] = $customer_invoice->documentType->cfdiType->code;
            $cfdi33['MetodoPago'] = $customer_invoice->paymentMethod->code;
            $cfdi33['LugarExpedicion'] = $customer_invoice->branchOffice->postcode;
            if (!empty($customer_invoice->confirmacion)) {
                $cfdi33['Confirmacion'] = $customer_invoice->confirmacion;
            }
            //---Cfdi Relacionados
            $cfdi33_relacionados = [];
            $cfdi33_relacionado = [];
            if (!empty($customer_invoice->cfdi_relation_id)) {
                $cfdi33_relacionados['TipoRelacion'] = $customer_invoice->cfdiRelation->code;
                if ($customer_invoice->customerInvoiceRelations->isNotEmpty()) {
                    foreach ($customer_invoice->customerInvoiceRelations as $key => $result) {
                        $cfdi33_relacionado[$key] = [];
                        $cfdi33_relacionado[$key]['UUID'] = $result->uuid_related;
                    }
                }
            }
            //---Emisor
            $cfdi33_emisor = [];
            $cfdi33_emisor['Rfc'] = $customer_invoice->company->taxid;
            $cfdi33_emisor['Nombre'] = trim($customer_invoice->company->name);
            $cfdi33_emisor['RegimenFiscal'] = !empty($customer_invoice->tax_regimen_id) ? $customer_invoice->taxRegimen->code : $customer_invoice->company->taxRegimen->code;
            //---Receptor
            $cfdi33_receptor = [];
            $cfdi33_receptor['Rfc'] = $customer_invoice->customer->taxid;
            $cfdi33_receptor['Nombre'] = trim($customer_invoice->customer->name);
            if ($customer_invoice->customer->taxid == 'XEXX010101000') {
                $cfdi33_receptor['ResidenciaFiscal'] = $customer_invoice->customer->country->code;
                $cfdi33_receptor['NumRegIdTrib'] = $customer_invoice->customer->numid;
            }
            $cfdi33_receptor['UsoCFDI'] = $customer_invoice->cfdiUse->code;
            //---Conceptos
            $cfdi33_conceptos = [];
            $cfdi33_conceptos_traslados = [];
            $cfdi33_conceptos_retenciones = [];
            $local_taxes_total = 0;
            $local_taxes_total_ret = 0;
            foreach ($customer_invoice->customerActiveInvoiceLines as $key => $result) {
                $cfdi33_conceptos [$key]['ClaveProdServ'] = trim($result->satProduct->code);
                if (!empty($result->product->code)) {
                    $cfdi33_conceptos [$key]['NoIdentificacion'] = trim($result->product->code);
                }
                $cfdi33_conceptos [$key]['Cantidad'] = Helper::numberFormat($result->quantity, 6, false);
                $cfdi33_conceptos [$key]['ClaveUnidad'] = trim($result->unitMeasure->code);
                $cfdi33_conceptos [$key]['Unidad'] = str_limit(trim($result->unitMeasure->name),20,'');
                $cfdi33_conceptos [$key]['Descripcion'] = trim($result->name);
                $cfdi33_conceptos [$key]['ValorUnitario'] = Helper::numberFormat($result->price_unit, 6, false);
                $cfdi33_conceptos [$key]['Importe'] = Helper::numberFormat($result->amount_untaxed + $result->amount_discount,
                    2, false);
                if($result->amount_discount>0) {
                    $cfdi33_conceptos [$key]['Descuento'] = Helper::numberFormat($result->amount_discount, 2, false);
                }
                //['InformacionAduanera']
                if(!empty($result->customerInvoiceLineComplement)) {
                    if(!empty($result->customerInvoiceLineComplement->numero_pedimento)) {
                        $cfdi33_conceptos_compl_pedimento[$key] = [];
                        $cfdi33_conceptos_compl_pedimento[$key]['numero_pedimento'] = trim($result->customerInvoiceLineComplement->numero_pedimento);
                    }
                }
                //['CuentaPredial']
                //['ComplementoConcepto']
                //['Parte']
                //Complemento
                if(\App\Helpers\Helper::companyComplementoCfdiIedu($customer_invoice->company_id)) {
                    if(!empty($result->customerInvoiceLineComplement) && !empty($result->customerInvoiceLineComplement->iedu_nombre_alumno)) {
                        $cfdi33_conceptos_compl_iedu[$key] = [];
                        $cfdi33_conceptos_compl_iedu[$key]['xmlns:iedu'] = 'http://www.sat.gob.mx/iedu';
                        $cfdi33_conceptos_compl_iedu[$key]['version'] = '1.0';
                        $cfdi33_conceptos_compl_iedu[$key]['nombreAlumno'] = trim($result->customerInvoiceLineComplement->iedu_nombre_alumno);
                        $cfdi33_conceptos_compl_iedu[$key]['CURP'] = trim($result->customerInvoiceLineComplement->iedu_curp);
                        $cfdi33_conceptos_compl_iedu[$key]['nivelEducativo'] = trim($result->customerInvoiceLineComplement->iedu_nivel_educativo);
                        $cfdi33_conceptos_compl_iedu[$key]['autRVOE'] = trim($result->customerInvoiceLineComplement->iedu_aut_rvoe);
                        if(!empty($result->customerInvoiceLineComplement->iedu_rfc_pago)) {
                            $cfdi33_conceptos_compl_iedu[$key]['rfcPago'] = trim($result->customerInvoiceLineComplement->iedu_rfc_pago);
                        }
                    }
                }

                //Impuestos por concepto
                $cfdi33_conceptos_traslados[$key] = [];
                $cfdi33_conceptos_retenciones[$key] = [];
                if ($result->taxes) {
                    foreach ($result->taxes as $key2 => $result2) {
                        $tmp = 0;
                        $rate = $result2->rate;
                        if($result->quota_ieps > 0){
                            $item_amount_untaxed_per_taxes = round($result->quantity * (($result->price_unit * (100 - $result->discount) / 100) - $result->quota_ieps), 2);
                            if($result->discount_type == 'A'){
                                $item_amount_untaxed_per_taxes -= $result->discount;
                            }
                        }else{
                            $item_amount_untaxed_per_taxes = $result->amount_untaxed;
                        }
                        if ($result2->factor == 'Tasa') {
                            $rate /= 100;
                            $tmp = ($item_amount_untaxed_per_taxes + (!empty($result->fix_amount_untaxed) && !empty($result2->code) ? 0.01 : 0 )) * $rate;
                        } elseif ($result2->factor == 'Cuota') {
                            $tmp = $rate;
                        }
                        $tmp = round($tmp, 2);
                        //Realiza sumatoria de impuestos locales pero los omite de los conceptos
                        if($result2->type == 'R') { //Retenciones
                            if($result2->local_taxes) {
                                $local_taxes_total_ret += abs($tmp);
                                continue;
                            }
                            $cfdi33_conceptos_retenciones[$key][$key2] = [];
                            $cfdi33_conceptos_retenciones[$key][$key2]['Base'] = Helper::numberFormat($item_amount_untaxed_per_taxes,
                                2, false);
                            $cfdi33_conceptos_retenciones[$key][$key2]['Impuesto'] = $result2->code;
                            $cfdi33_conceptos_retenciones[$key][$key2]['TipoFactor'] = $result2->factor; //Para retenciones no hay exento
                            $cfdi33_conceptos_retenciones[$key][$key2]['TasaOCuota'] = Helper::numberFormat(abs($rate),
                                6, false);
                            $cfdi33_conceptos_retenciones[$key][$key2]['Importe'] = Helper::numberFormat(abs($tmp), 2,
                                false);
                        } else { //Traslados
                            if($result2->local_taxes) {
                                $local_taxes_total += abs($tmp);
                                continue;
                            }
                            $cfdi33_conceptos_traslados[$key][$key2] = [];
                            $cfdi33_conceptos_traslados[$key][$key2]['Base'] = Helper::numberFormat($item_amount_untaxed_per_taxes,
                                2, false);
                            $cfdi33_conceptos_traslados[$key][$key2]['Impuesto'] = $result2->code;
                            $cfdi33_conceptos_traslados[$key][$key2]['TipoFactor'] = $result2->factor; //Para retenciones no hay exento
                            if ($result2->factor != 'Exento') {
                                $cfdi33_conceptos_traslados[$key][$key2]['TasaOCuota'] = Helper::numberFormat(abs($rate),
                                    6, false);
                                $cfdi33_conceptos_traslados[$key][$key2]['Importe'] = Helper::numberFormat(abs($tmp), 2,
                                    false);
                            }
                        }
                    }
                }
            }
            //Impuestos
            $cfdi33_retenciones = [];
            $cfdi33_traslados = [];
            if ($customer_invoice->customerInvoiceTaxes->isNotEmpty()) {
                foreach ($customer_invoice->customerInvoiceTaxes as $key => $result) {
                    $tmp = $result->amount_tax;
                    $rate = $result->tax->rate;
                    if ($result->tax->factor == 'Tasa') {
                        $rate /= 100;
                    }
                    //Omite los impuestos locales
                    if($result->tax->local_taxes) {
                        continue;
                    }
                    if($result->tax->type == 'R') { //Retenciones
                        $cfdi33_retenciones[$key] = [];
                        $cfdi33_retenciones[$key]['Impuesto'] = $result->tax->code;
                        $cfdi33_retenciones[$key]['Importe'] = Helper::numberFormat(abs($tmp), $customer_invoice->currency->decimal_place, false);
                    } else { //Traslados
                        if ($result->tax->factor != 'Exento') {
                            $cfdi33_traslados[$key] = [];
                            $cfdi33_traslados[$key]['Impuesto'] = $result->tax->code;
                            $cfdi33_traslados[$key]['TipoFactor'] = $result->tax->factor;
                            $cfdi33_traslados[$key]['TasaOCuota'] = Helper::numberFormat(abs($rate), 6, false);
                            $cfdi33_traslados[$key]['Importe'] = Helper::numberFormat(abs($tmp), 2, false);
                        }
                    }
                }
            }
            $cfdi33_impuestos = [];
            if (abs($customer_invoice->amount_tax_ret) > 0 || !empty($cfdi33_retenciones)) {
                $tmp = Helper::numberFormat(abs($customer_invoice->amount_tax_ret) - $local_taxes_total_ret, $customer_invoice->currency->decimal_place, false);
                if($tmp > 0 || !empty($cfdi33_retenciones)){
                    $cfdi33_impuestos['TotalImpuestosRetenidos'] = $tmp;
                }
            }
            if (abs($customer_invoice->amount_tax) > 0 || !empty($cfdi33_traslados)) {
                $cfdi33_impuestos['TotalImpuestosTrasladados'] = Helper::numberFormat(abs($customer_invoice->amount_tax) - $local_taxes_total,
                    $customer_invoice->currency->decimal_place, false);
            }

            //Genera XML
            $certificado = new \CfdiUtils\Certificado\Certificado(\Storage::path($customer_invoice->company->pathFileCer()));
            $creator = new \CfdiUtils\CfdiCreator33($cfdi33, $certificado);
            $creator->setXmlResolver(PacHelper::resourcePathCfdiUtils()); //Almacenamiento local
            $comprobante = $creator->comprobante();
            //Complemento IEDU
            $schema_location_iedu='';
            $schema_location_implocal='';
            $schema_location_leyendas_fisc='';
            $schema_location_ine='';
            if(\App\Helpers\Helper::companyComplementoCfdiIedu($customer_invoice->company_id) && !empty($customer_invoice->customerActiveInvoiceLines->first()->customerInvoiceLineComplement->iedu_nombre_alumno)) {
                $schema_location_iedu=' http://www.sat.gob.mx/iedu http://www.sat.gob.mx/sitio_internet/cfd/iedu/iedu.xsd';
            }
            //Impuestos locales
            if($local_taxes_total > 0 || $local_taxes_total_ret > 0) {
                $schema_location_implocal=' http://www.sat.gob.mx/implocal http://www.sat.gob.mx/sitio_internet/cfd/implocal/implocal.xsd';
            }
            //Complemento Leyendas Fiscales
            if($customer_invoice->customerActiveInvoiceTaxLegends->isNotEmpty()){
                $schema_location_leyendas_fisc=' http://www.sat.gob.mx/leyendasFiscales http://www.sat.gob.mx/sitio_internet/cfd/leyendasFiscales/leyendasFisc.xsd';
            }
            //Complemento Ine
            if(!empty($customer_invoice_complement->ine_process_type)){
                $schema_location_ine=' http://www.sat.gob.mx/ine http://www.sat.gob.mx/sitio_internet/cfd/ine/ine11.xsd';
            }
            if(!empty($schema_location_iedu) || !empty($schema_location_implocal) || !empty($schema_location_leyendas_fisc) || !empty($schema_location_ine)){
                $comprobante->addAttributes([
                    'xsi:schemaLocation' => 'http://www.sat.gob.mx/cfd/3 http://www.sat.gob.mx/sitio_internet/cfd/3/cfdv33.xsd' . $schema_location_iedu . $schema_location_implocal . $schema_location_leyendas_fisc . $schema_location_ine,
                ]);
                if(!empty($schema_location_implocal)){
                    $comprobante->addAttributes([
                        'xmlns:implocal' => 'http://www.sat.gob.mx/implocal'
                    ]);
                }
                if(!empty($schema_location_leyendas_fisc)){
                    $comprobante->addAttributes([
                        'xmlns:leyendasFisc' => 'http://www.sat.gob.mx/leyendasFiscales'
                    ]);
                }
                if(!empty($schema_location_ine)){
                    $comprobante->addAttributes([
                        'xmlns:ine' => 'http://www.sat.gob.mx/ine'
                    ]);
                }
            }

            if (!empty($cfdi33_relacionados)) {
                $comprobante->addCfdiRelacionados($cfdi33_relacionados);
            }
            if (!empty($cfdi33_relacionado)) {
                foreach ($cfdi33_relacionado as $key => $result) {
                    $comprobante->addCfdiRelacionado($result);
                }
            }
            $comprobante->addEmisor($cfdi33_emisor);
            $comprobante->addReceptor($cfdi33_receptor);
            //Conceptos
            foreach ($cfdi33_conceptos as $key => $result) {
                $concepto = $comprobante->addConcepto($result);
                if (!empty($cfdi33_conceptos_traslados[$key])) {
                    foreach ($cfdi33_conceptos_traslados[$key] as $result2) {
                        $concepto->multiTraslado($result2);
                    }
                }
                if (!empty($cfdi33_conceptos_retenciones[$key])) {
                    foreach ($cfdi33_conceptos_retenciones[$key] as $result2) {
                        $concepto->multiRetencion($result2);
                    }
                }
                //Complemento IEDU
                if(\App\Helpers\Helper::companyComplementoCfdiIedu($customer_invoice->company_id) && !empty($cfdi33_conceptos_compl_iedu[$key]['nombreAlumno'])) {
                    $nodo_iedu = new \CfdiUtils\Nodes\Node(
                        'iedu:instEducativas', // nombre del elemento raíz
                        $cfdi33_conceptos_compl_iedu[$key]
                    );
                    $concepto->addComplementoConcepto([], [$nodo_iedu]);
                }

                //Numero pedimento
                if(!empty($cfdi33_conceptos_compl_pedimento[$key]['numero_pedimento'])) {
                    $concepto->addInformacionAduanera(['NumeroPedimento' => $cfdi33_conceptos_compl_pedimento[$key]['numero_pedimento']]);
                }
            }
            //Impuestos
            if(!empty($cfdi33_impuestos)) {
                $comprobante->addImpuestos($cfdi33_impuestos);
                if (!empty($cfdi33_retenciones)) {
                    foreach ($cfdi33_retenciones as $result2) {
                        $comprobante->multiRetencion($result2);
                    }
                }
                if (!empty($cfdi33_traslados)) {
                    foreach ($cfdi33_traslados as $result2) {
                        $comprobante->multiTraslado($result2);
                    }
                }
            }

            //Complemento de impuestos locales
            if($local_taxes_total > 0 || $local_taxes_total_ret > 0) {
                $implocal = [];
                $implocal['version'] = '1.0';
                $implocal['TotaldeRetenciones'] = Helper::numberFormat($local_taxes_total_ret, 2, false);
                $implocal['TotaldeTraslados'] = Helper::numberFormat($local_taxes_total, 2, false);
                $nodo_implocal = new \CfdiUtils\Nodes\Node(
                    'implocal:ImpuestosLocales', // nombre del elemento raíz
                    $implocal
                );
                if ($customer_invoice->customerInvoiceTaxes->isNotEmpty()) {
                    foreach ($customer_invoice->customerInvoiceTaxes as $key => $result) {
                        $tmp = $result->amount_tax;
                        $rate = $result->tax->rate;
                        if ($result->tax->factor == 'Tasa') {
                            $rate = $rate;
                        }
                        //Solo los impuestos locales
                        if ($result->tax->local_taxes) {
                            if($result->tax->type == 'R'){
                                $implocal_retenciones = [];
                                $implocal_retenciones['ImpLocRetenido'] = $result->tax->name;
                                $implocal_retenciones['TasadeRetencion'] = Helper::numberFormat(abs($rate), 2, false);
                                $implocal_retenciones['Importe'] = Helper::numberFormat(abs($tmp), 2, false);
                                $nodo_implocal_retenciones = new \CfdiUtils\Nodes\Node(
                                    'implocal:RetencionesLocales', // nombre del elemento raíz
                                    $implocal_retenciones
                                );
                                $nodo_implocal->addChild($nodo_implocal_retenciones);
                            }else {
                                $implocal_traslados = [];
                                $implocal_traslados['ImpLocTrasladado'] = $result->tax->name;
                                $implocal_traslados['TasadeTraslado'] = Helper::numberFormat(abs($rate), 2, false);
                                $implocal_traslados['Importe'] = Helper::numberFormat(abs($tmp), 2, false);
                                $nodo_implocal_traslados = new \CfdiUtils\Nodes\Node(
                                    'implocal:TrasladosLocales', // nombre del elemento raíz
                                    $implocal_traslados
                                );
                                $nodo_implocal->addChild($nodo_implocal_traslados);
                            }
                        }
                    }
                }
                //Agregamos al complemento
                $comprobante->addComplemento($nodo_implocal);
            }

            //Complemento leyendas fiscales
            if($customer_invoice->customerActiveInvoiceTaxLegends->isNotEmpty()){
                $leyendas_fisc = [];
                $leyendas_fisc['version'] = '1.0';
                $nodo_leyendas_fisc = new \CfdiUtils\Nodes\Node(
                    'leyendasFisc:LeyendasFiscales', // nombre del elemento raíz
                    $leyendas_fisc
                );
                foreach ($customer_invoice->customerActiveInvoiceTaxLegends as $key => $result) {
                    $leyenda = [];
                    $leyenda['disposicionFiscal'] = $result->disposicion_fiscal;
                    $leyenda['norma'] = $result->norma;
                    $leyenda['textoLeyenda'] = $result->texto_leyenda;
                    $nodo_leyenda = new \CfdiUtils\Nodes\Node(
                        'leyendasFisc:Leyenda', // nombre del elemento raíz
                        $leyenda
                    );
                    $nodo_leyendas_fisc->addChild($nodo_leyenda);
                }
                //Agregamos al complemento
                $comprobante->addComplemento($nodo_leyendas_fisc);
            }

            //Complemento INE
            if(!empty($customer_invoice_complement->ine_process_type)){
                $ine = [];
                $ine['Version'] = '1.1';
                $ine['TipoProceso'] = $customer_invoice_complement->ine_process_type;
                $ine['TipoComite'] = $customer_invoice_complement->ine_committee_type;
                $ine['IdContabilidad'] = $customer_invoice_complement->ine_id_accounting;
                $nodo_ine = new \CfdiUtils\Nodes\Node(
                    'ine:INE', // nombre del elemento raíz
                    $ine
                );
                foreach ($customer_invoice->customerActiveInvoiceInes as $key => $result) {
                    $entidad = [];
                    $entidad['ClaveEntidad'] = $result->ine_entity;
                    $entidad['Ambito'] = $result->ambit;
                    $nodo_entidad = new \CfdiUtils\Nodes\Node(
                        'ine:Entidad', // nombre del elemento raíz
                        $entidad
                    );
                    $id_contabilidades = explode(',', $result->id_accounting);
                    if(!empty($id_contabilidades)){
                        foreach($id_contabilidades as $key => $result2){
                            $contabilidad = [];
                            $contabilidad['IdContabilidad'] = trim($result2);
                            $nodo_contabilidad = new \CfdiUtils\Nodes\Node(
                                'ine:Contabilidad', // nombre del elemento raíz
                                $contabilidad
                            );
                            $nodo_entidad->addChild($nodo_contabilidad);
                        }
                    }
                    $nodo_ine->addChild($nodo_entidad);
                }
                //Agregamos al complemento
                $comprobante->addComplemento($nodo_ine);
            }

            //Método de ayuda para establecer las sumas del comprobante e impuestos con base en la suma de los conceptos y la agrupación de sus impuestos
            //$creator->addSumasConceptos(null, 2);
            //Método de ayuda para generar el sello (obtener la cadena de origen y firmar con la llave privada)
            $creator->addSello('file://' . \Storage::path($customer_invoice->company->pathFileKeyPassPem()), Crypt::decryptString($customer_invoice->company->password_key));
            //Valida la estructura
            //$creator->validate();

            //Guarda XML
            //if(\Auth::user()->email == 'sksistemas@hotmail.com'){
                //dd($creator->asXml());
            //}
            $path_xml = Helper::setDirectory(CustomerInvoice::PATH_XML_FILES_CI, $customer_invoice->company_id) . '/';
            $file_xml = Helper::makeDirectoryCfdi($path_xml) . '/' . Str::random(40) . '.xml';
            $creator->saveXml(\Storage::path($path_xml . $file_xml));


            //Arreglo temporal para actualizar Customer Invoice CFDI
            $tmp = [
                'pac_id' => $pac->id,
                'cfdi_version' => setting('cfdi_version'),
                'uuid' => '',
                'date' => '',
                'path_xml' => $path_xml,
                'file_xml' => $file_xml,
                'file_xml_pac' => '',
                'pac' => $pac,
            ];

            //Timbrado de XML
            $class_pac = $pac->code;
            $tmp = PacHelper::$class_pac($tmp, $creator);

            return $tmp;
        } catch (\Exception $e) {
            throw $e;
        }
        @libxml_disable_entity_loader($current_entity_loader);
    }

    /**
     * Crear XML y enviar a timbrar CFDI 4.0
     *
     * @param CustomerInvoice $customer_invoice
     * @return array|\CfdiUtils\Elements\Cfdi33\Concepto|float|int
     * @throws \Exception
     */
    public function cfdi40(CustomerInvoice $customer_invoice)
    {
        $current_entity_loader = @libxml_disable_entity_loader(false);
        try {
            //Logica
            $pac = Pac::findOrFail(setting('default_pac_id')); //PAC
            $customer_invoice_complement = $customer_invoice->customerInvoiceComplement;

            //Arreglo CFDI 3.3
            $cfdi33 = [];
            if (!empty($customer_invoice->serie)) {
                $cfdi33['Serie'] = $customer_invoice->serie;
            }
            $cfdi33['Folio'] = $customer_invoice->folio;
            $cfdi33['Fecha'] = \Date::parse($customer_invoice->date)->format('Y-m-d\TH:i:s');
            //$cfdi33['Sello']
            $cfdi33['FormaPago'] = $customer_invoice->paymentWay->code;
            $cfdi33['NoCertificado'] = $customer_invoice->company->certificate_number;
            //$cfdi33['Certificado']
            $cfdi33['CondicionesDePago'] = $customer_invoice->paymentTerm->name;
            $cfdi33['SubTotal'] = Helper::numberFormat($customer_invoice->amount_untaxed + $customer_invoice->amount_discount,
                $customer_invoice->currency->decimal_place, false);
            if($customer_invoice->amount_discount>0) {
                $cfdi33['Descuento'] = Helper::numberFormat($customer_invoice->amount_discount,
                    $customer_invoice->currency->decimal_place, false);
            }
            $cfdi33['Moneda'] = $customer_invoice->currency->code;
            if ($customer_invoice->currency->code != 'MXN') {
                $cfdi33['TipoCambio'] = Helper::numberFormat($customer_invoice->currency_value, 4, false);
            }
            $cfdi33['Total'] = Helper::numberFormat($customer_invoice->amount_total,
                $customer_invoice->currency->decimal_place, false);
            $cfdi33['TipoDeComprobante'] = $customer_invoice->documentType->cfdiType->code;
            $cfdi33['Exportacion'] = '01';
            $cfdi33['MetodoPago'] = $customer_invoice->paymentMethod->code;
            $cfdi33['LugarExpedicion'] = $customer_invoice->branchOffice->postcode;
            if (!empty($customer_invoice->confirmacion)) {
                $cfdi33['Confirmacion'] = $customer_invoice->confirmacion;
            }
            //---Informacion global
            $cfdi33_informacion_global = [];
            if ($customer_invoice->customer->taxid == 'XAXX010101000' && $customer_invoice->customer->name == 'PUBLICO EN GENERAL') {
                $cfdi33_informacion_global['Periodicidad'] = $customer_invoice->global_periodicity ?? '01';
                $cfdi33_informacion_global['Meses'] = str_pad($customer_invoice->global_month ?? date('m'), 2, '0', STR_PAD_LEFT);
                $cfdi33_informacion_global['Año'] = $customer_invoice->global_year ?? date('Y');
            }
            //---Cfdi Relacionados
            $cfdi33_relacionados = [];
            $cfdi33_relacionado = [];
            if (!empty($customer_invoice->cfdi_relation_id)) {
                $cfdi33_relacionados['TipoRelacion'] = $customer_invoice->cfdiRelation->code;
                if ($customer_invoice->customerInvoiceRelations->isNotEmpty()) {
                    foreach ($customer_invoice->customerInvoiceRelations as $key => $result) {
                        $cfdi33_relacionado[$key] = [];
                        $cfdi33_relacionado[$key]['UUID'] = $result->uuid_related;
                    }
                }
            }
            //---Emisor
            $cfdi33_emisor = [];
            $cfdi33_emisor['Rfc'] = $customer_invoice->company->taxid;
            $cfdi33_emisor['Nombre'] = trim($customer_invoice->company->name);
            $cfdi33_emisor['RegimenFiscal'] = !empty($customer_invoice->tax_regimen_id) ? $customer_invoice->taxRegimen->code : $customer_invoice->company->taxRegimen->code;
            $cfdi33_emisor['FacAtrAdquirente'] = null;
            //---Receptor
            $cfdi33_receptor = [];
            $cfdi33_receptor['Rfc'] = $customer_invoice->customer->taxid;
            $cfdi33_receptor['Nombre'] = trim($customer_invoice->customer->name);
            if (!in_array($customer_invoice->customer->taxid, ['XAXX010101000','XEXX010101000'])) {
                $cfdi33_receptor['DomicilioFiscalReceptor'] = $customer_invoice->customer->postcode;
            }else{
                $cfdi33_receptor['DomicilioFiscalReceptor'] = $customer_invoice->branchOffice->postcode;
            }
            if ($customer_invoice->customer->taxid == 'XEXX010101000') {
                $cfdi33_receptor['ResidenciaFiscal'] = $customer_invoice->customer->country->code;
                $cfdi33_receptor['NumRegIdTrib'] = $customer_invoice->customer->numid;
            }
            $cfdi33_receptor['RegimenFiscalReceptor'] = !empty($customer_invoice->tax_regimen_customer_id) ? $customer_invoice->taxRegimenCustomer->code : $customer_invoice->customer->taxRegimen->code;
            $cfdi33_receptor['UsoCFDI'] = $customer_invoice->cfdiUse->code;
            //---Conceptos
            $cfdi33_conceptos = [];
            $cfdi33_conceptos_traslados = [];
            $cfdi33_conceptos_retenciones = [];
            $local_taxes_total = 0;
            $local_taxes_total_ret = 0;
            foreach ($customer_invoice->customerActiveInvoiceLines as $key => $result) {
                $cfdi33_conceptos [$key]['ClaveProdServ'] = trim($result->satProduct->code);
                if (!empty($result->product->code)) {
                    $cfdi33_conceptos [$key]['NoIdentificacion'] = trim($result->product->code);
                }
                $cfdi33_conceptos [$key]['Cantidad'] = Helper::numberFormat($result->quantity, 6, false);
                $cfdi33_conceptos [$key]['ClaveUnidad'] = trim($result->unitMeasure->code);
                $cfdi33_conceptos [$key]['Unidad'] = str_limit(trim($result->unitMeasure->name),20,'');
                $cfdi33_conceptos [$key]['Descripcion'] = trim($result->name);
                $cfdi33_conceptos [$key]['ValorUnitario'] = Helper::numberFormatRealDecimalPlace($result->price_unit, false);
                $cfdi33_conceptos [$key]['Importe'] = Helper::numberFormat($result->amount_untaxed + $result->amount_discount,
                    2, false);
                if($result->amount_discount>0) {
                    $cfdi33_conceptos [$key]['Descuento'] = Helper::numberFormat($result->amount_discount, 2, false);
                }

                //['InformacionAduanera']
                if(!empty($result->customerInvoiceLineComplement)) {
                    if(!empty($result->customerInvoiceLineComplement->numero_pedimento)) {
                        $cfdi33_conceptos_compl_pedimento[$key] = [];
                        $cfdi33_conceptos_compl_pedimento[$key]['numero_pedimento'] = trim($result->customerInvoiceLineComplement->numero_pedimento);
                    }
                }
                //['CuentaPredial']
                //['ComplementoConcepto']
                //['Parte']
                //Complemento
                if(\App\Helpers\Helper::companyComplementoCfdiIedu($customer_invoice->company_id)) {
                    if(!empty($result->customerInvoiceLineComplement) && !empty($result->customerInvoiceLineComplement->iedu_nombre_alumno)) {
                        $cfdi33_conceptos_compl_iedu[$key] = [];
                        $cfdi33_conceptos_compl_iedu[$key]['xmlns:iedu'] = 'http://www.sat.gob.mx/iedu';
                        $cfdi33_conceptos_compl_iedu[$key]['version'] = '1.0';
                        $cfdi33_conceptos_compl_iedu[$key]['nombreAlumno'] = trim($result->customerInvoiceLineComplement->iedu_nombre_alumno);
                        $cfdi33_conceptos_compl_iedu[$key]['CURP'] = trim($result->customerInvoiceLineComplement->iedu_curp);
                        $cfdi33_conceptos_compl_iedu[$key]['nivelEducativo'] = trim($result->customerInvoiceLineComplement->iedu_nivel_educativo);
                        $cfdi33_conceptos_compl_iedu[$key]['autRVOE'] = trim($result->customerInvoiceLineComplement->iedu_aut_rvoe);
                        if(!empty($result->customerInvoiceLineComplement->iedu_rfc_pago)) {
                            $cfdi33_conceptos_compl_iedu[$key]['rfcPago'] = trim($result->customerInvoiceLineComplement->iedu_rfc_pago);
                        }
                    }
                }

                //Impuestos por concepto
                $cfdi33_conceptos_traslados[$key] = [];
                $cfdi33_conceptos_retenciones[$key] = [];
                if ($result->taxes) {
                    foreach ($result->taxes as $key2 => $result2) {
                        $tmp = 0;
                        $rate = $result2->rate;
                        if($result->quota_ieps > 0){
                            $item_amount_untaxed_per_taxes = round($result->quantity * (($result->price_unit * (100 - $result->discount) / 100) - $result->quota_ieps), 2);
                            if($result->discount_type == 'A'){
                                $item_amount_untaxed_per_taxes -= $result->discount;
                            }
                        }else{
                            $item_amount_untaxed_per_taxes = $result->amount_untaxed;
                        }
                        if ($result2->factor == 'Tasa') {
                            $rate /= 100;
                            $tmp = ($item_amount_untaxed_per_taxes + (!empty($result->fix_amount_untaxed) && !empty($result2->code) ? 0.01 : 0 )) * $rate;
                        } elseif ($result2->factor == 'Cuota') {
                            $tmp = $rate;
                        }
                        $tmp = round($tmp, 2);
                        //Realiza sumatoria de impuestos locales pero los omite de los conceptos
                        if($result2->type == 'R') { //Retenciones
                            if($result2->local_taxes) {
                                $local_taxes_total_ret += abs($tmp);
                                continue;
                            }
                            $cfdi33_conceptos_retenciones[$key][$key2] = [];
                            $cfdi33_conceptos_retenciones[$key][$key2]['Base'] = Helper::numberFormat($item_amount_untaxed_per_taxes,
                                2, false);
                            $cfdi33_conceptos_retenciones[$key][$key2]['Impuesto'] = $result2->code;
                            $cfdi33_conceptos_retenciones[$key][$key2]['TipoFactor'] = $result2->factor; //Para retenciones no hay exento
                            $cfdi33_conceptos_retenciones[$key][$key2]['TasaOCuota'] = Helper::numberFormat(abs($rate),
                                6, false);
                            $cfdi33_conceptos_retenciones[$key][$key2]['Importe'] = Helper::numberFormat(abs($tmp), 2,
                                false);
                        } else { //Traslados
                            if($result2->local_taxes) {
                                $local_taxes_total += abs($tmp);
                                continue;
                            }
                            $cfdi33_conceptos_traslados[$key][$key2] = [];
                            $cfdi33_conceptos_traslados[$key][$key2]['Base'] = Helper::numberFormat($item_amount_untaxed_per_taxes,
                                2, false);
                            $cfdi33_conceptos_traslados[$key][$key2]['Impuesto'] = $result2->code;
                            $cfdi33_conceptos_traslados[$key][$key2]['TipoFactor'] = $result2->factor; //Para retenciones no hay exento
                            if ($result2->factor != 'Exento') {
                                $cfdi33_conceptos_traslados[$key][$key2]['TasaOCuota'] = Helper::numberFormat(abs($rate),
                                    6, false);
                                $cfdi33_conceptos_traslados[$key][$key2]['Importe'] = Helper::numberFormat(abs($tmp), 2,
                                    false);
                            }
                        }
                    }
                }
                $cfdi33_conceptos [$key]['ObjetoImp'] = !empty($result->tax_object) ? $result->tax_object  : ((empty($cfdi33_conceptos_traslados[$key]) && empty($cfdi33_conceptos_retenciones[$key])) ? '01' : '02');
            }
            //Impuestos
            $cfdi33_retenciones = [];
            $cfdi33_traslados = [];
            $cfdi33_exento = null;
            if ($customer_invoice->customerInvoiceTaxes->isNotEmpty()) {
                foreach ($customer_invoice->customerInvoiceTaxes as $key => $result) {
                    $tmp = $result->amount_tax;
                    $rate = $result->tax->rate;
                    if ($result->tax->factor == 'Tasa') {
                        $rate /= 100;
                    }
                    //Omite los impuestos locales
                    if($result->tax->local_taxes) {
                        continue;
                    }
                    if($result->tax->type == 'R') { //Retenciones
                        $cfdi33_retenciones[$key] = [];
                        $cfdi33_retenciones[$key]['Impuesto'] = $result->tax->code;
                        $cfdi33_retenciones[$key]['Importe'] = Helper::numberFormat(abs($tmp), $customer_invoice->currency->decimal_place, false);
                    } else { //Traslados
                        $cfdi33_traslados[$key] = [];
                        $cfdi33_traslados[$key]['Base'] = Helper::numberFormat(abs($result->amount_base), 2, false);
                        $cfdi33_traslados[$key]['Impuesto'] = $result->tax->code;
                        $cfdi33_traslados[$key]['TipoFactor'] = $result->tax->factor;
                        if($result->tax->factor != 'Exento'){
                            $cfdi33_traslados[$key]['TasaOCuota'] = Helper::numberFormat(abs($rate), 6, false);
                            $cfdi33_traslados[$key]['Importe'] = Helper::numberFormat(abs($tmp), 2, false);
                        }
						if($result->tax->factor == 'Exento' && ($cfdi33_exento || is_null($cfdi33_exento)))
						{
							$cfdi33_exento = true;
						}
						if($result->tax->factor != 'Exento')
						{
							$cfdi33_exento = false;
						}
                    }
                }
            }
            $cfdi33_impuestos = [];
            if (abs($customer_invoice->amount_tax_ret) > 0 || !empty($cfdi33_retenciones)) {
                $tmp = Helper::numberFormat(abs($customer_invoice->amount_tax_ret) - $local_taxes_total_ret, $customer_invoice->currency->decimal_place, false);
                if($tmp > 0 || !empty($cfdi33_retenciones)){
                    $cfdi33_impuestos['TotalImpuestosRetenidos'] = $tmp;
                }
            }
            if (abs($customer_invoice->amount_tax) > 0 || !empty($cfdi33_traslados)) {
                $cfdi33_impuestos['TotalImpuestosTrasladados'] = Helper::numberFormat(abs($customer_invoice->amount_tax) - $local_taxes_total,
                    $customer_invoice->currency->decimal_place, false);
            }
            if($cfdi33_exento){
                $cfdi33_impuestos['TotalImpuestosTrasladados'] = null;
            }

            //Genera XML
            $certificado = new \CfdiUtils\Certificado\Certificado(\Storage::path($customer_invoice->company->pathFileCer()));
            $creator = new \CfdiUtils\CfdiCreator40($cfdi33, $certificado);
            $creator->setXmlResolver(PacHelper::resourcePathCfdiUtils()); //Almacenamiento local
            $comprobante = $creator->comprobante();
            //Complemento IEDU
            $schema_location_iedu='';
            $schema_location_implocal='';
            $schema_location_leyendas_fisc='';
            $schema_location_ine='';
            if(\App\Helpers\Helper::companyComplementoCfdiIedu($customer_invoice->company_id) && !empty($customer_invoice->customerActiveInvoiceLines->first()->customerInvoiceLineComplement->iedu_nombre_alumno)) {
                $schema_location_iedu=' http://www.sat.gob.mx/iedu http://www.sat.gob.mx/sitio_internet/cfd/iedu/iedu.xsd';
            }
            //Impuestos locales
            if($local_taxes_total > 0 || $local_taxes_total_ret > 0) {
                $schema_location_implocal=' http://www.sat.gob.mx/implocal http://www.sat.gob.mx/sitio_internet/cfd/implocal/implocal.xsd';
            }
            //Complemento Leyendas Fiscales
            if($customer_invoice->customerActiveInvoiceTaxLegends->isNotEmpty()){
                $schema_location_leyendas_fisc=' http://www.sat.gob.mx/leyendasFiscales http://www.sat.gob.mx/sitio_internet/cfd/leyendasFiscales/leyendasFisc.xsd';
            }
            //Complemento Ine
            if(!empty($customer_invoice_complement->ine_process_type)){
                $schema_location_ine=' http://www.sat.gob.mx/ine http://www.sat.gob.mx/sitio_internet/cfd/ine/ine11.xsd';
            }
            if(!empty($schema_location_iedu) || !empty($schema_location_implocal) || !empty($schema_location_leyendas_fisc) || !empty($schema_location_ine)){
                $comprobante->addAttributes([
                    'xsi:schemaLocation' => 'http://www.sat.gob.mx/cfd/4 http://www.sat.gob.mx/sitio_internet/cfd/4/cfdv40.xsd' . $schema_location_iedu . $schema_location_implocal . $schema_location_leyendas_fisc . $schema_location_ine,
                ]);
                if(!empty($schema_location_implocal)){
                    $comprobante->addAttributes([
                        'xmlns:implocal' => 'http://www.sat.gob.mx/implocal'
                    ]);
                }
                if(!empty($schema_location_leyendas_fisc)){
                    $comprobante->addAttributes([
                        'xmlns:leyendasFisc' => 'http://www.sat.gob.mx/leyendasFiscales'
                    ]);
                }
                if(!empty($schema_location_ine)){
                    $comprobante->addAttributes([
                        'xmlns:ine' => 'http://www.sat.gob.mx/ine'
                    ]);
                }
            }
            if (!empty($cfdi33_informacion_global)) {
                $comprobante->addInformacionGlobal($cfdi33_informacion_global);
            }
            if (!empty($cfdi33_relacionados)) {
                $relacionados = $comprobante->addCfdiRelacionados($cfdi33_relacionados);
                if (!empty($cfdi33_relacionado)) {
                    foreach ($cfdi33_relacionado as $key => $result) {
                        $relacionados->addCfdiRelacionado($result);
                    }
                }
            }
            $comprobante->addEmisor($cfdi33_emisor);
            $comprobante->addReceptor($cfdi33_receptor);
            //Conceptos
            foreach ($cfdi33_conceptos as $key => $result) {
                $concepto = $comprobante->addConcepto($result);
                if (!empty($cfdi33_conceptos_traslados[$key])) {
                    foreach ($cfdi33_conceptos_traslados[$key] as $result2) {
                        $concepto->multiTraslado($result2);
                    }
                }
                if (!empty($cfdi33_conceptos_retenciones[$key])) {
                    foreach ($cfdi33_conceptos_retenciones[$key] as $result2) {
                        $concepto->multiRetencion($result2);
                    }
                }
                //Complemento IEDU
                if(\App\Helpers\Helper::companyComplementoCfdiIedu($customer_invoice->company_id) && !empty($cfdi33_conceptos_compl_iedu[$key]['nombreAlumno'])) {
                    $nodo_iedu = new \CfdiUtils\Nodes\Node(
                        'iedu:instEducativas', // nombre del elemento raíz
                        $cfdi33_conceptos_compl_iedu[$key]
                    );
                    $concepto->addComplementoConcepto($nodo_iedu);
                }

                //Numero pedimento
                if(!empty($cfdi33_conceptos_compl_pedimento[$key]['numero_pedimento'])) {
                    $concepto->addInformacionAduanera(['NumeroPedimento' => $cfdi33_conceptos_compl_pedimento[$key]['numero_pedimento']]);
                }
            }
            //Impuestos
            if(!empty($cfdi33_impuestos)) {
                $comprobante->addImpuestos($cfdi33_impuestos);
                if (!empty($cfdi33_retenciones)) {
                    foreach ($cfdi33_retenciones as $result2) {
                        $comprobante->multiRetencion($result2);
                    }
                }
                if (!empty($cfdi33_traslados)) {
                    foreach ($cfdi33_traslados as $result2) {
                        $comprobante->multiTraslado($result2);
                    }
                }
            }

            //Complemento de impuestos locales
            if($local_taxes_total > 0 || $local_taxes_total_ret > 0) {
                $implocal = [];
                $implocal['version'] = '1.0';
                $implocal['TotaldeRetenciones'] = Helper::numberFormat($local_taxes_total_ret, 2, false);
                $implocal['TotaldeTraslados'] = Helper::numberFormat($local_taxes_total, 2, false);
                $nodo_implocal = new \CfdiUtils\Nodes\Node(
                    'implocal:ImpuestosLocales', // nombre del elemento raíz
                    $implocal
                );
                if ($customer_invoice->customerInvoiceTaxes->isNotEmpty()) {
                    foreach ($customer_invoice->customerInvoiceTaxes as $key => $result) {
                        $tmp = $result->amount_tax;
                        $rate = $result->tax->rate;
                        if ($result->tax->factor == 'Tasa') {
                            $rate = $rate;
                        }
                        //Solo los impuestos locales
                        if ($result->tax->local_taxes) {
                            if($result->tax->type == 'R'){
                                $implocal_retenciones = [];
                                $implocal_retenciones['ImpLocRetenido'] = $result->tax->name;
                                $implocal_retenciones['TasadeRetencion'] = Helper::numberFormat(abs($rate), 2, false);
                                $implocal_retenciones['Importe'] = Helper::numberFormat(abs($tmp), 2, false);
                                $nodo_implocal_retenciones = new \CfdiUtils\Nodes\Node(
                                    'implocal:RetencionesLocales', // nombre del elemento raíz
                                    $implocal_retenciones
                                );
                                $nodo_implocal->addChild($nodo_implocal_retenciones);
                            }else {
                                $implocal_traslados = [];
                                $implocal_traslados['ImpLocTrasladado'] = $result->tax->name;
                                $implocal_traslados['TasadeTraslado'] = Helper::numberFormat(abs($rate), 2, false);
                                $implocal_traslados['Importe'] = Helper::numberFormat(abs($tmp), 2, false);
                                $nodo_implocal_traslados = new \CfdiUtils\Nodes\Node(
                                    'implocal:TrasladosLocales', // nombre del elemento raíz
                                    $implocal_traslados
                                );
                                $nodo_implocal->addChild($nodo_implocal_traslados);
                            }
                        }
                    }
                }
                //Agregamos al complemento
                $comprobante->addComplemento($nodo_implocal);
            }

            //Complemento leyendas fiscales
            if($customer_invoice->customerActiveInvoiceTaxLegends->isNotEmpty()){
                $leyendas_fisc = [];
                $leyendas_fisc['version'] = '1.0';
                $nodo_leyendas_fisc = new \CfdiUtils\Nodes\Node(
                    'leyendasFisc:LeyendasFiscales', // nombre del elemento raíz
                    $leyendas_fisc
                );
                foreach ($customer_invoice->customerActiveInvoiceTaxLegends as $key => $result) {
                    $leyenda = [];
                    $leyenda['disposicionFiscal'] = $result->disposicion_fiscal;
                    $leyenda['norma'] = $result->norma;
                    $leyenda['textoLeyenda'] = $result->texto_leyenda;
                    $nodo_leyenda = new \CfdiUtils\Nodes\Node(
                        'leyendasFisc:Leyenda', // nombre del elemento raíz
                        $leyenda
                    );
                    $nodo_leyendas_fisc->addChild($nodo_leyenda);
                }
                //Agregamos al complemento
                $comprobante->addComplemento($nodo_leyendas_fisc);
            }

            //Complemento INE
            if(!empty($customer_invoice_complement->ine_process_type)){
                $ine = [];
                $ine['Version'] = '1.1';
                $ine['TipoProceso'] = $customer_invoice_complement->ine_process_type;
                $ine['TipoComite'] = $customer_invoice_complement->ine_committee_type;
                $ine['IdContabilidad'] = $customer_invoice_complement->ine_id_accounting;
                $nodo_ine = new \CfdiUtils\Nodes\Node(
                    'ine:INE', // nombre del elemento raíz
                    $ine
                );
                foreach ($customer_invoice->customerActiveInvoiceInes as $key => $result) {
                    $entidad = [];
                    $entidad['ClaveEntidad'] = $result->ine_entity;
                    $entidad['Ambito'] = $result->ambit;
                    $nodo_entidad = new \CfdiUtils\Nodes\Node(
                        'ine:Entidad', // nombre del elemento raíz
                        $entidad
                    );
                    $id_contabilidades = explode(',', $result->id_accounting);
                    if(!empty($id_contabilidades)){
                        foreach($id_contabilidades as $key => $result2){
                            $contabilidad = [];
                            $contabilidad['IdContabilidad'] = trim($result2);
                            $nodo_contabilidad = new \CfdiUtils\Nodes\Node(
                                'ine:Contabilidad', // nombre del elemento raíz
                                $contabilidad
                            );
                            $nodo_entidad->addChild($nodo_contabilidad);
                        }
                    }
                    $nodo_ine->addChild($nodo_entidad);
                }
                //Agregamos al complemento
                $comprobante->addComplemento($nodo_ine);
            }


            //Método de ayuda para establecer las sumas del comprobante e impuestos con base en la suma de los conceptos y la agrupación de sus impuestos
            //$creator->addSumasConceptos(null, 2);
            //Método de ayuda para generar el sello (obtener la cadena de origen y firmar con la llave privada)
            $creator->addSello('file://' . \Storage::path($customer_invoice->company->pathFileKeyPassPem()), Crypt::decryptString($customer_invoice->company->password_key));
            //Valida la estructura
            //$creator->validate();

            //Guarda XML
            //if(\Auth::user()->email == 'sksistemas@hotmail.com'){
                //dd($creator->asXml());
			//}
            $path_xml = Helper::setDirectory(CustomerInvoice::PATH_XML_FILES_CI, $customer_invoice->company_id) . '/';
            $file_xml = Helper::makeDirectoryCfdi($path_xml) . '/' . Str::random(40) . '.xml';
            $creator->saveXml(\Storage::path($path_xml . $file_xml));


            //Arreglo temporal para actualizar Customer Invoice CFDI
            $tmp = [
                'pac_id' => $pac->id,
                'cfdi_version' => setting('cfdi_version'),
                'uuid' => '',
                'date' => '',
                'path_xml' => $path_xml,
                'file_xml' => $file_xml,
                'file_xml_pac' => '',
                'pac' => $pac,
            ];

            //Timbrado de XML
            $class_pac = $pac->code;
            $tmp = PacHelper::$class_pac($tmp, $creator);

            return $tmp;
        } catch (\Exception $e) {
            throw $e;
        }
        @libxml_disable_entity_loader($current_entity_loader);
    }

    /**
     * Descarga de archivo XML
     *
     * @param Request $request
     * @param CustomerInvoice $customer_invoice
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector|\Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function downloadXml(Request $request, CustomerInvoice $customer_invoice)
    {
        //Ruta y validacion del XML
        $path_xml = Helper::setDirectory(CustomerInvoice::PATH_XML_FILES_CI, $customer_invoice->company_id) . '/';
        $file_xml_pac = $path_xml . $customer_invoice->customerInvoiceCfdi->file_xml_pac;

        if(!empty($customer_invoice->addenda)){
            //Parsear XML
            $xml_tfd = new \DOMDocument('1.0', 'UTF-8');
            $xml_tfd->loadXML(\Storage::get($file_xml_pac));

            //Addenda a XML
            $xml_addenda = new \DOMDocument('1.0', 'UTF-8');
            libxml_use_internal_errors(true);
            $xml_addenda->loadXML(str_replace(array("\r", "\n", "\t"), '', $customer_invoice->addenda));

            //Crear nodo Addenda en XML
            $node_addenda = $xml_tfd->createElement('cfdi:Addenda');
            $node_addenda = $xml_tfd->documentElement->appendChild($node_addenda);

            // Importar nodo de addenda en el XML
            $addenda = $xml_tfd->importNode($xml_addenda->documentElement, true);
            $node_addenda->appendChild($addenda);

            //Guarda el archivo nuevo
            $file_xml_pac = 'temp/' . Str::random(40) . '.xml';
            \Storage::put($file_xml_pac, $xml_tfd->saveXML());
        }

        if (!empty($customer_invoice->customerInvoiceCfdi->file_xml_pac) && \Storage::exists($file_xml_pac)) {
            while (ob_get_level()) ob_end_clean();
            ob_start();

            return response()->download(\Storage::path($file_xml_pac), str_replace('/','',$customer_invoice->name) . '.xml',['Cache-Control' => 'no-cache, must-revalidate']);
        }

        //Mensaje
        flash(__('sales/customer_invoice.error_download_xml'))->error();

        //Redireccion
        return redirect('/sales/customer-invoices');
    }

    /**
     * Descarga de archivo PDF
     *
     * @param Request $request
     * @param CustomerInvoice $customer_invoice
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector|\Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function downloadPdf(Request $request, CustomerInvoice $customer_invoice)
    {
        //Descarga archivo
        return $this->print($customer_invoice, true);
    }

    /**
     * Cambiar estatus a abierta
     *
     * @param CustomerInvoice $customer_invoice
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function markOpen(CustomerInvoice $customer_invoice)
    {
        //Logica
        if ((int)$customer_invoice->status == CustomerInvoice::TO_PAY || (int)$customer_invoice->status == CustomerInvoice::PAID) {
            $customer_invoice->updated_uid = \Auth::user()->id;
            $customer_invoice->status = CustomerInvoice::OPEN;
            $customer_invoice->save();

            //Cancelacion del timbre fiscal

            //Mensaje
            flash(__('general.text_form_success_edit'))->success();
        } else {
            //Mensaje
            flash(__('sales/customer_invoice.error_change_status'))->error();
        }

        //Redireccion
        return redirect('/sales/customer-invoices');
    }

    /**
     * Cambiar estatus a abierta
     *
     * @param CustomerInvoice $customer_invoice
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function markToPay(CustomerInvoice $customer_invoice)
    {
        //Logica
        if ((int)$customer_invoice->status == CustomerInvoice::OPEN || (int)$customer_invoice->status == CustomerInvoice::PAID) {
            $customer_invoice->updated_uid = \Auth::user()->id;
            $customer_invoice->status = CustomerInvoice::TO_PAY;
            $customer_invoice->save();

            //Cancelacion del timbre fiscal

            //Mensaje
            flash(__('general.text_form_success_edit'))->success();
        } else {
            //Mensaje
            flash(__('sales/customer_invoice.error_change_status'))->error();
        }

        //Redireccion
        return redirect('/sales/customer-invoices');
    }

    /**
     * Cambiar estatus a enviada
     *
     * @param CustomerInvoice $customer_invoice
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function markSent(CustomerInvoice $customer_invoice)
    {
        //Logica
        $customer_invoice->updated_uid = \Auth::user()->id;
        $customer_invoice->mail_sent = 1;
        $customer_invoice->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/sales/customer-invoices');
    }

    /**
     * Cambiar estatus a pagada
     *
     * @param CustomerInvoice $customer_invoice
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function markPaid(CustomerInvoice $customer_invoice)
    {

        //Logica
        if ((int)$customer_invoice->status == CustomerInvoice::OPEN || (int)$customer_invoice->status == CustomerInvoice::TO_PAY) {
            $customer_invoice->updated_uid = \Auth::user()->id;
            $customer_invoice->status = CustomerInvoice::PAID;
            $customer_invoice->save();

            //Mensaje
            flash(__('general.text_form_success_edit'))->success();
        } else {
            //Mensaje
            flash(__('sales/customer_invoice.error_change_status'))->error();
        }

        //Redireccion
        return redirect('/sales/customer-invoices');
    }

    /**
     * Clase generica de impresion
     *
     * @param CustomerInvoice $customer_invoice
     * @return mixed
     */
    public function print(CustomerInvoice $customer_invoice, $download = false, $save = false)
    {
        try {
            //Logica
            $tmp = 'default';
            if (!empty($customer_invoice->customerInvoiceCfdi->cfdi_version)) {
                $tmp = $customer_invoice->customerInvoiceCfdi->cfdi_version;
            }
            $class_print = 'print' . ucfirst($tmp);
            return $this->$class_print($customer_invoice,$download,$save);

        } catch (\Exception $e) {
            flash($e->getMessage())->error();
            return redirect('/sales/customer-invoices');
        }
    }

    /**
     * Impresion default
     *
     * @param $customer_invoice
     * @return mixed
     */
    private function printDefault($customer_invoice, $download = false, $save = false)
    {
        //PDF
        $pdf = \PDF::loadView('sales.customer_invoices.pdf_default', compact('customer_invoice'));

        //Guardar
        if ($save) {
            return $pdf->output();
        }

        //Descarga
        if($download){
            return $pdf->download($customer_invoice->documentType->name . '_' . str_replace('/','',$customer_invoice->name) . '.pdf');
        }

        //Redireccion
        return $pdf->stream($customer_invoice->documentType->name . '_' . str_replace('/','',$customer_invoice->name) . '.pdf');
    }

    /**
     * Impresion CFDI 3.3
     *
     * @param $customer_invoice
     * @param bool $download
     * @return mixed
     * @throws \Exception
     */
    private function printCfdi33($customer_invoice, $download = false, $save = false)
    {
        //Datos del XML timbrado
        $path_xml = Helper::setDirectory(CustomerInvoice::PATH_XML_FILES_CI, $customer_invoice->company_id) . '/';
        $file_xml_pac = $path_xml . $customer_invoice->customerInvoiceCfdi->file_xml_pac;
        $data = [];
        $customer_invoice_complement = $customer_invoice->customerInvoiceComplement;

        //Valida que el archivo exista
        if (!empty($customer_invoice->customerInvoiceCfdi->file_xml_pac) && \Storage::exists($file_xml_pac)) {
            //Crea arreglo
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($file_xml_pac));
            $data = Cfdi33Helper::getQuickArrayCfdi($cfdi);

            //Genera codigo QR
            $image = QrCode::format('png')->size(150)->margin(0)->generate($data['qr_cadena']);
            $data['qr'] = 'data:image/png;base64,' . base64_encode($image);

            //Regimen fiscal
            $data['tax_regimen'] = TaxRegimen::where('code', $data['cfdi33']->emisor['RegimenFiscal'])->first()->name_sat;
        }

        //PDF
        $pdf_template = \App\Helpers\Helper::companyPdfTemplate($customer_invoice->company_id);//Plantilla PDF
        if($customer_invoice->cfdi_type2 == 'transfer'){ //Impresion de carta porte factura
            $list_document_type2 = [
                'transfer' => __('sales/customer_transfer.text_document_type2_transfer'),
                'invoice' => __('sales/customer_transfer.text_document_type2_invoice'),
            ];
            $customer_transfer = $customer_invoice;
            if(!empty($customer_transfer->customerInvoiceTransfer->transport_int)){
                $customer_transfer_transfer = $customer_transfer->customerInvoiceTransfer;
                $pdf = \PDF::loadView('sales.customer_transfers.pdf_cfdi33_v2_default', compact('customer_transfer', 'data','list_document_type2', 'customer_transfer_transfer'));
            }else{
                $pdf = \PDF::loadView('sales.customer_transfers.pdf_cfdi33_' . $pdf_template, compact('customer_transfer', 'data','list_document_type2'));
            }
        }elseif($customer_invoice->cfdi_type2 == 'foreign_trade'){ //Impresion de carta porte factura
            $customer_foreign_trade = $customer_invoice;
            $pdf = \PDF::loadView('sales.customer_foreign_trades.pdf_cfdi33_' . $pdf_template,
                compact('customer_foreign_trade','data'));
        }else {
            $pdf = \PDF::loadView('sales.customer_invoices.pdf_cfdi33_' . $pdf_template,
                compact('customer_invoice', 'data', 'customer_invoice_complement'));
        }

        //Guardar
        if ($save) {
            return $pdf->output();
        }

        //Descarga
        if($download){
            return $pdf->download(str_replace('/','',$customer_invoice->name) . '.pdf');
        }

        //Redireccion
        return $pdf->stream(str_replace('/','',$customer_invoice->name) . '.pdf');
    }

    /**
     * Impresion CFDI 4.0
     *
     * @param $customer_invoice
     * @param bool $download
     * @return mixed
     * @throws \Exception
     */
    private function printCfdi40($customer_invoice, $download = false, $save = false)
    {
        //Datos del XML timbrado
        $path_xml = Helper::setDirectory(CustomerInvoice::PATH_XML_FILES_CI, $customer_invoice->company_id) . '/';
        $file_xml_pac = $path_xml . $customer_invoice->customerInvoiceCfdi->file_xml_pac;
        $data = [];
        $customer_invoice_complement = $customer_invoice->customerInvoiceComplement;
        $global_periodicities = __('general.text_global_periodicities');

        //Valida que el archivo exista
        if (!empty($customer_invoice->customerInvoiceCfdi->file_xml_pac) && \Storage::exists($file_xml_pac)) {
            //Crea arreglo
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($file_xml_pac));
            $data = Cfdi33Helper::getQuickArrayCfdi($cfdi);

            //Genera codigo QR
            $image = QrCode::format('png')->size(150)->margin(0)->generate($data['qr_cadena']);
            $data['qr'] = 'data:image/png;base64,' . base64_encode($image);

            //Regimen fiscal
            $data['tax_regimen'] = TaxRegimen::where('code', $data['cfdi33']->emisor['RegimenFiscal'])->first()->name_sat;
            $data['tax_regimen_customer'] = TaxRegimen::where('code', $data['cfdi33']->receptor['RegimenFiscalReceptor'])->first()->name_sat;
        }

        //PDF
        $pdf_template = \App\Helpers\Helper::companyPdfTemplate($customer_invoice->company_id);//Plantilla PDF
        if($customer_invoice->cfdi_type2 == 'transfer'){ //Impresion de carta porte factura
            $list_document_type2 = [
                'transfer' => __('sales/customer_transfer.text_document_type2_transfer'),
                'invoice' => __('sales/customer_transfer.text_document_type2_invoice'),
            ];
            $customer_transfer = $customer_invoice;
            if(!empty($customer_transfer->customerInvoiceTransfer->transport_int)){
                $customer_transfer_transfer = $customer_transfer->customerInvoiceTransfer;
                $pdf = \PDF::loadView('sales.customer_transfers.pdf_cfdi40_v2_default', compact('customer_transfer', 'data','list_document_type2', 'customer_transfer_transfer'));
            }else{
                $pdf = \PDF::loadView('sales.customer_transfers.pdf_cfdi40_' . $pdf_template, compact('customer_transfer', 'data','list_document_type2'));
            }
        }elseif($customer_invoice->cfdi_type2 == 'foreign_trade'){ //Impresion de carta porte factura
            $customer_foreign_trade = $customer_invoice;
            $pdf = \PDF::loadView('sales.customer_foreign_trades.pdf_cfdi40_' . $pdf_template,
                compact('customer_foreign_trade','data'));
        }else {
            $pdf = \PDF::loadView('sales.customer_invoices.pdf_cfdi40_' . $pdf_template,
                compact('customer_invoice', 'data', 'customer_invoice_complement','global_periodicities'));
        }

        //Guardar
        if ($save) {
            return $pdf->output();
        }

        //Descarga
        if($download){
            return $pdf->download(str_replace('/','',$customer_invoice->name) . '.pdf');
        }

        //Redireccion
        return $pdf->stream(str_replace('/','',$customer_invoice->name) . '.pdf');
    }

    /**
     * Modal para envio de correo de factura
     *
     * @param Request $request
     * @param CustomerInvoice $customer_invoice
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function modalSendMail(Request $request, CustomerInvoice $customer_invoice)
    {
        //Variables
        $id = $request->id;

        //Logica
        if ($request->ajax() && !empty($id)) {
            //Correo default del cliente
            $reply = Helper::replyMails();
            $to = [];
            $to_selected = [];
            if (!empty($customer_invoice->customer->email)) {
                $tmp = explode(';',$customer_invoice->customer->email);
                if(!empty($tmp)) {
                    foreach($tmp as $email) {
                        $email = trim($email);
                        $to[$email] = $email;
                        $to_selected [] = $email;
                    }
                }
            }
            //Etiquetas solo son demostrativas
            if($customer_invoice->status != CustomerInvoice::DRAFT) {
                $files = [
                    'pdf' => str_replace('/','',$customer_invoice->name) . '.pdf',
                    'xml' => str_replace('/','',$customer_invoice->name) . '.xml'
                ];
            }else{
                $files = [
                    'pdf' => str_replace('/','',$customer_invoice->name) . '.pdf',
                ];
            }
            $files_selected = array_keys($files);

            //modal de mensaje
            $html = view('layouts.partials.customer_invoices.modal_send_mail',
                compact('customer_invoice', 'to', 'to_selected', 'files', 'files_selected', 'reply'))->render();

            //Mensaje predefinido
            $custom_message = view('layouts.partials.customer_invoices.message_send_mail',
                compact('customer_invoice'))->render();

            return response()->json(['html' => $html, 'custom_message' => $custom_message]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Envio de factura por correo
     *
     * @param Request $request
     * @param CustomerInvoice $customer_invoice
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function sendMail(Request $request, CustomerInvoice $customer_invoice)
    {
        //Validaciones
        $validator = \Validator::make($request->all(), [
            'subject' => 'required',
            'to' => 'required',
            'to.*' => 'email',
            'message' => 'required'
        ], [
            'subject.*' => __('general.error_mail_subject'),
            'to.required' => __('general.error_mail_to'),
            'to.*.email' => __('general.error_mail_to_format'),
            'message.*' => __('general.error_mail_message'),
        ]);
        if ($validator->fails()) {
            $errors = '<ul>';
            foreach ($validator->errors()->all() as $message) {
                $errors .= '<li>'.$message . '</li>';
            }
            $errors .= '</ul>';
            return response()->json(['error' => $errors], 422);
        }

        //Creamos PDF en stream
        $pdf = $this->print($customer_invoice);
        //Ruta del XML
        $path_xml = Helper::setDirectory(CustomerInvoice::PATH_XML_FILES_CI, $customer_invoice->company_id) . '/';
        $file_xml_pac = $path_xml . $customer_invoice->customerInvoiceCfdi->file_xml_pac;

        if(!empty($customer_invoice->addenda)){
            //Parsear XML
            $xml_tfd = new \DOMDocument('1.0', 'UTF-8');
            $xml_tfd->loadXML(\Storage::get($file_xml_pac));

            //Addenda a XML
            $xml_addenda = new \DOMDocument('1.0', 'UTF-8');
            libxml_use_internal_errors(true);
            $xml_addenda->loadXML(str_replace(array("\r", "\n", "\t"), '', $customer_invoice->addenda));

            //Crear nodo Addenda en XML
            $node_addenda = $xml_tfd->createElement('cfdi:Addenda');
            $node_addenda = $xml_tfd->documentElement->appendChild($node_addenda);

            // Importar nodo de addenda en el XML
            $addenda = $xml_tfd->importNode($xml_addenda->documentElement, true);
            $node_addenda->appendChild($addenda);

            //Guarda el archivo nuevo
            $file_xml_pac = 'temp/' . Str::random(40) . '.xml';
            \Storage::put($file_xml_pac, $xml_tfd->saveXML());
        }

        //Envio de correo
        $to = $request->to;
        $reply = $request->reply;
        \Mail::to($to)->send(new SendCustomerInvoice($customer_invoice, $request->subject, $request->message, $pdf,
            $file_xml_pac, $reply));

        //Actualiza campo de enviado
        $customer_invoice->updated_uid = \Auth::user()->id;
        $customer_invoice->mail_sent = 1;
        $customer_invoice->save();

        //Mensaje
        return response()->json([
            'success' => sprintf(__('sales/customer_invoice.text_success_send_mail'), $customer_invoice->name)
        ]);
    }

    private function autoSendMail($customer_invoice){
        $to = [];
        if (!empty($customer_invoice->customer->email)) {
            $tmp = explode(';',$customer_invoice->customer->email);
            if(!empty($tmp)) {
                foreach($tmp as $email) {
                    $email = trim($email);
                    $to[$email] = $email;
                }
            }
        }
        if(!empty($to)) {
            $subject = $customer_invoice->documentType->name .' ('.$customer_invoice->name.') - ' . $customer_invoice->company->name;
            $message = view('layouts.partials.customer_invoices.message_send_mail',compact('customer_invoice'))->render();
            $pdf = $this->print($customer_invoice);
            $path_xml = Helper::setDirectory(CustomerInvoice::PATH_XML_FILES_CI, $customer_invoice->company_id) . '/';
            $file_xml_pac = $path_xml . $customer_invoice->customerInvoiceCfdi->file_xml_pac;

            if(!empty($customer_invoice->addenda)){
                //Parsear XML
                $xml_tfd = new \DOMDocument('1.0', 'UTF-8');
                $xml_tfd->loadXML(\Storage::get($file_xml_pac));

                //Addenda a XML
                $xml_addenda = new \DOMDocument('1.0', 'UTF-8');
                libxml_use_internal_errors(true);
                $xml_addenda->loadXML(str_replace(array("\r", "\n", "\t"), '', $customer_invoice->addenda));

                //Crear nodo Addenda en XML
                $node_addenda = $xml_tfd->createElement('cfdi:Addenda');
                $node_addenda = $xml_tfd->documentElement->appendChild($node_addenda);

                // Importar nodo de addenda en el XML
                $addenda = $xml_tfd->importNode($xml_addenda->documentElement, true);
                $node_addenda->appendChild($addenda);

                //Guarda el archivo nuevo
                $file_xml_pac = 'temp/' . Str::random(40) . '.xml';
                \Storage::put($file_xml_pac, $xml_tfd->saveXML());
            }

            \Mail::to($to)->send(new SendCustomerInvoice($customer_invoice, $subject, $message, $pdf, $file_xml_pac, Helper::replyMails()));
            //Actualiza campo de enviado
            $customer_invoice->updated_uid = \Auth::user()->id;
            $customer_invoice->mail_sent = 1;
            $customer_invoice->save();
        }
    }

    /**
     * Obtener registro
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCustomerInvoice(Request $request)
    {
        //Variables
        $id = $request->id;

        //Logica
        if ($request->ajax() && !empty($id)) {
            $customer_invoice = CustomerInvoice::findOrFail($id);
            $customer_invoice->uuid = $customer_invoice->customerInvoiceCfdi->uuid ?? '';
            return response()->json($customer_invoice, 200);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Autoacompletar select2 de facturas solo activas del SAT
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function autocompleteCfdi(Request $request)
    {
        //Variables
        $term = $request->term;
        $customer_id = !empty($request->customer_id) ? $request->customer_id : '111111111111'; //Filtra las facturas por cliente

        //Logica
        if ($request->ajax() && !empty($term)) {
            $tmp = CustomerInvoice::filter([
                'filter_search_cfdi_select2' => $term,
                'filter_customer_id' => $customer_id,
                'filter_document_type_code' => $this->document_type_code,
            ])->sortable('date')->limit(16)->get();
            $results = [];
            if ($tmp->isNotEmpty()) {
                foreach ($tmp as $result) {
                    $results[] = [
                        'id' => $result->id,
                        'text' => $result->text_select2,
                        'description' => $result->description_select2
                    ];
                }
            }
            return response()->json($results, 200);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Facturas con saldo
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function balances(Request $request)
    {
        //Variables
        $customer_id = $request->customer_id;
        $filter_currency_id = $request->filter_currency_id;
        //Logica
        if ($request->ajax() && !empty($customer_id)) {

            $currency = Currency::findOrFail($request->currency_id);
            $currency_value = $request->currency_value;
            $filter =[
                'filter_balances' => true,
                'filter_customer_id' => $customer_id,
                'filter_currency_id' => ((!empty($currency) && $currency->code != 'MXN') || !empty($filter_currency_id)) ? $currency->id : '', //Solo pagos en MXN se pueden aplicar a otras monedas
                'filter_document_type_code' => [$this->document_type_code,'customer.lease','customer.fee'],
            ];

            //Consulta
            $results = CustomerInvoice::filter($filter)->sortable('date')->get();

            $html = view('layouts.partials.customer_invoices.balances', compact('results','currency','filter_currency_id'))->render();

            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Calcula el total de las lineas
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function totalLines(Request $request)
    {
        //Variables
        $json = new \stdClass;
        $input_items = $request->item;
        $currency_id = $request->currency_id;
        $currency_code = 'MXN';
        $taxes = [];

        if ($request->ajax()) {
            //Datos de moneda
            if (!empty($currency_id)) {
                $currency = Currency::findOrFail($currency_id);
                $currency_code = $currency->code;
            }
            //Variables de totales
            $amount_discount = 0;
            $amount_untaxed = 0;
            $amount_tax = 0;
            $amount_tax_ret = 0;
            $amount_total = 0;
            $balance = 0;
            $items = [];
            if (!empty($input_items)) {
                foreach ($input_items as $key => $item) {
                    //Logica
                    $item_quantity = (double)$item['quantity'];
                    $item_price_unit = (double)$item['price_unit'];
                    $item_quota_ieps = (double)$item['quota_ieps'];
                    //Ajuste para cuando tiene IVA incluido
                    if (!empty($item['taxes']) && !empty($item['includes_iva'])) {
                        foreach ($item['taxes'] as $tax_id) {
                            if (!empty($tax_id)) {
                                $tax = Tax::findOrFail($tax_id);
                                if($tax->code == '002' && $tax->rate>0) {
                                    if ($tax->factor == 'Tasa') {
                                        $item_price_unit = $item_price_unit / (1+($tax->rate/100));
                                    } elseif ($tax->factor == 'Cuota') {
                                        $item_price_unit -= $tax->rate;
                                    }
                                    $item_price_unit = round($item_price_unit, \App\Helpers\Helper::companyProductPriceDecimalPlace());
                                    $input_items[$key]['price_unit'] = $item_price_unit;
                                }
                            }
                            break;
                        }
                    }
                    $item_discount = round((double)$item['discount'], 2);
                    $item_price_reduce = $item['discount_type'] == 'A' ? $item_price_unit : ($item_price_unit * (100 - $item_discount) / 100);
                    $item_amount_untaxed = round($item_quantity * $item_price_reduce, 2);
                    if($item_quota_ieps > 0){
                        $item_amount_untaxed_per_taxes = round($item_quantity * (($item_price_unit * (100 - $item_discount) / 100) - $item_quota_ieps), 2);
                    }else{
                        $item_amount_untaxed_per_taxes = $item_amount_untaxed;
                    }
                    if($item['discount_type'] == 'A'){
                        $item_amount_untaxed -= $item_discount;
                        $item_amount_untaxed_per_taxes -= $item_discount;
                    }
                    $item_amount_discount = $item['discount_type'] == 'A' ? $item_discount : round($item_quantity * $item_price_unit, 2) - $item_amount_untaxed;
                    $item_amount_tax = 0;
                    $item_amount_tax_ret = 0;
                    if (!empty($item['taxes'])) {
                        foreach ($item['taxes'] as $tax_id) {
                            if (!empty($tax_id)) {
                                $tax = Tax::findOrFail($tax_id);
                                $tmp = 0;
                                if ($tax->factor == 'Tasa') {
                                    $tmp = ($item_amount_untaxed_per_taxes + (!empty($item['fix_amount_untaxed']) && !empty($tax->code) ? 0.01 : 0 )) * $tax->rate / 100;
                                } elseif ($tax->factor == 'Cuota') {
                                    $tmp = $tax->rate;
                                }
                                $tmp = round($tmp, 2);
                                if ($tax->type == 'R') { //Retenciones
                                    $item_amount_tax_ret += $tmp;
                                } else { //Traslados
                                    $item_amount_tax += $tmp;
                                }

                                //Sumatoria de impuestos
                                $taxes[$tax_id] = array(
                                    'amount_base' => $item_amount_untaxed_per_taxes + (isset($taxes[$tax_id]['amount_base']) ? $taxes[$tax_id]['amount_base'] : 0),
                                    'amount_tax' => $tmp + (isset($taxes[$tax_id]['amount_tax']) ? $taxes[$tax_id]['amount_tax'] : 0),
                                );
                            }
                        }
                    }
                    $item_amount_total = $item_amount_untaxed + $item_amount_tax + $item_amount_tax_ret;
                    //Sumatoria totales
                    $amount_discount += $item_amount_discount;
                    $amount_untaxed += $item_amount_untaxed;
                    $amount_tax += $item_amount_tax;
                    $amount_tax_ret += $item_amount_tax_ret;
                    $amount_total += $item_amount_total;
                    //Subtotales por cada item
                    $items[$key] = money($item_amount_untaxed, $currency_code, true)->format();
                }
            }
            //Respuesta
            $taxes_tmp = [];
            if (!empty($taxes)) {
                foreach ($taxes as $tax_id => $result) {
                    $tax = Tax::findOrFail($tax_id);
                    $taxes_tmp[] = [
                        'name' => $tax->name,
                        'tax_id' => $tax_id,
                        'amount_base' => money($result['amount_base'], $currency_code, true)->format(),
                        'amount_tax' => money($result['amount_tax'], $currency_code, true)->format(),
                    ];
                }
            }

            //
            $json->items = $items;
            $json->taxes = $taxes_tmp;
            $json->amount_discount = money($amount_discount, $currency_code, true)->format();
            $json->amount_untaxed = money($amount_untaxed, $currency_code, true)->format();
            $json->amount_tax = money($amount_tax + $amount_tax_ret, $currency_code, true)->format();
            $json->amount_total = money($amount_total, $currency_code, true)->format();
            $json->amount_total_tmp = $amount_total;
            return response()->json($json);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Modal para cancelar factura
     *
     * @param Request $request
     * @param CustomerInvoice $customer_invoice
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function modalCancel(Request $request, CustomerInvoice $customer_invoice)
    {
        //Variables
        $id = $request->id;

        //Logica
        if ($request->ajax() && !empty($id)) {
            //Obtener informacion de estatus
            $data_status_sat = [
                'cancelable' => 1,
                'pac_is_cancelable' => ''
            ];
            $reason_cancellations = ReasonCancellation::populateSelect()->get()->pluck('name_sat', 'id');
            if (!empty($customer_invoice->customerInvoiceCfdi->cfdi_version) && !empty($customer_invoice->customerInvoiceCfdi->uuid)) {

                //Obtener el sellos del CFDI
                $path_xml = Helper::setDirectory(CustomerInvoice::PATH_XML_FILES_CI, $customer_invoice->company_id) . '/';
                $file_xml_pac = $path_xml . $customer_invoice->customerInvoiceCfdi->file_xml_pac;
                $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($file_xml_pac));
                $data = Cfdi33Helper::getQuickArrayCfdi($cfdi);

                $tmp = [
                    'rfcR' => $data['cfdi33']->Receptor['Rfc'] ?? $customer_invoice->customer->taxid,
                    'uuid' => $customer_invoice->customerInvoiceCfdi->uuid,
                    'total' => Helper::numberFormat($customer_invoice->amount_total,
                        $customer_invoice->currency->decimal_place, false),
                    'fe' => substr($data['cfdi33']->complemento->timbreFiscalDigital['SelloCFD'],-8), //Los últimos 8 caracteres del Atributo Sello del CFDI
                    'file_xml_pac' => $file_xml_pac
                ];
                $class_pac = $customer_invoice->customerInvoiceCfdi->pac->code . 'Status';
                $data_status_sat = PacHelper::$class_pac($tmp,$customer_invoice->company,$customer_invoice->customerInvoiceCfdi->pac);
            }
            $is_cancelable = true;
            if($data_status_sat['cancelable'] == 3){
                $is_cancelable = false;
            }

            //modal de cancelar
            $html = view('layouts.partials.customer_invoices.modal_cancel', compact('customer_invoice','data_status_sat','is_cancelable', 'reason_cancellations'))->render();


            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Exportar datos a excel
     *
     * @param Request $request
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function exportToExcel(Request $request)
    {
        while (ob_get_level()) ob_end_clean();
        ob_start();

        return Excel::download(new CustomerInvoicesExport($request),
            __('sales/customer_invoice.document_title') . '-' . config('app.name') . '.xlsx');
    }

    /**
     * Modal para historial de pagos y notas de credito aplicadas
     *
     * @param Request $request
     * @param CustomerInvoice $customer_invoice
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function modalPaymentHistory(Request $request, CustomerInvoice $customer_invoice)
    {
        //Variables
        $id = $request->id;

        //Logica
        if ($request->ajax() && !empty($id)) {

            //modal de historial de pagos
            $html = view('layouts.partials.customer_invoices.modal_payment_history',
                compact('customer_invoice'))->render();


            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Funcion para timbrar factura con estatus de prefactura
     *
     * @param CustomerInvoice $customer_invoice
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function stamp(CustomerInvoice $customer_invoice){

        \DB::connection('tenant')->beginTransaction();
        try {
            $invoiced = false;
            if((int) $customer_invoice->status == CustomerInvoice::DRAFT) {
                //Valida que tenga exista la clase de facturacion
                $class_cfdi = setting('cfdi_version');
                if (empty($class_cfdi)) {
                    throw new \Exception(__('general.error_cfdi_version'));
                }
                if (!method_exists($this, $class_cfdi)) {
                    throw new \Exception(__('general.error_cfdi_class_exists'));
                }

                //Valida que tenga folios disponibles
                if(BaseHelper::getAvailableFolios()<=0){
                    throw new \Exception(__('general.error_available_folios'));
                }

                //CFDI
                $customer_invoice_cfdi = $customer_invoice->CustomerInvoiceCfdi;

                //Actualiza registr principal
                $customer_invoice->updated_uid = \Auth::user()->id;
                $customer_invoice->date = Helper::dateTimeToSql(BaseHelper::getDateTimeBranchOffice($customer_invoice->branch_office_id));
                $customer_invoice->status = CustomerInvoice::OPEN;
                $customer_invoice->save();

                //Actualiza folios
                //Obtiene folio
                $document_type = Helper::getNextDocumentTypeByCode($this->document_type_code, $customer_invoice->company->id,false, $customer_invoice->branch_office_id);
                $customer_invoice->draft = $customer_invoice->name;
                $customer_invoice->name = $document_type['name'];
                $customer_invoice->serie = $document_type['serie'];
                $customer_invoice->folio = $document_type['folio'];
                $customer_invoice->save();
                $customer_invoice_cfdi->name = $customer_invoice->name;
                $customer_invoice_cfdi->save();

                //Valida Empresa y PAC para timbrado
                PacHelper::validateSatActions($customer_invoice->company);

                //Crear XML y timbra
                $tmp = $this->$class_cfdi($customer_invoice);

                //Guardar registros de CFDI
                $customer_invoice_cfdi->fill(array_only($tmp, [
                    'pac_id',
                    'cfdi_version',
                    'uuid',
                    'date',
                    'file_xml',
                    'file_xml_pac',
                ]));
                $customer_invoice_cfdi->save();

                //Disminuye folios
                BaseHelper::decrementFolios();

                //Actualiza cotizacion en caso de estar relacionada
                if(!empty($customer_invoice->customer_quotation_id)) {
                    $customer_quotation = CustomerQuotation::findOrFail($customer_invoice->customer_quotation_id);
                    $customer_quotation->status = CustomerQuotation::BILLED;
                    $customer_quotation->save();
                }

                if(!empty($customer_invoice->customer_remission_id)) {
                    $customer_remission = CustomerRemission::findOrFail($customer_invoice->customer_remission_id);
                    $customer_remission->status = CustomerRemission::BILLED;
                    $customer_remission->save();
                }

                //Mensaje
                flash(__('general.text_success_customer_invoice_cfdi'))->success();

                $invoiced = true;
            }

            \DB::connection('tenant')->commit();

            if($invoiced) {
                $this->saveCfdiDownloads($customer_invoice, $customer_invoice_cfdi);
            }

        } catch (\Exception $e) {
            //Fix fechas

            \DB::connection('tenant')->rollback();
            flash( __('general.error_cfdi_pac').'<br>'.$e->getMessage())->error();
            return back()->withInput();
        }

        //Almacenamiento dropbox
        self::dropboxBackup($customer_invoice);

        //Redireccion
        return redirect('/sales/customer-invoices');
    }

    /**
     * Funcion para autorizar la cancelacion
     *
     * @param CustomerInvoice $customer_invoice
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function cancelAuthorized(CustomerInvoice $customer_invoice){

        \DB::connection('tenant')->beginTransaction();
        try {

            if((int) $customer_invoice->status == CustomerInvoice::CANCEL_PER_AUTHORIZED) {

                //Actualiza registro principal
                $customer_invoice->updated_uid = \Auth::user()->id;
                $customer_invoice->status = CustomerInvoice::CANCEL;
                $customer_invoice->save();

                //Mensaje
                flash(__('general.text_form_success_cancel'))->success();
            }

            \DB::connection('tenant')->commit();

            //Redireccion
            return redirect('/sales/customer-invoices');
        } catch (\Exception $e) {
            \DB::connection('tenant')->rollback();
            flash($e->getMessage())->error();
            return back()->withInput();
        }
    }

    /**
     * Funcion para cancelar la cancelacion
     *
     * @param CustomerInvoice $customer_invoice
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function cancelRejected(CustomerInvoice $customer_invoice){

        \DB::connection('tenant')->beginTransaction();
        try {

            if((int) $customer_invoice->status == CustomerInvoice::CANCEL_PER_AUTHORIZED) {

                //Actualiza registro principal
                $customer_invoice->updated_uid = \Auth::user()->id;
                $customer_invoice->status = CustomerInvoice::OPEN;
                $customer_invoice->save();

                //Actualiza el cfdi
                $customer_invoice->CustomerInvoiceCfdi->cancel_date = null;
                $customer_invoice->CustomerInvoiceCfdi->cancel_response = 'Cancelación rechazada - ' . $customer_invoice->CustomerInvoiceCfdi->cancel_response;
                //$customer_invoice->CustomerInvoiceCfdi->cancel_state = null;
                $customer_invoice->CustomerInvoiceCfdi->status = 1;
                $customer_invoice->CustomerInvoiceCfdi->save();

                $cfdi_download = CfdiDownload::where('uuid', $customer_invoice->customerInvoiceCfdi->uuid)->where('type', 2)->first();
                if(!empty($cfdi_download)){
                    $cfdi_download->status = 'Vigente';
                    $cfdi_download->save();
                }

                //Mensaje
                flash(__('general.text_success_customer_invoice_rejected'))->success();
            }

            \DB::connection('tenant')->commit();

            //Redireccion
            return redirect('/sales/customer-invoices');
        } catch (\Exception $e) {
            \DB::connection('tenant')->rollback();
            flash($e->getMessage())->error();
            return back()->withInput();
        }
    }

    /**
     * Modal para estatus de CFDI
     *
     * @param Request $request
     * @param CustomerInvoice $customer_invoice
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function modalStatusSat(Request $request, CustomerInvoice $customer_invoice)
    {
        //Variables
        $id = $request->id;

        //Logica
        if ($request->ajax() && !empty($id)) {
            //Obtener informacion de estatus
            $data_status_sat = [
                'cancelable' => 1,
                'pac_is_cancelable' => ''
            ];
            if (!empty($customer_invoice->customerInvoiceCfdi->cfdi_version) && !empty($customer_invoice->customerInvoiceCfdi->uuid)) {

                //Obtener el sellos del CFDI
                $path_xml = Helper::setDirectory(CustomerInvoice::PATH_XML_FILES_CI, $customer_invoice->company_id) . '/';
                $file_xml_pac = $path_xml . $customer_invoice->customerInvoiceCfdi->file_xml_pac;
                $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($file_xml_pac));
                $data = Cfdi33Helper::getQuickArrayCfdi($cfdi);

                $tmp = [
                    'rfcR' => $data['cfdi33']->Receptor['Rfc'] ?? $customer_invoice->customer->taxid,
                    'uuid' => $customer_invoice->customerInvoiceCfdi->uuid,
                    'total' => Helper::numberFormat($customer_invoice->amount_total, $customer_invoice->currency->decimal_place, false),
                    'fe' => substr($data['cfdi33']->complemento->timbreFiscalDigital['SelloCFD'],-8), //Los últimos 8 caracteres del Atributo Sello del CFDI
                    'file_xml_pac' => $file_xml_pac
                ];
                $class_pac = $customer_invoice->customerInvoiceCfdi->pac->code . 'Status';
                $data_status_sat = PacHelper::$class_pac($tmp,$customer_invoice->company,$customer_invoice->customerInvoiceCfdi->pac);
            }
            $is_cancelable = true;
            if($data_status_sat['cancelable'] == 3){
                $is_cancelable = false;
            }

            //modal de visualizar estatus en el SAT
            $html = view('layouts.partials.customer_invoices.modal_status_sat', compact('customer_invoice','data_status_sat','is_cancelable'))->render();

            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function createImport()
    {
        $salespersons = Salesperson::populateSelect()->pluck('name', 'id');
        $branch_offices = BranchOffice::populateSelect()->pluck('name', 'id');
        $currencies = Currency::populateSelect()->get()->pluck('name_sat', 'id');
        $payment_terms = PaymentTerm::populateSelect()->pluck('name', 'id');
        $payment_ways = PaymentWay::populateSelect()->get()->pluck('name_sat', 'id');
        $payment_methods = PaymentMethod::populateSelect()->get()->pluck('name_sat', 'id');
        $cfdi_uses = CfdiUse::populateSelect()->get()->pluck('name_sat', 'id');
        $cfdi_relations = CfdiRelation::populateSelect()->get()->pluck('name_sat', 'id');
        $taxes = Tax::populateSelect()->pluck('name', 'id');

        $import_results [] = (object)[
            'customer_taxid' => 'AMS840526THA',
            'customer' => 'Alberto Perez Perez',
            'branch_office' => 'Matriz',
            'currency' => 'MXN',
            'payment_term' => 'Contado',
            'date_due' => '22-05-2019',
            'salesperson' => 'Angel Hernandez Cruz',
            'payment_way' => 'Transferencia electrónica de fondos',
            'payment_method' => 'PUE',
            'cfdi_use' => 'Gastos en general',
            'cfdi_relation' => 'Sustitución de los CFDI previos',
            'uuids' => 'uuid-1, uuid-2',
            'quantity' => '1',
            'product' => 'PRO001',
            'name' => 'Servicio general',
            'unit_measure' => 'H87-Pieza',
            'sat_product' => '01010101', //80101500
            'price_unit' => '525',
            'includes_iva' => 'S',
            'taxes' => 'IVA 16%',
            'numero_pedimento' => '',
            'send_mail' => 'N',
        ];
        $import_results [] = (object)[
            'customer_taxid' => 'WTS780329TY2',
            'customer' => 'Aceros del centro SA DE CV',
            'branch_office' => 'Matriz',
            'currency' => 'MXN',
            'payment_term' => 'Contado',
            'date_due' => '',
            'salesperson' => '',
            'payment_way' => 'Efectivo',
            'payment_method' => 'PUE',
            'cfdi_use' => 'Gastos en general',
            'cfdi_relation' => '',
            'uuids' => '',
            'quantity' => '1',
            'product' => 'PRO001',
            'name' => 'Servicio general',
            'unit_measure' => 'H87-Pieza',
            'sat_product' => '01010101', //80101500
            'price_unit' => '525',
            'includes_iva' => 'S',
            'taxes' => 'IVA 16%',
            'numero_pedimento' => '',
            'send_mail' => 'N',
        ];
        $import_results [] = (object)[
            'customer_taxid' => '',
            'customer' => '',
            'branch_office' => '',
            'currency' => '',
            'payment_term' => '',
            'date_due' => '',
            'salesperson' => '',
            'payment_way' => '',
            'payment_method' => '',
            'cfdi_use' => '',
            'cfdi_relation' => '',
            'uuids' => '',
            'quantity' => '1',
            'product' => 'PRO001',
            'name' => 'Servicio de mantenimiento',
            'unit_measure' => 'H87-Pieza',
            'sat_product' => '01010101',
            'price_unit' => '525',
            'includes_iva' => 'S',
            'taxes' => 'IVA 16%',
            'numero_pedimento' => '22  47  1864  2008747',
            'send_mail' => '',
        ];
        $import_results [] = (object)[
            'customer_taxid' => 'XAXX010101000',
            'customer' => 'Martha Montiel Mendez',
            'branch_office' => 'Matriz',
            'currency' => 'MXN',
            'payment_term' => '15 días de crédito',
            'date_due' => '',
            'salesperson' => '',
            'payment_way' => 'Efectivo',
            'payment_method' => 'PPD',
            'cfdi_use' => 'Gastos en general',
            'cfdi_relation' => '',
            'uuids' => '',
            'quantity' => '1',
            'product' => 'PRO001',
            'name' => 'Servicio general',
            'unit_measure' => 'H48-Unidad de servicio',
            'sat_product' => '80101500',
            'price_unit' => '525',
            'includes_iva' => 'S',
            'taxes' => 'IVA 16%',
            'numero_pedimento' => '',
            'send_mail' => 'N',
        ];

        return view('sales.customer_invoices.import',
            compact('salespersons', 'branch_offices', 'currencies', 'payment_ways', 'payment_terms', 'payment_methods',
                'cfdi_uses', 'taxes', 'cfdi_relations','import_results'));
    }

    /**
     * Descargar plantilla
     *
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function downloadTemplateImport(Request $request)
    {
        //Datos
        $customers = Customer::filter($request->all())->active()->orderBy('sort_order')->orderBy('name')->get();
        //Descargar archivo
        while (ob_get_level()) ob_end_clean();
        ob_start();

        return Excel::download(new CustomerInvoiceTemplateImportExport($request, $customers), __('sales/customer_invoice.text_template_import') . '-' . config('app.name') . '.xlsx',\Maatwebsite\Excel\Excel::XLSX);
    }

    /**
     * Obtener informacion antes de procesar el archivo
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function beforeImport(Request $request)
    {
        //Validaciones
        //Validacion de archivos por extension
        if ($request->hasFile('file_customer_invoices_import')) {
            $request->merge(['file_customer_invoices_import_ext' => request()->file('file_customer_invoices_import')->getClientOriginalExtension()]);
        }
        $validator = \Validator::make($request->all(), [
            'file_customer_invoices_import' => 'required|max:2048',
            'file_customer_invoices_import_ext' => 'nullable|in:xls,xlsx'
        ], [
            'file_customer_invoices_import.*' => __('sales/customer_invoice.error_file_customer_invoices_import'),
            'file_customer_invoices_import_ext.*' => __('sales/customer_invoice.error_file_customer_invoices_import'),
        ]);
        //Validaciones manuales
        //$validator = \Validator::make([], []);

        //Errores
        if ($validator->fails()) {
            $errors = '';
            foreach ($validator->errors()->all() as $message) {
                $errors = $message;
            }
            return response()->json(['error' => $errors], 422);
        }

        try {
            //Lógica

            //Importar
            $import = new CustomerInvoiceTemplateBeforeImportImport;
            Excel::import($import, request()->file('file_customer_invoices_import'));

            //Mensaje
            return response()->json([
                'success' => 'ok',
                'total_import_customer_invoices' => sprintf(__('sales/customer_invoice.help_import_sweet_alert_1'),Helper::numberFormat($import->total_customer_invoices),Helper::numberFormatMoney($import->total_amount_customer_invoices)),
            ]);
        }catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 422);
        }
    }

    /**
     * Importar cuotas desde archivo
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     * @throws \Illuminate\Validation\ValidationException
     */
    public function storeImport(Request $request)
    {
        //Validacion
        //Validacion de archivos por extension
        if ($request->hasFile('file_customer_invoices_import')) {
            $request->merge(['file_customer_invoices_import_ext' => request()->file('file_customer_invoices_import')->getClientOriginalExtension()]);
        }
        $this->validate($request, [
            'file_customer_invoices_import' => 'required|max:2048',
            'file_customer_invoices_import_ext' => 'nullable|in:xls,xlsx'
        ], [
            'file_customer_invoices_import.*' => __('sales/customer_invoice.error_file_customer_invoices_import'),
            'file_customer_invoices_import_ext.*' => __('sales/customer_invoice.error_file_customer_invoices_import'),
        ]);

        //filas
        $rows = null;
        $data_customer_invoices = [];
        $customer_invoices = [];
        $customer_invoices_send_mail = [];

        //Validaciones y arreglo de valores
        try {
            //Lógica

            //Importar
            $import = new CustomerInvoiceTemplateImportImport;
            Excel::import($import, request()->file('file_customer_invoices_import'));

            $rows = $import->rows;

            //Validacion
            foreach ($rows as $key => $row) {
                $num_row = $key + 2;

                if(!empty($row['fecha_de_vencimiento'])){
                    if (strpos($row['fecha_de_vencimiento'], "'") || strpos($row['fecha_de_vencimiento'], "-") || strpos($row['fecha_de_vencimiento'], "/")) {
                        $rows[$key]['fecha_de_vencimiento'] = $row['fecha_de_vencimiento'] = str_replace('\'', '', str_replace('/', '-', $row['fecha_de_vencimiento']));
                    } else {
                        $rows[$key]['fecha_de_vencimiento'] = \Date::instance(\PhpOffice\PhpSpreadsheet\Shared\Date::excelToDateTimeObject($row['fecha_de_vencimiento']))->format(setting('date_format', 'd-m-Y'));
                    }
                }

                \Validator::make($row->toArray(), [
                    'cantidad' => 'required|numeric|min:0.00001',
                    'descripcion' => 'required',
                    'u_m' => 'required',
                    'prodserv_sat' => 'required',
                    'precio' => 'required|numeric|min:0.00001',
                ], [
                    'cantidad.*' => __('sales/customer_invoice.error_line_quantity') . sprintf(__('sales/customer_invoice.error_row'), $num_row,$row['cantidad']),
                    'descripcion.*' => __('sales/customer_invoice.error_line_name') . sprintf(__('sales/customer_invoice.error_row'), $num_row,$row['descripcion']),
                    'u_m.*' => __('sales/customer_invoice.error_line_unit_measure_id') . sprintf(__('sales/customer_invoice.error_row'), $num_row,$row['u_m']),
                    'prodserv_sat.*' => __('sales/customer_invoice.error_line_sat_product_id') . sprintf(__('sales/customer_invoice.error_row'), $num_row,$row['prodserv_sat']),
                    'precio.*' => __('sales/customer_invoice.error_line_price_unit') . sprintf(__('sales/customer_invoice.error_row'), $num_row,$row['precio']),
                ])->validate();
            }

            //Agrupamos los datos correctamente
            $count=0;
            foreach ($rows as $key => $row) {
                $num_row = $key + 2;

                //Valida producto
                $product = null;
                if (!empty($row['producto'])) {
                    if(!empty(setting('products_per_taxid'))){
                        $product = Product::where('code', '=', $row['producto'])->where('company_id', '=', Helper::defaultCompany()->id)->first();
                    }else{
                        $product = Product::where('code', '=', $row['producto'])->where('company_id', '=', Helper::firstCompany()->id)->first();
                    }
                    if (empty($product)) {
                        throw new \Exception(__('sales/customer_invoice.error_line_product_id2') . sprintf(__('sales/customer_invoice.error_row'), $num_row,$row['producto']));
                    }
                }
                //Unidades de medida
                $unit_measure = null;
                if (!empty($row['u_m'])) {
                    $tpm = explode('-',$row['u_m']);
                    $unit_measure = UnitMeasure::where('code', '=', $tpm[0])->first();
                    if (empty($unit_measure)) {
                        throw new \Exception(__('sales/customer_invoice.error_line_unit_measure_id2') . sprintf(__('sales/customer_invoice.error_row'), $num_row,$tpm[0]));
                    }
                }
                //Catalogo sat
                $sat_product = null;
                if (!empty($row['prodserv_sat'])) {
                    $sat_product = SatProduct::where('code', '=', $row['prodserv_sat'])->first();
                    if (empty($sat_product)) {
                        throw new \Exception(__('sales/customer_invoice.error_line_sat_product_id2') . sprintf(__('sales/customer_invoice.error_row'), $num_row,$row['prodserv_sat']));
                    }
                }
                //Valida impuestos
                $tax = null;
                $taxes = [];
                $taxesArr = [];
                if (!empty($row['impuestos'])) {
                    $taxesArr = explode(', ', $row['impuestos']);
                    for($i = 0; $i < count($taxesArr); $i++){
                        $tax = Tax::where('name', '=', $taxesArr[$i])->first();
                        if(!empty($tax)){
                            array_push($taxes, $tax->id);
                        }else{
                            throw new \Exception(__('sales/customer_invoice.error_line_taxes2') . sprintf(__('sales/customer_invoice.error_row'), $num_row,$row['impuestos']));
                        }
                    }
                }
                //Encabezado
                if (!empty($row['rfc_cliente']) && !empty($row['cliente'])) {
                    //Cliente
                    $customer = null;
                    if (!empty($row['rfc_cliente'])) {
                        if(in_array($row['rfc_cliente'], ['XAXX010101000','XEXX010101000'])){
                            if(!empty(setting('customers_per_taxid'))){
                                $customer = Customer::where('taxid', '=', $row['rfc_cliente'])->where('name', '=', $row['cliente'])->where('company_id', '=', Helper::defaultCompany()->id)->first();
                            }else{
                                $customer = Customer::where('taxid', '=', $row['rfc_cliente'])->where('name', '=', $row['cliente'])->where('company_id', '=', Helper::firstCompany()->id)->first();
                            }
                        }else{
                            if(!empty(setting('customers_per_taxid'))){
                                $customer = Customer::where('taxid', '=', $row['rfc_cliente'])->where('company_id', '=', Helper::defaultCompany()->id)->first();
                            }else{
                                $customer = Customer::where('taxid', '=', $row['rfc_cliente'])->where('company_id', '=', Helper::firstCompany()->id)->first();
                            }
                        }
                        if (empty($customer)) {
                            throw new \Exception(__('sales/customer_invoice.error_customer_id2') . sprintf(__('sales/customer_invoice.error_row'), $num_row, $row['rfc_cliente']));
                        }else{
                            if(setting('cfdi_version') == 'cfdi40'){
                                if (empty($customer->tax_regimen_id)) {
                                    throw new \Exception(__('sales/customer_invoice.error_tax_regimen_customer_id2') . sprintf(__('sales/customer_invoice.error_row'), $num_row, $row['cliente']));
                                }
                                if (empty($customer->postcode)) {
                                    throw new \Exception(__('sales/customer_invoice.error_postcode_customer') . sprintf(__('sales/customer_invoice.error_row'), $num_row, $row['cliente']));
                                }
                            }
                        }
                    }
                    //Sucursales
                    $branch_office = null;
                    if (!empty($row['sucursal'])) {
                        $branch_office = BranchOffice::where('name', '=', $row['sucursal'])->where('company_id', '=', Helper::defaultCompany()->id)->first();
                        if (empty($branch_office)) {
                            throw new \Exception(__('sales/customer_invoice.error_branch_office_id2') . sprintf(__('sales/customer_invoice.error_row'), $num_row,$row['sucursal']));
                        }
                    }else{
                        throw new \Exception(__('sales/customer_invoice.error_branch_office_id') . sprintf(__('sales/customer_invoice.error_row'), $num_row,$row['sucursal']));
                    }
                    //Moneda
                    $currency = null;
                    if (!empty($row['moneda'])) {
                        $currency = Currency::where('code', '=', $row['moneda'])->first();
                        if (empty($currency)) {
                            throw new \Exception(__('sales/customer_invoice.error_currency_id2') . sprintf(__('sales/customer_invoice.error_row'), $num_row,$row['moneda']));
                        }
                    }else{
                        throw new \Exception(__('sales/customer_invoice.error_currency_id') . sprintf(__('sales/customer_invoice.error_row'), $num_row,$row['moneda']));
                    }
                    //Termino de pago
                    $payment_term = null;
                    if (!empty($row['terminos_de_pago'])) {
                        $payment_term = PaymentTerm::where('name', '=', $row['terminos_de_pago'])->first();
                        if (empty($payment_term)) {
                            throw new \Exception(__('sales/customer_invoice.error_payment_term_id2') . sprintf(__('sales/customer_invoice.error_row'), $num_row,$row['terminos_de_pago']));
                        }
                    }else{
                        throw new \Exception(__('sales/customer_invoice.error_payment_term_id') . sprintf(__('sales/customer_invoice.error_row'), $num_row,$row['terminos_de_pago']));
                    }
                    //Vendedor
                    $salesperson = null;
                    if (!empty($row['vendedor'])) {
                        $salesperson = Salesperson::where('name', '=', $row['vendedor'])->first();
                        if (empty($salesperson)) {
                            throw new \Exception(__('sales/customer_invoice.error_salesperson_id2') . sprintf(__('sales/customer_invoice.error_row'), $num_row,$row['vendedor']));
                        }
                    }
                    //Termino de pago
                    $payment_way = null;
                    if (!empty($row['forma_de_pago'])) {
                        $payment_way = PaymentWay::where('name', '=', $row['forma_de_pago'])->first();
                        if (empty($payment_way)) {
                            throw new \Exception(__('sales/customer_invoice.error_payment_way_id2') . sprintf(__('sales/customer_invoice.error_row'), $num_row,$row['forma_de_pago']));
                        }
                    }else{
                        throw new \Exception(__('sales/customer_invoice.error_payment_way_id') . sprintf(__('sales/customer_invoice.error_row'), $num_row,$row['forma_de_pago']));
                    }
                    //Metodo de pago
                    $payment_method = null;
                    if (!empty($row['metodo_de_pago'])) {
                        $payment_method = PaymentMethod::where('code', '=', $row['metodo_de_pago'])->first();
                        if (empty($payment_method)) {
                            throw new \Exception(__('sales/customer_invoice.error_payment_method_id2') . sprintf(__('sales/customer_invoice.error_row'), $num_row,$row['metodo_de_pago']));
                        }
                    }else{
                        throw new \Exception(__('sales/customer_invoice.error_payment_method_id') . sprintf(__('sales/customer_invoice.error_row'), $num_row,$row['metodo_de_pago']));
                    }
                    //Uso de CFDI
                    $cfdi_use = null;
                    if (!empty($row['uso_de_cfdi'])) {
                        $cfdi_use = CfdiUse::where('name', '=', $row['uso_de_cfdi'])->first();
                        if (empty($cfdi_use)) {
                            throw new \Exception(__('sales/customer_invoice.error_cfdi_use_id2') . sprintf(__('sales/customer_invoice.error_row'), $num_row,$row['uso_de_cfdi']));
                        }
                    }else{
                        throw new \Exception(__('sales/customer_invoice.error_cfdi_use_id') . sprintf(__('sales/customer_invoice.error_row'), $num_row,$row['uso_de_cfdi']));
                    }
                    //Tipo de relacion
                    $cfdi_relation = null;
                    if (!empty($row['tipo_de_relacion'])) {
                        $cfdi_relation = CfdiRelation::where('name', '=', $row['tipo_de_relacion'])->first();
                        if (empty($cfdi_relation)) {
                            throw new \Exception(__('sales/customer_invoice.error_cfdi_relation_id2') . sprintf(__('sales/customer_invoice.error_row'), $num_row,$row['tipo_de_relacion']));
                        }
                        if(empty($row['uuids'])){
                            throw new \Exception(__('sales/customer_invoice.error_cfdi_relation_id_uuids') . sprintf(__('sales/customer_invoice.error_row'), $num_row,$row['uuids']));
                        }
                    }

                    $data_customer_invoices[$count] = [
                        'customer_id' => $customer->id,
                        'taxid' => $customer->taxid,
                        'customer' => $customer->name,
                        'branch_office_id' => $branch_office->id,
                        'currency_id' => $currency->id,
                        'payment_term_id' => $payment_term->id,
                        'date_due' => $row['fecha_de_vencimiento'],
                        'salesperson_id' => !empty($salesperson) ? $salesperson->id : null,
                        'payment_way_id' => $payment_way->id,
                        'payment_method_id' => $payment_method->id,
                        'cfdi_use_id' => $cfdi_use->id,
                        'cfdi_relation_id' => !empty($cfdi_relation) ? $cfdi_relation->id : null,
                        'uuids' => !empty($row['uuids']) ? explode(',', $row['uuids']) : [],
                        'send_mail' => $row['enviar_correo'],
                        'tax_regimen_customer_id' => $customer->tax_regimen_id,
                    ];
                    if(!empty($row['descripcion']) && !empty($row['precio'])) {
                        $data_customer_invoices[$count]['item'][] = [
                            'quantity' => $row['cantidad'],
                            'product_id' => !empty($product) ? $product->id : null,
                            'name' => $row['descripcion'],
                            'unit_measure_id' => $unit_measure->id,
                            'sat_product_id' => $sat_product->id,
                            'price_unit' => $row['precio'],
                            'includes_iva' => $row['iva_incluido'] == 'S' ? 1 : 0,
                            'taxes' => $taxes,
                            'discount' => 0,
                            'discount_type' => 'P',
                            'quota_ieps' => !empty($product) ? $product->quota_ieps : 0,
                            'numero_pedimento' => $row['numero_pedimento'],

                        ];
                    }
                    $count++;
                }elseif(!empty($row['descripcion']) && !empty($row['precio'])){
                    $data_customer_invoices[$count-1]['item'][] = [
                        'quantity' => $row['cantidad'],
                        'product_id' => !empty($product) ? $product->id : null,
                        'name' => $row['descripcion'],
                        'unit_measure_id' => $unit_measure->id,
                        'sat_product_id' => $sat_product->id,
                        'price_unit' => $row['precio'],
                        'includes_iva' => $row['iva_incluido'] == 'S' ? 1 : 0,
                        'taxes' => $taxes,
                        'discount' => 0,
                        'discount_type' => 'P',
                        'quota_ieps' => !empty($product) ? $product->quota_ieps : 0,
                        'numero_pedimento' => $row['numero_pedimento'],
                    ];
                }
            }

        } catch (\Illuminate\Validation\ValidationException $e ) {
            return back()->withErrors($e->errors());
        } catch (\Exception $e) {
            flash($e->getMessage())->error();
            return back()->withInput();
        }

        //Genera facturas
        $rfc_error = '';
        try {


            if(!empty($data_customer_invoices)) {

                //Valida los folios a utilizar
                if(BaseHelper::getAvailableFolios()<$count){
                    throw new \Exception(__('general.error_available_folios'));
                }

                //Logica
                $company = Helper::defaultCompany(); //Empresa
                foreach($data_customer_invoices as $request) {

                    \DB::connection('tenant')->beginTransaction();
                    $rfc_error = ($request->taxid ?? '') . ' ' . ($request->customer ?? '');

                    $request = (object)$request;

                    //Logica
                    $date = BaseHelper::getDateTimeBranchOffice($request->branch_office_id);
                    $date_due = $date;
                    if (!empty($request->date_due)) {
                        $date_due = Helper::createDate($request->date_due);
                    } else {
                        $payment_term = PaymentTerm::findOrFail($request->payment_term_id);
                        $date_due = $payment_term->days > 0 ? $date->copy()->addDays($payment_term->days) : $date->copy();
                    }

                    //Obtiene folio
                    $document_type = Helper::getNextDocumentTypeByCode($this->document_type_code,$company->id,false, $request->branch_office_id);

                    //Guardar
                    //Registro principal
                    $customer_invoice = CustomerInvoice::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'name' => $document_type['name'],
                        'serie' => $document_type['serie'],
                        'folio' => $document_type['folio'],
                        'date' => Helper::dateTimeToSql($date),
                        'date_due' => Helper::dateToSql($date_due),
                        'customer_id' => $request->customer_id,
                        'branch_office_id' => $request->branch_office_id,
                        'payment_term_id' => $request->payment_term_id,
                        'payment_way_id' => $request->payment_way_id,
                        'payment_method_id' => $request->payment_method_id,
                        'cfdi_use_id' => $request->cfdi_use_id,
                        'salesperson_id' => $request->salesperson_id,
                        'currency_id' => $request->currency_id,
                        'currency_value' => 1,
                        'document_type_id' => $document_type['id'],
                        'status' => CustomerInvoice::OPEN,
                        'company_id' => $company->id,
                        'cfdi_relation_id' => $request->cfdi_relation_id,
                        'tax_regimen_customer_id' => $request->tax_regimen_customer_id,
                    ]);

                    //Registro de lineas
                    $amount_discount = 0;
                    $amount_untaxed = 0;
                    $amount_tax = 0;
                    $amount_tax_ret = 0;
                    $amount_total = 0;
                    $balance = 0;
                    $taxes = array();
                    //Lineas
                    if (!empty($request->item)) {
                        foreach ($request->item as $key => $item) {
                            //Logica
                            $item_quantity = (double)$item['quantity'];
                            $item_price_unit = (double)$item['price_unit'];
                            $item_quota_ieps = (double)$item['quota_ieps'];
                            //Ajuste para cuando tiene IVA incluido
                            if (!empty($item['taxes']) && !empty($item['includes_iva'])) {
                                foreach ($item['taxes'] as $tax_id) {
                                    if (!empty($tax_id)) {
                                        $tax = Tax::findOrFail($tax_id);
                                        if($tax->code == '002' && $tax->rate>0) {
                                            if ($tax->factor == 'Tasa') {
                                                $item_price_unit = $item_price_unit / (1+($tax->rate/100));
                                            } elseif ($tax->factor == 'Cuota') {
                                                $item_price_unit -= $tax->rate;
                                            }
                                            $item_price_unit = round($item_price_unit, \App\Helpers\Helper::companyProductPriceDecimalPlace());
                                            $request->item[$key]['price_unit'] = $item_price_unit;
                                        }
                                    }
                                    break;
                                }
                            }
                            $item_discount = round((double)$item['discount'], 2);
                            $item_price_reduce = $item['discount_type'] == 'A' ? $item_price_unit : ($item_price_unit * (100 - $item_discount) / 100);
                            $item_amount_untaxed = round($item_quantity * $item_price_reduce, 2);
                            if($item_quota_ieps > 0){
                                $item_amount_untaxed_per_taxes = round($item_quantity * (($item_price_unit * (100 - $item_discount) / 100) - $item_quota_ieps), 2);
                            }else{
                                $item_amount_untaxed_per_taxes = $item_amount_untaxed;
                            }
                            if($item['discount_type'] == 'A'){
                                $item_amount_untaxed -= $item_discount;
                                $item_amount_untaxed_per_taxes -= $item_discount;
                            }
                            $item_amount_discount = $item['discount_type'] == 'A' ? $item_discount : round($item_quantity * $item_price_unit, 2) - $item_amount_untaxed;
                            $item_amount_tax = 0;
                            $item_amount_tax_ret = 0;
                            if (!empty($item['taxes'])) {
                                foreach ($item['taxes'] as $tax_id) {
                                    if (!empty($tax_id)) {
                                        $tax = Tax::findOrFail($tax_id);
                                        $tmp = 0;
                                        if ($tax->factor == 'Tasa') {
                                            $tmp = $item_amount_untaxed_per_taxes * $tax->rate / 100;
                                        } elseif ($tax->factor == 'Cuota') {
                                            $tmp = $tax->rate;
                                        }
                                        $tmp = round($tmp, 2);
                                        if ($tax->type == 'R') { //Retenciones
                                            $item_amount_tax_ret += $tmp;
                                        } else { //Traslados
                                            $item_amount_tax += $tmp;
                                        }

                                        //Sumatoria de impuestos
                                        $taxes[$tax_id] = array(
                                            'amount_base' => $item_amount_untaxed_per_taxes + (isset($taxes[$tax_id]['amount_base']) ? $taxes[$tax_id]['amount_base'] : 0),
                                            'amount_tax' => $tmp + (isset($taxes[$tax_id]['amount_tax']) ? $taxes[$tax_id]['amount_tax'] : 0),
                                        );
                                    }
                                }
                            }
                            $item_amount_total = $item_amount_untaxed + $item_amount_tax + $item_amount_tax_ret;
                            //Sumatoria totales
                            $amount_discount += $item_amount_discount;
                            $amount_untaxed += $item_amount_untaxed;
                            $amount_tax += $item_amount_tax;
                            $amount_tax_ret += $item_amount_tax_ret;
                            $amount_total += $item_amount_total;

                            //Guardar linea
                            $customer_invoice_line = CustomerInvoiceLine::create([
                                'created_uid' => \Auth::user()->id,
                                'updated_uid' => \Auth::user()->id,
                                'customer_invoice_id' => $customer_invoice->id,
                                'name' => $item['name'],
                                'product_id' => $item['product_id'],
                                'sat_product_id' => $item['sat_product_id'],
                                'unit_measure_id' => $item['unit_measure_id'],
                                'quantity' => $item_quantity,
                                'price_unit' => $item_price_unit,
                                'discount' => $item_discount,
                                'price_reduce' => $item_price_reduce,
                                'amount_discount' => $item_amount_discount,
                                'amount_untaxed' => $item_amount_untaxed,
                                'amount_tax' => $item_amount_tax,
                                'amount_tax_ret' => $item_amount_tax_ret,
                                'amount_total' => $item_amount_total,
                                'sort_order' => $key,
                                'status' => 1,
                                'quota_ieps' => $item_quota_ieps,
                                'tax_object' => empty($item['taxes']) ? '01' : '02',
                                'discount_type' => $item['discount_type'],
                            ]);

                            //Guardar impuestos por linea
                            if (!empty($item['taxes'])) {
                                $customer_invoice_line->taxes()->sync($item['taxes']);
                            } else {
                                $customer_invoice_line->taxes()->sync([]);
                            }

                            //Numero pedimento
                            $customer_invoice_line_complement = CustomerInvoiceLineComplement::create([
                                'created_uid' => \Auth::user()->id,
                                'updated_uid' => \Auth::user()->id,
                                'customer_invoice_line_id' => $customer_invoice_line->id,
                                'name' => $item['name'],
                                'numero_pedimento' => !empty($item['numero_pedimento']) ? $item['numero_pedimento'] : '',
                                'sort_order' => $key,
                                'status' => 1,
                            ]);
                        }
                    }

                    //Resumen de impuestos
                    if (!empty($taxes)) {
                        $i = 0;
                        foreach ($taxes as $tax_id => $result) {
                            $tax = Tax::findOrFail($tax_id);
                            $customer_invoice_tax = CustomerInvoiceTax::create([
                                'created_uid' => \Auth::user()->id,
                                'updated_uid' => \Auth::user()->id,
                                'customer_invoice_id' => $customer_invoice->id,
                                'name' => $tax->name,
                                'tax_id' => $tax_id,
                                'amount_base' => $result['amount_base'],
                                'amount_tax' => $result['amount_tax'],
                                'sort_order' => $i,
                                'status' => 1,
                            ]);
                            $i++;
                        }
                    }

                    //Cfdi relacionados
                    if (!empty($request->uuids)) {
                        foreach ($request->uuids as $key => $uuid) {
                            //Guardar
                            if(!empty($uuid)){
                                $customer_invoice_relation = CustomerInvoiceRelation::create([
                                    'created_uid' => \Auth::user()->id,
                                    'updated_uid' => \Auth::user()->id,
                                    'customer_invoice_id' => $customer_invoice->id,
                                    'relation_id' => null,
                                    'sort_order' => $key,
                                    'status' => 1,
                                    'uuid_related' => $uuid,
                                ]);
                            }
                        }
                    }

                    //Valida que tenga exista la clase de facturacion
                    $class_cfdi = setting('cfdi_version');
                    if (empty($class_cfdi)) {
                        throw new \Exception(__('general.error_cfdi_version'));
                    }
                    if (!method_exists($this, $class_cfdi)) {
                        throw new \Exception(__('general.error_cfdi_class_exists'));
                    }

                    //Registros de cfdi
                    $customer_invoice_cfdi = CustomerInvoiceCfdi::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'customer_invoice_id' => $customer_invoice->id,
                        'name' => $customer_invoice->name,
                        'cfdi_version' => $class_cfdi,
                        'status' => 1,
                    ]);

                    //Actualiza registro principal con totales
                    $customer_invoice->amount_discount = $amount_discount;
                    $customer_invoice->amount_untaxed = $amount_untaxed;
                    $customer_invoice->amount_tax = $amount_tax;
                    $customer_invoice->amount_tax_ret = $amount_tax_ret;
                    $customer_invoice->amount_total = $amount_total;
                    $customer_invoice->balance = $amount_total;
                    $customer_invoice->update();

                    //Valida Empresa y PAC para timbrado
                    PacHelper::validateSatActions($company);

                    //Crear XML y timbra
                    $tmp = $this->$class_cfdi($customer_invoice);

                    //Guardar registros de CFDI
                    $customer_invoice_cfdi->fill(array_only($tmp, [
                        'pac_id',
                        'cfdi_version',
                        'uuid',
                        'date',
                        'file_xml',
                        'file_xml_pac',
                    ]));
                    $customer_invoice_cfdi->save();

                    //Disminuye folios
                    BaseHelper::decrementFolios();

                    \DB::connection('tenant')->commit();

                    $this->saveCfdiDownloads($customer_invoice, $customer_invoice_cfdi);

                    //Guardamos facturas creadas
                    $customer_invoices[] = $customer_invoice;
                    if($request->send_mail == 'S'){
                        $customer_invoices_send_mail[$customer_invoice->id] = $customer_invoice->id;
                    }

                }
            }


            //Mensaje
            flash(__('sales/customer_invoice.text_form_success_import'))->success();

        } catch (\Illuminate\Validation\ValidationException $e ) {
            \DB::connection('tenant')->rollback();
            if(!empty($rfc_error))
                flash($rfc_error)->error();
            return back()->withErrors($e->errors());
        } catch (\Exception $e) {
            \DB::connection('tenant')->rollback();
            if(!empty($rfc_error))
                flash($rfc_error)->error();
            flash($e->getMessage())->error();
            return back()->withInput();
        }

        //Almacenamiento dropbox
        if(!empty($customer_invoices)){
            foreach($customer_invoices as $customer_invoice){
                self::dropboxBackup($customer_invoice);
            }
        }

        //Envio por correo electronico
        if(!empty($customer_invoices_send_mail)){
            session(['customer_invoices_send_mail' => $customer_invoices_send_mail]);
            return redirect()->route('customer-invoices/send-mails-import');
        }

        //Redireccion
        return redirect()->route('customer-invoices.index');
    }


    /**
     * Respaldo en dropbox si esta activo
     *
     * @param $customer_invoice
     */
    public function dropboxBackup($customer_invoice, $save_xml = true, $save_pdf = true){
        try{
            if(!empty($customer_invoice) && !empty(setting('dropbox_backup')) && !empty(setting('dropbox_access_token'))){
                $path_xml = Helper::setDirectory(CustomerInvoice::PATH_XML_FILES_CI, $customer_invoice->company_id) . '/';
                $file_xml_pac = $path_xml . $customer_invoice->customerInvoiceCfdi->file_xml_pac;
                if (!empty($customer_invoice->customerInvoiceCfdi->file_xml_pac) && \Storage::exists($file_xml_pac)) {

                    $folder = !empty(setting('dropbox_folder','')) ? setting('dropbox_folder','') . '/' : setting('dropbox_folder','');
                    $folder .= strtoupper(\Date::parse($customer_invoice->date)->format('F Y'));
                    $folder .= '/' .preg_replace("/[^a-zA-Z0-9\s]/", '', $customer_invoice->company->name);

                    //Guarda XML
                    if($save_xml) {
                        \Storage::disk('dropbox')->putFileAs($folder,
                            new \Illuminate\Http\File(\Storage::path($file_xml_pac)),
                            $customer_invoice->name . '.xml');
                    }

                    //Guarda PDF
                    if($save_pdf) {
                        $pdf = $this->print($customer_invoice, false, true);
                        \Storage::disk('dropbox')->put($folder . '/' . str_replace('/','',$customer_invoice->name) . '.pdf', $pdf);
                    }
                }


            }
        }catch (\Exception $e){
            flash($e->getMessage())->error();
        }
    }

    public function sendMailsImport(){
        try{
            $customer_invoices_send_mail = session('customer_invoices_send_mail', null);
            if(!empty($customer_invoices_send_mail)){
                foreach($customer_invoices_send_mail as $id){
                    $customer_invoice = CustomerInvoice::find($id);
                    $this->autoSendMail($customer_invoice);
                    unset($customer_invoices_send_mail[$id]);
                    session(['customer_invoices_send_mail' => $customer_invoices_send_mail]);
                }
            }
        }catch (\Exception $e){
            flash($e->getMessage())->error();
            return redirect()->route('customer-invoices.index');
        }

        session()->forget('customer_invoices_send_mail');

        return redirect()->route('customer-invoices.index');
    }

    private function saveCfdiDownloads($customer_invoice, $customer_invoice_cfdi){
        //Directorio de XML
        $path_files = Helper::setDirectory(CfdiDownload::PATH_FILES, $customer_invoice->company->id);
        $tmp_path = str_replace(['files/', '/xml'], '', CustomerInvoice::PATH_XML_FILES_CI);
        if (!\Storage::exists($path_files . '/' . $tmp_path)) {
            \Storage::makeDirectory($path_files . '/' . $tmp_path, 0777, true, true);
        }

        //Genera PDF
        $pdf = $this->print($customer_invoice, false, true);
        $path_xml = Helper::setDirectory(CustomerInvoice::PATH_XML_FILES_CI, $customer_invoice->company_id) . '/';
        $file_xml = $tmp_path . '/' . $customer_invoice_cfdi->uuid . '.xml';
        $file_pdf = str_replace('.xml', '.pdf', $file_xml);
        \Storage::put($path_files . '/' . $file_pdf, $pdf);

        if(!empty($customer_invoice_cfdi->file_xml_pac)){
            \Storage::copy($path_xml . $customer_invoice_cfdi->file_xml_pac, $path_files . '/' . $file_xml);
        }

        //Guarda registro en CFDIS descargados
        $cfdi_download = CfdiDownload::create([
            'created_uid' => \Auth::user()->id,
            'updated_uid' => \Auth::user()->id,
            'type' => 2,
            'uuid' => $customer_invoice_cfdi->uuid,
            'file_xml' => $file_xml,
            'file_pdf' => $file_pdf,
            'file_acuse' => null,
            'status' => 'Vigente',
            'is_cancelable' => 'Cancelable sin aceptación',
            'date_cancel' => null,
            'company_id' => $customer_invoice->company_id,
        ]);
        $data_xml = Helper::parseXmlToArrayCfdi33($path_xml . '/' . $customer_invoice_cfdi->file_xml_pac);
        $data_xml['data']['customer_id'] = $customer_invoice->customer_id;
        $data_xml['data']['caccounting_type_id'] = $customer_invoice->customer->caccounting_type_sale_id ?? null;
        $cfdi_download->fill($data_xml['data']);
        $cfdi_download->save();
    }

    /**
     * Descarga de archivo XML y PDF
     *
     * @param Request $request
     * @param CustomerInvoice $customer_invoice
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector|\Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function downloadXmlPdf(Request $request, CustomerInvoice $customer_invoice)
    {
        $file_zip_name = str_replace('/','',$customer_invoice->name).'.zip';
        if(\Storage::exists('temp/' . $file_zip_name)) {
            @\Storage::delete('temp/' . $file_zip_name);
        }
        $zip =\Zipper::make(\Storage::path('temp/' . $file_zip_name));

        //Ruta y validacion del XML
        $path_xml = Helper::setDirectory(CustomerInvoice::PATH_XML_FILES_CI, $customer_invoice->company_id) . '/';
        $file_xml_pac = $path_xml . $customer_invoice->customerInvoiceCfdi->file_xml_pac;

        if(!empty($customer_invoice->addenda)){
            //Parsear XML
            $xml_tfd = new \DOMDocument('1.0', 'UTF-8');
            $xml_tfd->loadXML(\Storage::get($file_xml_pac));

            //Addenda a XML
            $xml_addenda = new \DOMDocument('1.0', 'UTF-8');
            libxml_use_internal_errors(true);
            $xml_addenda->loadXML(str_replace(array("\r", "\n", "\t"), '', $customer_invoice->addenda));

            //Crear nodo Addenda en XML
            $node_addenda = $xml_tfd->createElement('cfdi:Addenda');
            $node_addenda = $xml_tfd->documentElement->appendChild($node_addenda);

            // Importar nodo de addenda en el XML
            $addenda = $xml_tfd->importNode($xml_addenda->documentElement, true);
            $node_addenda->appendChild($addenda);

            //Guarda el archivo nuevo
            $file_xml_pac = 'temp/' . Str::random(40) . '.xml';
            \Storage::put($file_xml_pac, $xml_tfd->saveXML());
        }

        if (!empty($customer_invoice->customerInvoiceCfdi->file_xml_pac) && \Storage::exists($file_xml_pac)) {
            $zip->add(\Storage::path($file_xml_pac), str_replace('/','',$customer_invoice->name) . '.xml');
        }

        //Ruta y validacion del pdf
        $pdf = $this->print($customer_invoice, false, true);
        $file_pdf_name = Str::random().'.pdf';
        //Guardamos en directorio temporal
        \Storage::put('temp/' . $file_pdf_name, $pdf);
        $zip->add(\Storage::path('temp/' . $file_pdf_name), str_replace('/','',$customer_invoice->name) . '.pdf');

        $zip->close();
        if(\Storage::exists('temp/' . $file_zip_name)) {
            while (ob_get_level()) ob_end_clean();
            ob_start();

            return response()->download(\Storage::path('temp/' . $file_zip_name));
        }

        //Mensaje
        flash(__('sales/customer_invoice.error_download_xmlpdf'))->error();

        //Redireccion
        return redirect('/sales/customer-invoices');
    }

    /**
     * Modal para envio de addenda
     *
     * @param Request $request
     * @param CustomerInvoice $customer_invoice
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function modalAddenda(Request $request, CustomerInvoice $customer_invoice)
    {
        //Variables
        $id = $request->id;

        //Logica
        if ($request->ajax() && !empty($id)) {
            $html = view('layouts.partials.customer_invoices.modal_addenda',compact('customer_invoice'))->render();

            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Addenda
     *
     * @param Request $request
     * @param CustomerInvoice $customer_invoice
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function addenda(Request $request, CustomerInvoice $customer_invoice)
    {
        //Actualiza campo de enviado
        $customer_invoice->updated_uid = \Auth::user()->id;
        $customer_invoice->addenda = $request->addenda;
        $customer_invoice->save();

        //Mensaje
        return response()->json([
            'success' => __('general.text_form_success_add')
        ]);
    }

}
