<?php

namespace App\Http\Controllers\Sales;

use App\Helpers\Helper;
use App\Models\Base\Pac;
use App\Helpers\PacHelper;
use App\Helpers\BaseHelper;
use Illuminate\Support\Str;
use App\Models\Catalogs\Tax;
use Illuminate\Http\Request;
use App\Helpers\Cfdi33Helper;
use App\Models\Sales\Customer;
use App\Models\Catalogs\CfdiUse;
use App\Models\Catalogs\Product;
use App\Models\Catalogs\Project;
use App\Models\Base\BranchOffice;
use App\Models\Catalogs\Currency;
use App\Models\Sales\Salesperson;
use App\Models\Sales\CustomerIedu;
use App\Mail\SendCustomerRemission;
use App\Models\Catalogs\PaymentWay;
use App\Models\Catalogs\SatProduct;
use App\Models\Catalogs\TaxRegimen;
use App\Http\Controllers\Controller;
use App\Models\Catalogs\PaymentTerm;
use App\Models\Catalogs\UnitMeasure;
use Maatwebsite\Excel\Facades\Excel;
use App\Models\Catalogs\CfdiRelation;
use Illuminate\Support\Facades\Crypt;
use App\Models\Catalogs\PaymentMethod;
use App\Models\Sales\CustomerQuotation;
use App\Models\Sales\CustomerRemission;
use App\Exports\CustomerRemissionsExport;
use App\Models\Sales\CustomerRemissionTax;
use App\Models\Sales\CustomerRemissionLine;
use SimpleSoftwareIO\QrCode\Facades\QrCode;

class CustomerRemissionController extends Controller
{
    private $list_status = [];
    private $document_type_code = 'customer.remission';

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->list_status = [
            CustomerRemission::OPEN => __('sales/customer_remission.text_status_open'),
            CustomerRemission::BILLED => __('sales/customer_remission.text_status_billed'),
            CustomerRemission::CANCEL => __('sales/customer_remission.text_status_cancel'),
        ];
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);
        $salespersons = Salesperson::populateSelect()->pluck('name', 'id');
        $branch_offices = BranchOffice::populateSelect()->pluck('name', 'id');
        $list_status = $this->list_status;
        if (empty($request->filter_date_from)) {
            $request->request->add([
                'filter_date_from' => Helper::date(\Date::parse('first day of this month'))
            ]);
        }
        if (empty($request->filter_date_to)) {
            $request->request->add([
                'filter_date_to' => Helper::date(\Date::parse('last day of this month'))
            ]);
        }
        $request->request->add(['filter_document_type_code' => $this->document_type_code]); //Filtra tipo de documento

        //Consulta
        $results = CustomerRemission::filter($request->all())
            ->with('customer')
            ->with('salesperson')
            ->with('currency')
            ->sortable(['date' => 'desc'])->paginate($limit);

        //Vista
        return view('sales.customer_remissions.index',
            compact('results', 'salespersons', 'branch_offices', 'list_status'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        $salespersons = Salesperson::populateSelect()->pluck('name', 'id');
        $branch_offices = BranchOffice::populateSelect()->pluck('name', 'id');
        $currencies = Currency::populateSelect()->get()->pluck('name_sat', 'id');
        $payment_terms = PaymentTerm::populateSelect()->pluck('name', 'id');
        $payment_ways = PaymentWay::populateSelect()->get()->pluck('name_sat', 'id');
        $payment_methods = PaymentMethod::populateSelect()->get()->pluck('name_sat', 'id');
        $cfdi_uses = CfdiUse::populateSelect()->get()->pluck('name_sat', 'id');
        $taxes = Tax::populateSelect()->pluck('name', 'id');
        $projects = Project::populateSelect()->pluck('name', 'id');
        $tax_regimen_customers = TaxRegimen::populateSelect()->get()->pluck('name_sat', 'id');

        $customer_quotation = null;
        if(!empty($request->cq_id)){
            $customer_quotation = CustomerQuotation::findOrFail($request->cq_id);
        }
        $duplicate_ci = null;
        if($request->duplicate_id){
            $duplicate_ci = CustomerRemission::findOrFail($request->duplicate_id);
        }

        return view('sales.customer_remissions.create',
            compact('salespersons', 'branch_offices', 'currencies', 'payment_ways', 'payment_terms', 'payment_methods',
                'cfdi_uses', 'taxes','projects','tax_regimen_customers', 'customer_quotation', 'duplicate_ci'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {

        //Validacion
        $this->validation($request);

        //Almacenamiento de archivos
        $customer_remission = null;

        \DB::connection('tenant')->beginTransaction();
        try {
            //Logica
            $company = Helper::defaultCompany(); //Empresa
            $request->merge(['created_uid' => \Auth::user()->id]);
            $request->merge(['updated_uid' => \Auth::user()->id]);
            $request->merge(['status' => CustomerRemission::OPEN]);
            $request->merge(['company_id' => $company->id]);
            //Ajusta fecha y genera fecha de vencimiento
            $date = Helper::createDateTime($request->date);
            $request->merge(['date' => Helper::dateTimeToSql($date)]);
            $expiration_date_fix = $request->expiration_date;//Fix valida si la fecha de vencimiento esta vacia en caso de error
            $expiration_date = '';
            if (!empty($request->expiration_date)) {
                $expiration_date = Helper::createDate($request->expiration_date);
            }
            $request->merge(['expiration_date' => !empty($expiration_date) ? Helper::dateToSql($expiration_date) : null]);

            //Obtiene folio
            $document_type = Helper::getNextDocumentTypeByCode($this->document_type_code,$company->id, false, $request->branch_office_id);
            $request->merge(['document_type_id' => $document_type['id']]);
            $request->merge(['name' => $document_type['name']]);
            $request->merge(['serie' => $document_type['serie']]);
            $request->merge(['folio' => $document_type['folio']]);

            //Guardar
            //Registro principal
            $customer_remission = CustomerRemission::create($request->input());

            //Registro de lineas
            $amount_discount = 0;
            $amount_untaxed = 0;
            $amount_tax = 0;
            $amount_tax_ret = 0;
            $amount_total = 0;
            $balance = 0;
            $taxes = array();
            //Lineas
            if (!empty($request->item)) {
                foreach ($request->item as $key => $item) {
                    //Logica
                    $item_quantity = (double)$item['quantity'];
                    $item_price_unit = (double)$item['price_unit'];
                    //Ajuste para cuando tiene IVA incluido
                    if (!empty($item['taxes']) && !empty($item['includes_iva'])) {
                        foreach ($item['taxes'] as $tax_id) {
                            if (!empty($tax_id)) {
                                $tax = Tax::findOrFail($tax_id);
                                if($tax->code == '002' && $tax->rate>0) {
                                    if ($tax->factor == 'Tasa') {
                                        $item_price_unit = $item_price_unit / (1+($tax->rate/100));
                                    } elseif ($tax->factor == 'Cuota') {
                                        $item_price_unit -= $tax->rate;
                                    }
                                    $item_price_unit = round($item_price_unit, \App\Helpers\Helper::companyProductPriceDecimalPlace());
                                    $input_items[$key]['price_unit'] = $item_price_unit;
                                }
                            }
                            break;
                        }
                    }
                    $item_discount = (double)$item['discount'];
                    $item_price_reduce = ($item_price_unit * (100 - $item_discount) / 100);
                    $item_amount_untaxed = round($item_quantity * $item_price_reduce, 2);
                    $item_amount_discount = round($item_quantity * $item_price_unit, 2) - $item_amount_untaxed;
                    $item_amount_tax = 0;
                    $item_amount_tax_ret = 0;
                    if (!empty($item['taxes'])) {
                        foreach ($item['taxes'] as $tax_id) {
                            if (!empty($tax_id)) {
                                $tax = Tax::findOrFail($tax_id);
                                $tmp = 0;
                                if ($tax->factor == 'Tasa') {
                                    $tmp = $item_amount_untaxed * $tax->rate / 100;
                                } elseif ($tax->factor == 'Cuota') {
                                    $tmp = $tax->rate;
                                }
                                $tmp = round($tmp, 2);
                                if ($tax->type == 'R') { //Retenciones
                                    $item_amount_tax_ret += $tmp;
                                } else { //Traslados
                                    $item_amount_tax += $tmp;
                                }

                                //Sumatoria de impuestos
                                $taxes[$tax_id] = array(
                                    'amount_base' => $item_amount_untaxed + (isset($taxes[$tax_id]['amount_base']) ? $taxes[$tax_id]['amount_base'] : 0),
                                    'amount_tax' => $tmp + (isset($taxes[$tax_id]['amount_tax']) ? $taxes[$tax_id]['amount_tax'] : 0),
                                );
                            }
                        }
                    }
                    $item_amount_total = $item_amount_untaxed + $item_amount_tax + $item_amount_tax_ret;
                    //Sumatoria totales
                    $amount_discount += $item_amount_discount;
                    $amount_untaxed += $item_amount_untaxed;
                    $amount_tax += $item_amount_tax;
                    $amount_tax_ret += $item_amount_tax_ret;
                    $amount_total += $item_amount_total;

                    //Guardar linea
                    $customer_remission_line = CustomerRemissionLine::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'customer_remission_id' => $customer_remission->id,
                        'name' => $item['name'],
                        'product_id' => $item['product_id'],
                        'sat_product_id' => $item['sat_product_id'],
                        'unit_measure_id' => $item['unit_measure_id'],
                        'quantity' => $item_quantity,
                        'price_unit' => $item_price_unit,
                        'discount' => $item_discount,
                        'price_reduce' => $item_price_reduce,
                        'amount_discount' => $item_amount_discount,
                        'amount_untaxed' => $item_amount_untaxed,
                        'amount_tax' => $item_amount_tax,
                        'amount_tax_ret' => $item_amount_tax_ret,
                        'amount_total' => $item_amount_total,
                        'sort_order' => $key,
                        'status' => 1,
                        'tax_object' => !empty($item['tax_object']) ? $item['tax_object'] : (empty($item['taxes']) ? '01' : '02'),
                    ]);

                    //Guardar impuestos por linea
                    if (!empty($item['taxes'])) {
                        $customer_remission_line->taxes()->sync($item['taxes']);
                    } else {
                        $customer_remission_line->taxes()->sync([]);
                    }
                }
            }

            //Resumen de impuestos
            if (!empty($taxes)) {
                $i = 0;
                foreach ($taxes as $tax_id => $result) {
                    $tax = Tax::findOrFail($tax_id);
                    $customer_remission_tax = CustomerRemissionTax::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'customer_remission_id' => $customer_remission->id,
                        'name' => $tax->name,
                        'tax_id' => $tax_id,
                        'amount_base' => $result['amount_base'],
                        'amount_tax' => $result['amount_tax'],
                        'sort_order' => $i,
                        'status' => 1,
                    ]);
                    $i++;
                }
            }

            //Actualiza registro principal con totales
            $customer_remission->amount_discount = $amount_discount;
            $customer_remission->amount_untaxed = $amount_untaxed;
            $customer_remission->amount_tax = $amount_tax;
            $customer_remission->amount_tax_ret = $amount_tax_ret;
            $customer_remission->amount_total = $amount_total;
            $customer_remission->balance = $amount_total;
            $customer_remission->update();

            //Mensaje
            if(!isset($request->generate_invoice)) {
                flash(__('general.text_success_customer_remission'))->success();
            }

            \DB::connection('tenant')->commit();

        } catch (\Exception $e) {
            //Fix fechas
            $request->merge([
                'date' => Helper::convertSqlToDateTime($request->date),
            ]);

            if (!empty($expiration_date_fix)) {
                $request->merge([
                    'expiration_date' => Helper::convertSqlToDate($request->expiration_date),
                ]);
            }else{
                $request->merge([
                    'expiration_date' => '',
                ]);
            }

            \DB::connection('tenant')->rollback();
            $company = Helper::defaultCompany(); //Empresa
            \Log::error('(' . $company->taxid . ') ' . $e->getMessage());
            flash($e->getMessage())->error();
            return back()->withInput();
        }

        //Redirecciona para cargar vista de facturacion
        if(isset($request->generate_invoice)){
            return redirect()->route('customer-invoices.create',['cr_id' => $customer_remission->id]);
        }

        //Redireccion
        return redirect('/sales/customer-remissions');

    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Sales\CustomerRemission $customer_remission
     * @return \Illuminate\Http\Response
     */
    public function show(CustomerRemission $customer_remission)
    {
        return view('sales.customer_remissions.show', compact('customer_remission'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Sales\CustomerRemission $customer_remission
     * @return \Illuminate\Http\Response
     */
    public function edit(CustomerRemission $customer_remission)
    {
        $salespersons = Salesperson::populateSelect()->pluck('name', 'id');
        $branch_offices = BranchOffice::populateSelect()->pluck('name', 'id');
        $currencies = Currency::populateSelect()->get()->pluck('name_sat', 'id');
        $payment_terms = PaymentTerm::populateSelect()->pluck('name', 'id');
        $payment_ways = PaymentWay::populateSelect()->get()->pluck('name_sat', 'id');
        $payment_methods = PaymentMethod::populateSelect()->get()->pluck('name_sat', 'id');
        $cfdi_uses = CfdiUse::populateSelect()->get()->pluck('name_sat', 'id');
        $taxes = Tax::populateSelect()->pluck('name', 'id');
        $projects = Project::populateSelect()->pluck('name', 'id');
        $tax_regimen_customers = TaxRegimen::populateSelect()->get()->pluck('name_sat', 'id');

        return view('sales.customer_remissions.edit',
            compact('customer_remission','salespersons', 'branch_offices', 'currencies', 'payment_ways', 'payment_terms', 'payment_methods',
                'cfdi_uses', 'taxes','projects','tax_regimen_customers'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @param  \App\Models\Sales\CustomerRemission $customer_remission
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, CustomerRemission $customer_remission)
    {
        //Validacion
        $this->validation($request);

        \DB::connection('tenant')->beginTransaction();
        try {

            //Logica
            $request->merge(['created_uid' => \Auth::user()->id]);
            $request->merge(['updated_uid' => \Auth::user()->id]);
            $request->merge(['status' => CustomerRemission::OPEN]);
            //Ajusta fecha y genera fecha de vencimiento
            $date = Helper::createDateTime($request->date);
            $request->merge(['date' => Helper::dateTimeToSql($date)]);
            $expiration_date_fix = $request->expiration_date;//Fix valida si la fecha de vencimiento esta vacia en caso de error
            $expiration_date = '';
            if (!empty($request->expiration_date)) {
                $expiration_date = Helper::createDate($request->expiration_date);
            }
            $request->merge(['expiration_date' => !empty($expiration_date) ? Helper::dateToSql($expiration_date) : null]);
            $customer_remission->fill($request->only([
                'updated_uid',
                'date',
                'expiration_date',
                'reference',
                'customer_id',
                'branch_office_id',
                'payment_term_id',
                'payment_way_id',
                'payment_method_id',
                'cfdi_use_id',
                'salesperson_id',
                'currency_id',
                'currency_value',
                'comment',
                'terms_of_sale',
                'status',
                'project_id',
                'tax_regimen_customer_id'
            ]));

            //Guardar
            //Registro principal
            $customer_remission->save();


            //Elimina partidas
            if (!empty($request->delete_item)) {
                foreach ($request->delete_item as $key => $result) {
                    //Actualizar status
                    $customer_remission_line = CustomerRemissionLine::findOrFail($result);
                    $customer_remission_line->updated_uid = \Auth::user()->id;
                    $customer_remission_line->status = 0;
                    $customer_remission_line->save();
                }
            }
            //Registro de lineas
            $amount_discount = 0;
            $amount_untaxed = 0;
            $amount_tax = 0;
            $amount_tax_ret = 0;
            $amount_total = 0;
            $balance = 0;
            $taxes = array();
            //Lineas
            if (!empty($request->item)) {
                foreach ($request->item as $key => $item) {
                    //Logica
                    $item_quantity = (double)$item['quantity'];
                    $item_price_unit = (double)$item['price_unit'];
                    //Ajuste para cuando tiene IVA incluido
                    if (!empty($item['taxes']) && !empty($item['includes_iva'])) {
                        foreach ($item['taxes'] as $tax_id) {
                            if (!empty($tax_id)) {
                                $tax = Tax::findOrFail($tax_id);
                                if($tax->code == '002' && $tax->rate>0) {
                                    if ($tax->factor == 'Tasa') {
                                        $item_price_unit = $item_price_unit / (1+($tax->rate/100));
                                    } elseif ($tax->factor == 'Cuota') {
                                        $item_price_unit -= $tax->rate;
                                    }
                                    $item_price_unit = round($item_price_unit, \App\Helpers\Helper::companyProductPriceDecimalPlace());
                                    $input_items[$key]['price_unit'] = $item_price_unit;
                                }
                            }
                            break;
                        }
                    }
                    $item_discount = (double)$item['discount'];
                    $item_price_reduce = ($item_price_unit * (100 - $item_discount) / 100);
                    $item_amount_untaxed = round($item_quantity * $item_price_reduce, 2);
                    $item_amount_discount = round($item_quantity * $item_price_unit, 2) - $item_amount_untaxed;
                    $item_amount_tax = 0;
                    $item_amount_tax_ret = 0;
                    if (!empty($item['taxes'])) {
                        foreach ($item['taxes'] as $tax_id) {
                            if (!empty($tax_id)) {
                                $tax = Tax::findOrFail($tax_id);
                                $tmp = 0;
                                if ($tax->factor == 'Tasa') {
                                    $tmp = $item_amount_untaxed * $tax->rate / 100;
                                } elseif ($tax->factor == 'Cuota') {
                                    $tmp = $tax->rate;
                                }
                                $tmp = round($tmp, 2);
                                if ($tax->type == 'R') { //Retenciones
                                    $item_amount_tax_ret += $tmp;
                                } else { //Traslados
                                    $item_amount_tax += $tmp;
                                }

                                //Sumatoria de impuestos
                                $taxes[$tax_id] = array(
                                    'amount_base' => $item_amount_untaxed + (isset($taxes[$tax_id]['amount_base']) ? $taxes[$tax_id]['amount_base'] : 0),
                                    'amount_tax' => $tmp + (isset($taxes[$tax_id]['amount_tax']) ? $taxes[$tax_id]['amount_tax'] : 0),
                                );
                            }
                        }
                    }
                    $item_amount_total = $item_amount_untaxed + $item_amount_tax + $item_amount_tax_ret;
                    //Sumatoria totales
                    $amount_discount += $item_amount_discount;
                    $amount_untaxed += $item_amount_untaxed;
                    $amount_tax += $item_amount_tax;
                    $amount_tax_ret += $item_amount_tax_ret;
                    $amount_total += $item_amount_total;

                    //Guardar linea
                    $data = [
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'customer_remission_id' => $customer_remission->id,
                        'name' => $item['name'],
                        'product_id' => $item['product_id'],
                        'sat_product_id' => $item['sat_product_id'],
                        'unit_measure_id' => $item['unit_measure_id'],
                        'quantity' => $item_quantity,
                        'price_unit' => $item_price_unit,
                        'discount' => $item_discount,
                        'price_reduce' => $item_price_reduce,
                        'amount_discount' => $item_amount_discount,
                        'amount_untaxed' => $item_amount_untaxed,
                        'amount_tax' => $item_amount_tax,
                        'amount_tax_ret' => $item_amount_tax_ret,
                        'amount_total' => $item_amount_total,
                        'sort_order' => $key,
                        'status' => 1,
                        'tax_object' => !empty($item['tax_object']) ? $item['tax_object'] : (empty($item['taxes']) ? '01' : '02'),
                    ];
                    if (!empty($item['id'])) {
                        $customer_remission_line = CustomerRemissionLine::findOrFail($item['id']);
                        $customer_remission_line->fill(array_only($data, [
                            'updated_uid',
                            'name',
                            'product_id',
                            'sat_product_id',
                            'unit_measure_id',
                            'quantity',
                            'price_unit',
                            'discount',
                            'price_reduce',
                            'amount_discount',
                            'amount_untaxed',
                            'amount_tax',
                            'amount_tax_ret',
                            'amount_total',
                            'sort_order',
                            'tax_object'
                        ]));
                        $customer_remission_line->save();
                    }else{
                        $customer_remission_line = CustomerRemissionLine::create($data);
                    }

                    //Guardar impuestos por linea
                    if (!empty($item['taxes'])) {
                        $customer_remission_line->taxes()->sync($item['taxes']);
                    } else {
                        $customer_remission_line->taxes()->sync([]);
                    }
                }
            }

            //Resumen de impuestos
            CustomerRemissionTax::where('customer_remission_id','=',$customer_remission->id)->delete(); //Borra todo e inserta nuevamente
            if (!empty($taxes)) {
                $i = 0;
                foreach ($taxes as $tax_id => $result) {
                    $tax = Tax::findOrFail($tax_id);
                    $customer_remission_tax = CustomerRemissionTax::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'customer_remission_id' => $customer_remission->id,
                        'name' => $tax->name,
                        'tax_id' => $tax_id,
                        'amount_base' => $result['amount_base'],
                        'amount_tax' => $result['amount_tax'],
                        'sort_order' => $i,
                        'status' => 1,
                    ]);
                    $i++;
                }
            }

            //Actualiza registro principal con totales
            $customer_remission->amount_discount = $amount_discount;
            $customer_remission->amount_untaxed = $amount_untaxed;
            $customer_remission->amount_tax = $amount_tax;
            $customer_remission->amount_tax_ret = $amount_tax_ret;
            $customer_remission->amount_total = $amount_total;
            $customer_remission->balance = $amount_total;
            $customer_remission->update();

            //Mensaje
            if(!isset($request->generate_invoice)) {
                flash(__('general.text_success_customer_remission'))->success();
            }

            \DB::connection('tenant')->commit();


        } catch (\Exception $e) {
            //Fix fechas
            $request->merge([
                'date' => Helper::convertSqlToDateTime($request->date),
            ]);
            if (!empty($expiration_date_fix)) {
                $request->merge([
                    'expiration_date' => Helper::convertSqlToDate($request->expiration_date),
                ]);
            }else{
                $request->merge([
                    'expiration_date' => '',
                ]);
            }

            \DB::connection('tenant')->rollback();
            \Log::error('(' . $customer_remission->company->taxid . ') ' . $e->getMessage());
            flash($e->getMessage())->error();
            return back()->withInput();
        }

        //Redirecciona para cargar vista de facturacion
        if(isset($request->generate_invoice)){
            return redirect()->route('customer-invoices.create',['cr_id' => $customer_remission->id]);
        }

        //Redireccion
        return redirect('/sales/customer-remissions');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Sales\CustomerRemission $customer_remission
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, CustomerRemission $customer_remission)
    {
        \DB::connection('tenant')->beginTransaction();
        try {
            //Logica
            if ((int)$customer_remission->status != CustomerRemission::CANCEL) {
                //Actualiza status
                $customer_remission->updated_uid = \Auth::user()->id;
                $customer_remission->status = CustomerRemission::CANCEL;
                $customer_remission->save();
            }
            \DB::connection('tenant')->commit();

            //Mensaje
            flash(__('general.text_form_success_cancel'))->success();

        } catch (\Exception $e) {
            \DB::connection('tenant')->rollback();
            \Log::error('(' . $customer_remission->company->taxid . ') ' . $e->getMessage());
            flash($e->getMessage())->error();
            return redirect('/sales/customer-remissions');
        }

        //Redireccion
        return redirect('/sales/customer-remissions');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {

        $this->validate($request, [
            'customer_id' => 'required|integer',
            'branch_office_id' => 'required|integer',
            'date' => 'required|date|date_format:"'.setting('datetime_format', 'd-m-Y H:i:s').'"',
            'expiration_date' => 'nullable|date|date_format:"'.setting('date_format', 'd-m-Y').'"|after_or_equal:' . Helper::date(Helper::createDateTime($request->date)),
            'currency_id' => 'required|integer',
            'currency_value' => 'required|numeric|min:0.1',
            'item' => 'required',
            'item.*.name' => 'required',
            'item.*.quantity' => 'required|numeric|min:0.00001',
            'item.*.price_unit' => 'required|numeric|min:0.00001',
            'item.*.discount' => 'nullable|numeric|min:0',

        ], [
            'customer_id.*' => __('sales/customer_remission.error_customer_id'),
            'branch_office_id.*' => __('sales/customer_remission.error_branch_office_id'),
            'date.required' => __('sales/customer_remission.error_date'),
            'date.date' => __('sales/customer_remission.error_date_format'),
            'date.date_format' => __('sales/customer_remission.error_date_format'),
            'expiration_date.date' => __('sales/customer_remission.error_expiration_date_format'),
            'expiration_date.date_format' => __('sales/customer_remission.error_expiration_date_format'),
            'expiration_date.after_or_equal' => __('sales/customer_remission.error_expiration_date_format'),
            'currency_id.*' => __('sales/customer_remission.error_currency_id'),
            'currency_value.*' => __('sales/customer_remission.error_currency_value'),
            'item.required' => __('sales/customer_remission.error_item'),
            'item.*.name.*' => __('sales/customer_remission.error_line_name'),
            'item.*.quantity.*' => __('sales/customer_remission.error_line_quantity'),
            'item.*.price_unit.*' => __('sales/customer_remission.error_line_price_unit'),
        ]);
    }

    /**
     * Descarga de archivo PDF
     *
     * @param Request $request
     * @param CustomerRemission $customer_remission
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector|\Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function downloadPdf(Request $request, CustomerRemission $customer_remission)
    {
        //Descarga archivo
        return $this->print($customer_remission, true);
    }

    /**
     * Cambiar estatus a abierta
     *
     * @param CustomerRemission $customer_remission
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function markOpen(CustomerRemission $customer_remission)
    {
        //Logica
        if ((int)$customer_remission->status == CustomerRemission::BILLED) {
            $customer_remission->updated_uid = \Auth::user()->id;
            $customer_remission->status = CustomerRemission::OPEN;
            $customer_remission->save();

            //Cancelacion del timbre fiscal

            //Mensaje
            flash(__('general.text_form_success_edit'))->success();
        } else {
            //Mensaje
            flash(__('sales/customer_remission.error_change_status'))->error();
        }

        //Redireccion
        return redirect('/sales/customer-remissions');
    }

    /**
     * Cambiar estatus a abierta
     *
     * @param CustomerRemission $customer_remission
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function markBilled(CustomerRemission $customer_remission)
    {
        //Logica
        if ((int)$customer_remission->status == CustomerRemission::OPEN) {
            $customer_remission->updated_uid = \Auth::user()->id;
            $customer_remission->status = CustomerRemission::BILLED;
            $customer_remission->save();

            //Cancelacion del timbre fiscal

            //Mensaje
            flash(__('general.text_form_success_edit'))->success();
        } else {
            //Mensaje
            flash(__('sales/customer_remission.error_change_status'))->error();
        }

        //Redireccion
        return redirect('/sales/customer-remissions');
    }

    /**
     * Cambiar estatus a enviada
     *
     * @param CustomerRemission $customer_remission
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function markSent(CustomerRemission $customer_remission)
    {
        //Logica
        $customer_remission->updated_uid = \Auth::user()->id;
        $customer_remission->mail_sent = 1;
        $customer_remission->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/sales/customer-remissions');
    }

    /**
     * Clase generica de impresion
     *
     * @param CustomerRemission $customer_remission
     * @return mixed
     */
    public function print(CustomerRemission $customer_remission, $download = false, $save = false)
    {
        try {
            //Logica
            $tmp = 'default';
            $tmp = 'cfdi33';
            $class_print = 'print' . ucfirst($tmp);
            return $this->$class_print($customer_remission, $download, $save);

        } catch (\Exception $e) {
            flash($e->getMessage())->error();
            return redirect('/sales/customer-remissions');
        }
    }

    /**
     * Impresion default
     *
     * @param $customer_remission
     * @return mixed
     */
    private function printDefault($customer_remission, $download = false, $save = false)
    {
        //PDF
        $pdf = \PDF::loadView('sales.customer_remissions.pdf_default', compact('customer_remission'));

        //Guardar
        if ($save) {
            return $pdf->output();
        }

        //Descarga
        if($download){
            return $pdf->download($customer_remission->documentType->name . '_' . str_replace('/','',$customer_remission->name) . '.pdf');
        }

        //Redireccion
        return $pdf->stream($customer_remission->documentType->name . '_' . str_replace('/','',$customer_remission->name) . '.pdf');
    }

    /**
     * Impresion
     *
     * @param $customer_remission
     * @param bool $download
     * @return mixed
     * @throws \Exception
     */
    private function printCfdi33($customer_remission, $download = false, $save = false)
    {


        //PDF
        $pdf_template = \App\Helpers\Helper::companyPdfTemplate($customer_remission->company_id);//Plantilla
        $pdf = \PDF::loadView('sales.customer_remissions.pdf_cfdi33_' . $pdf_template,
            compact('customer_remission'));

        //Guardar
        if ($save) {
            return $pdf->output();
        }

        //Descargar
        if ($download) {
            return $pdf->download($customer_remission->documentType->name . '_' . str_replace('/','',$customer_remission->name) . '.pdf');
        }

        //Redireccion
        return $pdf->stream($customer_remission->documentType->name . '_' . str_replace('/','',$customer_remission->name) . '.pdf');
    }

    /**
     * Modal para envio de correo de factura
     *
     * @param Request $request
     * @param CustomerRemission $customer_remission
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function modalSendMail(Request $request, CustomerRemission $customer_remission)
    {
        //Variables
        $id = $request->id;

        //Logica
        if ($request->ajax() && !empty($id)) {
            //Correo default del cliente
            $reply = Helper::replyMails();
            $to = [];
            $to_selected = [];
            if (!empty($customer_remission->customer->email)) {
                $tmp = explode(';',$customer_remission->customer->email);
                if(!empty($tmp)) {
                    foreach($tmp as $email) {
                        $email = trim($email);
                        $to[$email] = $email;
                        $to_selected [] = $email;
                    }
                }
            }
            //Etiquetas solo son demostrativas
            $files = [
                'pdf' => str_replace('/','',$customer_remission->name) . '.pdf',
            ];
            $files_selected = array_keys($files);

            //modal de mensaje
            $html = view('layouts.partials.customer_remissions.modal_send_mail',
                compact('customer_remission', 'to', 'to_selected', 'files', 'files_selected', 'reply'))->render();

            //Mensaje predefinido
            $custom_message = view('layouts.partials.customer_remissions.message_send_mail',
                compact('customer_remission'))->render();

            return response()->json(['html' => $html, 'custom_message' => $custom_message]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Envio de factura por correo
     *
     * @param Request $request
     * @param CustomerRemission $customer_remission
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function sendMail(Request $request, CustomerRemission $customer_remission)
    {
        //Validaciones
        $validator = \Validator::make($request->all(), [
            'subject' => 'required',
            'to' => 'required',
            'to.*' => 'email',
            'message' => 'required'
        ], [
            'subject.*' => __('general.error_mail_subject'),
            'to.required' => __('general.error_mail_to'),
            'to.*.email' => __('general.error_mail_to_format'),
            'message.*' => __('general.error_mail_message'),
        ]);
        if ($validator->fails()) {
            $errors = '<ul>';
            foreach ($validator->errors()->all() as $message) {
                $errors .= '<li>'.$message . '</li>';
            }
            $errors .= '</ul>';
            return response()->json(['error' => $errors], 422);
        }

        //Creamos PDF en stream
        $pdf = $this->print($customer_remission);

        //Envio de correo
        $to = $request->to;
        $reply = $request->reply;
        \Mail::to($to)->send(new SendCustomerRemission($customer_remission, $request->subject, $request->message, $pdf, $reply));

        //Actualiza campo de enviado
        $customer_remission->updated_uid = \Auth::user()->id;
        $customer_remission->mail_sent = 1;
        $customer_remission->save();

        //Mensaje
        return response()->json([
            'success' => sprintf(__('sales/customer_remission.text_success_send_mail'), $customer_remission->name)
        ]);
    }

    /**
     * Obtener registro
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCustomerRemission(Request $request)
    {
        //Variables
        $id = $request->id;

        //Logica
        if ($request->ajax() && !empty($id)) {
            $customer_remission = CustomerRemission::findOrFail($id);
            $customer_remission->uuid = $customer_remission->customerInvoiceCfdi->uuid ?? '';
            return response()->json($customer_remission, 200);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Calcula el total de las lineas
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function totalLines(Request $request)
    {

        //Variables
        $json = new \stdClass;
        $input_items = $request->item;
        $currency_id = $request->currency_id;
        $currency_code = 'MXN';
        $taxes = [];

        if ($request->ajax()) {
            //Datos de moneda
            if (!empty($currency_id)) {
                $currency = Currency::findOrFail($currency_id);
                $currency_code = $currency->code;
            }
            //Variables de totales
            $amount_discount = 0;
            $amount_untaxed = 0;
            $amount_tax = 0;
            $amount_tax_ret = 0;
            $amount_total = 0;
            $balance = 0;
            $items = [];
            if (!empty($input_items)) {
                foreach ($input_items as $key => $item) {
                    //Logica
                    $item_quantity = (double)$item['quantity'];
                    $item_price_unit = (double)$item['price_unit'];
                    //Ajuste para cuando tiene IVA incluido
                    if (!empty($item['taxes']) && !empty($item['includes_iva'])) {
                        foreach ($item['taxes'] as $tax_id) {
                            if (!empty($tax_id)) {
                                $tax = Tax::findOrFail($tax_id);
                                if($tax->code == '002' && $tax->rate>0) {
                                    if ($tax->factor == 'Tasa') {
                                        $item_price_unit = $item_price_unit / (1+($tax->rate/100));
                                    } elseif ($tax->factor == 'Cuota') {
                                        $item_price_unit -= $tax->rate;
                                    }
                                    $item_price_unit = round($item_price_unit, \App\Helpers\Helper::companyProductPriceDecimalPlace());
                                    $input_items[$key]['price_unit'] = $item_price_unit;
                                }
                            }
                            break;
                        }
                    }
                    $item_discount = (double)$item['discount'];
                    $item_price_reduce = ($item_price_unit * (100 - $item_discount) / 100);
                    $item_amount_untaxed = round($item_quantity * $item_price_reduce, 2);
                    $item_amount_discount = round($item_quantity * $item_price_unit, 2) - $item_amount_untaxed;
                    $item_amount_tax = 0;
                    $item_amount_tax_ret = 0;

                    if (!empty($item['taxes'])) {
                        foreach ($item['taxes'] as $tax_id) {
                            if (!empty($tax_id)) {
                                $tax = Tax::findOrFail($tax_id);
                                $tmp = 0;
                                if ($tax->factor == 'Tasa') {
                                    $tmp = $item_amount_untaxed * $tax->rate / 100;
                                } elseif ($tax->factor == 'Cuota') {
                                    $tmp = $tax->rate;
                                }
                                $tmp = round($tmp, 2);
                                if ($tax->type == 'R') { //Retenciones
                                    $item_amount_tax_ret += $tmp;
                                } else { //Traslados
                                    $item_amount_tax += $tmp;
                                }

                                //Sumatoria de impuestos
                                $taxes[$tax_id] = array(
                                    'amount_base' => $item_amount_untaxed + (isset($taxes[$tax_id]['amount_base']) ? $taxes[$tax_id]['amount_base'] : 0),
                                    'amount_tax' => $tmp + (isset($taxes[$tax_id]['amount_tax']) ? $taxes[$tax_id]['amount_tax'] : 0),
                                );
                            }
                        }
                    }
                    $item_amount_total = $item_amount_untaxed + $item_amount_tax + $item_amount_tax_ret;
                    //Sumatoria totales
                    $amount_discount += $item_amount_discount;
                    $amount_untaxed += $item_amount_untaxed;
                    $amount_tax += $item_amount_tax;
                    $amount_tax_ret += $item_amount_tax_ret;
                    $amount_total += $item_amount_total;
                    //Subtotales por cada item
                    $items[$key] = money($item_amount_untaxed, $currency_code, true)->format();
                }
            }
            //Respuesta
            $taxes_tmp = [];
            if (!empty($taxes)) {
                foreach ($taxes as $tax_id => $result) {
                    $tax = Tax::findOrFail($tax_id);
                    $taxes_tmp[] = [
                        'name' => $tax->name,
                        'tax_id' => $tax_id,
                        'amount_base' => money($result['amount_base'], $currency_code, true)->format(),
                        'amount_tax' => money($result['amount_tax'], $currency_code, true)->format(),
                    ];
                }
            }

            //
            $json->items = $items;
            $json->taxes = $taxes_tmp;
            $json->amount_discount = money($amount_discount, $currency_code, true)->format();
            $json->amount_untaxed = money($amount_untaxed, $currency_code, true)->format();
            $json->amount_tax = money($amount_tax + $amount_tax_ret, $currency_code, true)->format();
            $json->amount_total = money($amount_total, $currency_code, true)->format();
            $json->amount_total_tmp = $amount_total;
            return response()->json($json);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Modal para cancelar factura
     *
     * @param Request $request
     * @param CustomerRemission $customer_remission
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function modalCancel(Request $request, CustomerRemission $customer_remission)
    {
        //Variables
        $id = $request->id;

        //Logica
        if ($request->ajax() && !empty($id)) {
            //Obtener informacion de estatus
            $data_status_sat = [
                'cancelable' => 1,
                'pac_is_cancelable' => ''
            ];

            $is_cancelable = true;
            if($data_status_sat['cancelable'] == 3){
                $is_cancelable = false;
            }

            //modal de cancelar
            $html = view('layouts.partials.customer_remissions.modal_cancel', compact('customer_remission','data_status_sat','is_cancelable'))->render();

            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Exportar datos a excel
     *
     * @param Request $request
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function exportToExcel(Request $request)
    {
        while (ob_get_level()) ob_end_clean();
        ob_start();

        return Excel::download(new CustomerRemissionsExport($request),
            __('sales/customer_remission.document_title') . '-' . config('app.name') . '.xlsx');
    }

}
