<?php

namespace App\Http\Controllers\Sales;

use App\Helpers\Helper;
use App\Models\Base\Pac;
use App\Helpers\PacHelper;
use App\Helpers\BaseHelper;
use Illuminate\Support\Str;
use App\Models\Catalogs\Tax;
use Illuminate\Http\Request;
use App\Helpers\Cfdi33Helper;
use App\Models\Sales\Customer;
use App\Models\Catalogs\CfdiUse;
use App\Models\Catalogs\Country;
use App\Models\Catalogs\Project;
use App\Mail\SendCustomerInvoice;
use App\Models\Base\BranchOffice;
use App\Models\Base\CfdiDownload;
use App\Models\Catalogs\Currency;
use App\Models\Sales\Salesperson;
use App\Mail\SendCustomerTransfer;
use App\Models\Catalogs\PaymentWay;
use App\Models\Catalogs\PermitType;
use App\Models\Catalogs\SatProduct;
use App\Models\Catalogs\TaxRegimen;
use App\Http\Controllers\Controller;
use App\Models\Catalogs\PaymentTerm;
use App\Models\Catalogs\StationType;
use App\Models\Catalogs\TrailerType;
use App\Models\Catalogs\UnitMeasure;
use Maatwebsite\Excel\Facades\Excel;
use App\Models\Catalogs\CfdiRelation;
use App\Models\Sales\CustomerInvoice;
use Illuminate\Support\Facades\Crypt;
use App\Models\Catalogs\PackagingType;
use App\Models\Catalogs\PaymentMethod;
use App\Models\Catalogs\TransportPart;
use App\Models\Catalogs\TransportType;
use App\Models\Catalogs\TariffFraction;
use App\Exports\CustomerTransfersExport;
use App\Models\Catalogs\DangerousProduct;
use App\Models\Catalogs\SettingTransport;
use App\Models\Catalogs\ReasonCancellation;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use App\Models\Sales\CustomerInvoiceTransfer;
use Illuminate\Validation\ValidationException;
use App\Models\Sales\CustomerInvoiceLineTransfer;
use App\Models\Sales\CustomerInvoice as CustomerTransfer;
use App\Exports\CustomerTransferTemplateTransferImportExport;
use App\Imports\CustomerTransferTemplateTransferImportImport;
use App\Models\Sales\CustomerInvoiceTax as CustomerTransferTax;
use App\Models\Sales\CustomerInvoiceCfdi as CustomerTransferCfdi;
use App\Models\Sales\CustomerInvoiceLine as CustomerTransferLine;
use App\Models\Sales\CustomerInvoiceRelation as CustomerTransferRelation;
use App\Models\Sales\CustomerInvoiceReconciled as CustomerTransferReconciled;
use App\Models\Sales\CustomerInvoiceLineTransfer as CustomerTransferLineTransfer;

class CustomerTransferController extends Controller
{
    private $list_status = [];
    private $list_document_type2 = [];
    private $document_type_code = 'customer.transfer';

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->list_status = [
            CustomerTransfer::DRAFT => __('sales/customer_transfer.text_status_draft'),
            CustomerTransfer::OPEN => __('sales/customer_transfer.text_status_open'),
            CustomerTransfer::TO_PAY => __('sales/customer_transfer.text_status_to_pay'),
            CustomerTransfer::PAID => __('sales/customer_transfer.text_status_paid'),
            CustomerTransfer::CANCEL => __('sales/customer_transfer.text_status_cancel'),
            CustomerTransfer::CANCEL_PER_AUTHORIZED => __('sales/customer_transfer.text_status_cancel_per_authorized'),
        ];
        $this->list_document_type2 = [
            'transfer' => __('sales/customer_transfer.text_document_type2_transfer'),
            'invoice' => __('sales/customer_transfer.text_document_type2_invoice'),
        ];
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);
        $salespersons = Salesperson::populateSelect()->pluck('name', 'id');
        $branch_offices = BranchOffice::populateSelect()->pluck('name', 'id');
        $list_status = $this->list_status;
        $list_document_type2 = $this->list_document_type2;
        if (empty($request->filter_date_from)) {
            $request->request->add([
                'filter_date_from' => Helper::date(\Date::parse('first day of this month'))
            ]);
        }
        if (empty($request->filter_date_to)) {
            $request->request->add([
                'filter_date_to' => Helper::date(\Date::parse('last day of this month'))
            ]);
        }
        $request->request->add(['filter_cfdi_type2' => 'transfer']); //Filtra tipo de documento

        //Consulta
        $results = CustomerTransfer::filter($request->all())
            ->with('customerInvoiceCfdi')
            ->with('customer')
            ->with('currency')
            ->sortable(['date' => 'desc'])->paginate($limit);

        //Vista
        return view('sales.customer_transfers.index', compact('results','salespersons', 'branch_offices','list_status','list_document_type2'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        $salespersons = Salesperson::populateSelect()->pluck('name', 'id');
        $branch_offices = BranchOffice::populateSelect()->pluck('name', 'id');
        $currencies = Currency::populateSelect()->get()->pluck('name_sat', 'id');
        $payment_terms = PaymentTerm::populateSelect()->pluck('name', 'id');
        $payment_ways = PaymentWay::populateSelect()->get()->pluck('name_sat', 'id');
        $payment_methods = PaymentMethod::populateSelect()->get()->pluck('name_sat', 'id');
        $cfdi_uses = CfdiUse::populateSelect()->get()->pluck('name_sat', 'id');
        $taxes = Tax::populateSelect()->pluck('name', 'id');
        $cfdi_relations = CfdiRelation::populateSelect()->get()->pluck('name_sat', 'id');
        $list_document_type2 = $this->list_document_type2;
        $projects = Project::populateSelect()->pluck('name', 'id');
        $in_out_products = ['Entrada' => 'Entrada', 'Salida' => 'Salida'];
        $transport_types = TransportType::populateSelect()->get()->pluck('name_sat', 'id');
        $station_types = StationType::populateSelect()->get()->pluck('name_sat', 'id');
        $countries = Country::populateSelect()->get()->pluck('name_sat', 'id');
        $countries2 = Country::populateSelect()->where('code','<>', 'MEX')->get()->pluck('name_sat', 'id');
        $traffic_navigations = ['Altura' => 'Altura', 'Cabotaje' => 'Cabotaje'];
        $permit_types = PermitType::populateSelect()->get()->pluck('name_sat', 'id');
        $setting_transports = SettingTransport::populateSelect()->get()->pluck('name_sat', 'id');
        $packaging_types = PackagingType::populateSelect()->get()->pluck('name_sat', 'id');
        $trailer_types = TrailerType::populateSelect()->get()->pluck('name_sat', 'id');
        $yes_nos = ['No' => 'No', 'Sí' => 'Sí'];
        $transport_parts = TransportPart::populateSelect()->get()->pluck('name_sat', 'id');
        $company = Helper::defaultCompany(); //Empresa
        if(!empty($company->tax_regimen_id2)){
            $tax_regimens = TaxRegimen::populateSelect()->whereIn('id',[$company->tax_regimen_id, $company->tax_regimen_id2])->get()->pluck('name_sat', 'id');
        }else{
            $tax_regimens = collect([]);
        }
        $tax_regimen_customers = TaxRegimen::populateSelect()->get()->pluck('name_sat', 'id');

        $duplicate_ci = null;
        $duplicate_ci_transfer = null;
        if($request->duplicate_id){
            $duplicate_ci = CustomerTransfer::findOrFail($request->duplicate_id);
            $duplicate_ci_transfer = $duplicate_ci->customerInvoiceTransfer;
        }
        $tax_objects = __('general.text_tax_objects');

        return view('sales.customer_transfers.create',
            compact('salespersons','branch_offices', 'currencies', 'payment_ways', 'payment_terms', 'payment_methods', 'cfdi_uses', 'taxes', 'cfdi_relations','list_document_type2','duplicate_ci', 'duplicate_ci_transfer','projects','in_out_products','transport_types','station_types','countries', 'countries2', 'traffic_navigations', 'permit_types','setting_transports','packaging_types','yes_nos', 'trailer_types','transport_parts','tax_regimens','tax_regimen_customers','tax_objects'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);

        //Almacenamiento de archivos
        $customer_transfer = null;

        \DB::connection('tenant')->beginTransaction();
        try {
            //Logica
            $company = Helper::defaultCompany(); //Empresa
            $request->merge(['created_uid' => \Auth::user()->id]);
            $request->merge(['updated_uid' => \Auth::user()->id]);
            $request->merge(['cfdi_type2' => 'transfer']);
            $request->merge(['status' => isset($request->pre_transfer) ? CustomerTransfer::DRAFT : CustomerTransfer::OPEN]);
            $request->merge(['company_id' => $company->id]);
            if($request->document_type2 != 'invoice'){
                $request->merge(['salesperson_id' => null]);
                $request->merge(['payment_way_id' => null]);
                $request->merge(['payment_method_id' => null]);
            }
            //Ajusta fecha y genera fecha de vencimiento
            $date = Helper::createDateTime($request->date);
            $request->merge(['date' => Helper::dateTimeToSql($date)]);
            $date_due = $date;
            $date_due_fix = $request->date_due;//Fix valida si la fecha de vencimiento esta vacia en caso de error
            if (!empty($request->date_due)) {
                $date_due = Helper::createDate($request->date_due);
            } else {
                $payment_term = PaymentTerm::findOrFail($request->payment_term_id);
                $date_due = $payment_term->days > 0 ? $date->copy()->addDays($payment_term->days) : $date->copy();
            }
            $request->merge(['date_due' => Helper::dateToSql($date_due)]);
            //$request->merge(['date_delivery' => Helper::convertDateToSql($request->date_delivery)]);
            $request->merge(['date_delivery' => null]);
            $request->merge(['valid_amount' => (double)$request->valid_amount]);
            $request->merge(['appraisal_fee' => (double)$request->appraisal_fee]);
            $source_date = Helper::createDateTime($request->source_date);
            $destination_date = Helper::createDateTime($request->destination_date);
            $request->merge(['source_date' => Helper::convertDateTimeToSql($request->source_date)]);
            $request->merge(['destination_date' => Helper::convertDateTimeToSql($request->destination_date)]);
            $request->merge(['distance_traveled' => $request->destination_distance_traveled]);

            //Obtiene folio
            $document_type_code = $this->document_type_code;
            if($request->document_type2 == 'invoice'){
                $document_type_code = 'customer.invoice';
            }
            $document_type = Helper::getNextDocumentTypeByCode($document_type_code,$company->id,isset($request->pre_transfer) ? true : false, $request->branch_office_id);
            $request->merge(['document_type_id' => $document_type['id']]);
            $request->merge(['name' => $document_type['name']]);
            $request->merge(['serie' => $document_type['serie']]);
            $request->merge(['folio' => $document_type['folio']]);

            //Valida que tenga folios disponibles
            if(BaseHelper::getAvailableFolios()<=0 && !isset($request->pre_transfer)){
                throw new \Exception(__('general.error_available_folios'));
            }

            //Guardar
            //Registro principal
            $customer_transfer = CustomerTransfer::create($request->input());
            $request->merge(['customer_invoice_id' => $customer_transfer->id]);
            $customer_transfer_transfer = CustomerInvoiceTransfer::create($request->input());

            //Registro de materiales
            if (!empty($request->item_transfer)) {
                foreach ($request->item_transfer as $key => $result) {
                    //Guardar
                    $customer_transfer_line_transfer = CustomerTransferLineTransfer::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'customer_invoice_id' => $customer_transfer->id,
                        'sat_product_id' => $result['sat_product_id'],
                        'name' => $result['name'],
                        'quantity' => (double)$result['quantity'],
                        'unit_measure_id' => $result['unit_measure_id'],
                        'weight' => $result['weight'],
                        'packaging_type_id' => $result['packaging_type_id'],
                        'dangerous_product' => $result['dangerous_product'],
                        'dangerous_product_id' => $result['dangerous_product_id'],
                        'dimensions' => $result['dimensions'],
                        'amount_total' => (double)$result['amount_total'],
                        'currency_id' => $result['currency_id'],
                        'tariff_fraction_id' => $result['tariff_fraction_id'],
                        'uuid_ext' => $result['uuid_ext'],
                        'm3' => $result['m3'] ?? 0,
                        'liters' => $result['liters'] ?? 0,
                        'sort_order' => $key,
                        'status' => 1,
                    ]);
                }
            }

            //Registro de lineas
            $amount_discount = 0;
            $amount_untaxed = 0;
            $amount_tax = 0;
            $amount_tax_ret = 0;
            $amount_total = 0;
            $balance = 0;
            $taxes = array();
            //Lineas
            if (!empty($request->item)) {
                foreach ($request->item as $key => $item) {
                    //Logica
                    $item_quantity = (double)$item['quantity'];
                    $item_price_unit = (double)$item['price_unit'];
                    $item_discount = (double)$item['discount'];
                    $item_price_reduce = ($item_price_unit * (100 - $item_discount) / 100);
                    $item_amount_untaxed = round($item_quantity * $item_price_reduce, 2);
                    $item_amount_discount = round($item_quantity * $item_price_unit, 2) - $item_amount_untaxed;
                    $item_amount_tax = 0;
                    $item_amount_tax_ret = 0;
                    if (!empty($item['taxes'])) {
                        foreach ($item['taxes'] as $tax_id) {
                            if (!empty($tax_id)) {
                                $tax = Tax::findOrFail($tax_id);
                                $tmp = 0;
                                if ($tax->factor == 'Tasa') {
                                    $tmp = $item_amount_untaxed * $tax->rate / 100;
                                } elseif ($tax->factor == 'Cuota') {
                                    $tmp = $tax->rate;
                                }
                                $tmp = round($tmp, 2);
                                if ($tax->type == 'R') { //Retenciones
                                    $item_amount_tax_ret += $tmp;
                                } else { //Traslados
                                    $item_amount_tax += $tmp;
                                }

                                //Sumatoria de impuestos
                                $taxes[$tax_id] = array(
                                    'amount_base' => $item_amount_untaxed + (isset($taxes[$tax_id]['amount_base']) ? $taxes[$tax_id]['amount_base'] : 0),
                                    'amount_tax' => $tmp + (isset($taxes[$tax_id]['amount_tax']) ? $taxes[$tax_id]['amount_tax'] : 0),
                                );
                            }
                        }
                    }
                    $item_amount_total = $item_amount_untaxed + $item_amount_tax + $item_amount_tax_ret;
                    //Sumatoria totales
                    $amount_discount += $item_amount_discount;
                    $amount_untaxed += $item_amount_untaxed;
                    $amount_tax += $item_amount_tax;
                    $amount_tax_ret += $item_amount_tax_ret;
                    $amount_total += $item_amount_total;

                    //Guardar linea
                    $customer_transfer_line = CustomerTransferLine::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'customer_invoice_id' => $customer_transfer->id,
                        'name' => $item['name'],
                        'product_id' => $item['product_id'],
                        'sat_product_id' => $item['sat_product_id'],
                        'unit_measure_id' => $item['unit_measure_id'],
                        'quantity' => $item_quantity,
                        'price_unit' => $item_price_unit,
                        'discount' => $item_discount,
                        'price_reduce' => $item_price_reduce,
                        'amount_discount' => $item_amount_discount,
                        'amount_untaxed' => $item_amount_untaxed,
                        'amount_tax' => $item_amount_tax,
                        'amount_tax_ret' => $item_amount_tax_ret,
                        'amount_total' => $item_amount_total,
                        'sort_order' => $key,
                        'status' => 1,
                        'tax_object' => !empty($item['tax_object']) ? $item['tax_object'] : (empty($item['taxes']) ? '01' : '02'),
                    ]);

                    //Guardar impuestos por linea
                    if (!empty($item['taxes'])) {
                        $customer_transfer_line->taxes()->sync($item['taxes']);
                    } else {
                        $customer_transfer_line->taxes()->sync([]);
                    }
                }
            }

            //Resumen de impuestos
            if (!empty($taxes)) {
                $i = 0;
                foreach ($taxes as $tax_id => $result) {
                    $tax = Tax::findOrFail($tax_id);
                    $customer_transfer_tax = CustomerTransferTax::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'customer_invoice_id' => $customer_transfer->id,
                        'name' => $tax->name,
                        'tax_id' => $tax_id,
                        'amount_base' => $result['amount_base'],
                        'amount_tax' => $result['amount_tax'],
                        'sort_order' => $i,
                        'status' => 1,
                    ]);
                    $i++;
                }
            }

            //Cfdi relacionados
            if (!empty($request->item_relation)) {
                foreach ($request->item_relation as $key => $result) {
                    //Guardar
                    $customer_transfer_relation = CustomerTransferRelation::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'customer_invoice_id' => $customer_transfer->id,
                        'relation_id' => !empty($result['relation_id']) ? $result['relation_id'] : null,
                        'sort_order' => $key,
                        'status' => 1,
                        'uuid_related' => $result['uuid_related'],
                    ]);
                }
            }

            //Valida que tenga exista la clase de facturacion
            $class_cfdi = setting('cfdi_version');
            if (empty($class_cfdi)) {
                throw new \Exception(__('general.error_cfdi_version'));
            }
            if (!method_exists($this, $class_cfdi)) {
                throw new \Exception(__('general.error_cfdi_class_exists'));
            }

            //Registros de cfdi
            $customer_transfer_cfdi = CustomerTransferCfdi::create([
                'created_uid' => \Auth::user()->id,
                'updated_uid' => \Auth::user()->id,
                'customer_invoice_id' => $customer_transfer->id,
                'name' => $customer_transfer->name,
                'cfdi_version' => $class_cfdi,
                'status' => 1,
            ]);

            //Actualiza registro principal con totales
            $customer_transfer->amount_discount = $amount_discount;
            $customer_transfer->amount_untaxed = $amount_untaxed;
            $customer_transfer->amount_tax = $amount_tax;
            $customer_transfer->amount_tax_ret = $amount_tax_ret;
            $customer_transfer->amount_total = $amount_total;
            $customer_transfer->balance = $amount_total;
            $customer_transfer->update();

            //Solo guarda los datos sin timbrar en caso de pre-factura
            if(isset($request->pre_transfer)) {
                //Mensaje
                flash(__('general.text_success_pre_customer_transfer'))->success();

            }else{ //Crear CFDI
                //Valida Empresa y PAC para timbrado
                PacHelper::validateSatActions($company);

                //Crear XML y timbra
                $tmp = $this->$class_cfdi($customer_transfer);

                //Guardar registros de CFDI
                $customer_transfer_cfdi->fill(array_only($tmp, [
                    'pac_id',
                    'cfdi_version',
                    'uuid',
                    'date',
                    'file_xml',
                    'file_xml_pac',
                ]));
                $customer_transfer_cfdi->save();

                //Disminuye folios
                BaseHelper::decrementFolios();

                //Mensaje
                flash(__('general.text_success_customer_transfer_cfdi'))->success();
            }

            \DB::connection('tenant')->commit();

            if(!isset($request->pre_transfer)) {
                $this->saveCfdiDownloads($customer_transfer, $customer_transfer_cfdi);
            }


        } catch (\Exception $e) {
            //Fix fechas
            $request->merge([
                'date' => Helper::convertSqlToDateTime($request->date),
            ]);
            if (!empty($date_due_fix)) {
                $request->merge([
                    'date_due' => Helper::convertSqlToDate($request->date_due),
                ]);
            }else{
                $request->merge([
                    'date_due' => '',
                ]);
            }
            /*$request->merge([
                'date_delivery' => Helper::convertSqlToDate($request->date_delivery),
            ]);*/

            $request->merge([
                'source_date' => Helper::convertSqlToDateTime($request->source_date),
            ]);
            $request->merge([
                'destination_date' => Helper::convertSqlToDateTime($request->destination_date),
            ]);

            \DB::connection('tenant')->rollback();
            $company = Helper::defaultCompany(); //Empresa
            \Log::error('(' . $company->taxid . ') ' . $e->getMessage());
            flash((isset($request->pre_transfer) ? '' : __('general.error_cfdi_pac').'<br>').$e->getMessage())->error();
            return back()->withInput();
        }

        //Almacenamiento dropbox
        self::dropboxBackup($customer_transfer);

        //Redireccion
        return redirect('/sales/customer-transfers');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Sales\CustomerTransfer  $customer_transfer
     * @return \Illuminate\Http\Response
     */
    public function show(CustomerTransfer $customer_transfer)
    {
        //Datos
        $list_document_type2 = $this->list_document_type2;
        $data = [];

        //Si tiene CFDI obtiene la informacion de los nodos
        if(!empty($customer_transfer->customerInvoiceCfdi->file_xml_pac) && !empty($customer_transfer->customerInvoiceCfdi->uuid)){

            $path_xml = Helper::setDirectory(CustomerTransfer::PATH_XML_FILES_CTR, $customer_transfer->company_id) . '/';
            if($customer_transfer->document_type2 == 'invoice'){
                $path_xml = Helper::setDirectory(CustomerTransfer::PATH_XML_FILES_CI, $customer_transfer->company_id) . '/';
            }
            $file_xml_pac = $path_xml . $customer_transfer->customerInvoiceCfdi->file_xml_pac;

            //Valida que el archivo exista
            if(!empty($customer_transfer->customerInvoiceCfdi->file_xml_pac) && \Storage::exists($file_xml_pac)) {
                $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($file_xml_pac));
                $data = Cfdi33Helper::getQuickArrayCfdi($cfdi);

                //Genera codigo QR
                $image = QrCode::format('png')->size(150)->margin(0)->generate($data['qr_cadena']);
                $data['qr'] = 'data:image/png;base64,' . base64_encode($image);

                $data['tax_regimen'] = TaxRegimen::where('code', $data['cfdi33']->emisor['RegimenFiscal'])->first()->name_sat;
                $data['tax_regimen_customer'] = !empty($data['cfdi33']->receptor['RegimenFiscalReceptor']) ? TaxRegimen::where('code', $data['cfdi33']->receptor['RegimenFiscalReceptor'])->first()->name_sat : '';
            }
        }

        return view('sales.customer_transfers.show', compact('customer_transfer','data','list_document_type2'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Sales\CustomerTransfer  $customer_transfer
     * @return \Illuminate\Http\Response
     */
    public function edit(CustomerTransfer $customer_transfer)
    {
        $salespersons = Salesperson::populateSelect()->pluck('name', 'id');
        $branch_offices = BranchOffice::populateSelect()->pluck('name', 'id');
        $currencies = Currency::populateSelect()->get()->pluck('name_sat', 'id');
        $payment_terms = PaymentTerm::populateSelect()->pluck('name', 'id');
        $payment_ways = PaymentWay::populateSelect()->get()->pluck('name_sat', 'id');
        $payment_methods = PaymentMethod::populateSelect()->get()->pluck('name_sat', 'id');
        $cfdi_uses = CfdiUse::populateSelect()->get()->pluck('name_sat', 'id');
        $taxes = Tax::populateSelect()->pluck('name', 'id');
        $cfdi_relations = CfdiRelation::populateSelect()->get()->pluck('name_sat', 'id');
        $list_document_type2 = $this->list_document_type2;
        $projects = Project::populateSelect()->pluck('name', 'id');
        $in_out_products = ['Entrada' => 'Entrada', 'Salida' => 'Salida'];
        $transport_types = TransportType::populateSelect()->get()->pluck('name_sat', 'id');
        $station_types = StationType::populateSelect()->get()->pluck('name_sat', 'id');
        $countries = Country::populateSelect()->get()->pluck('name_sat', 'id');
        $countries2 = Country::populateSelect()->where('code','<>', 'MEX')->get()->pluck('name_sat', 'id');
        $traffic_navigations = ['Altura' => 'Altura', 'Cabotaje' => 'Cabotaje'];
        $permit_types = PermitType::populateSelect()->get()->pluck('name_sat', 'id');
        $setting_transports = SettingTransport::populateSelect()->get()->pluck('name_sat', 'id');
        $packaging_types = PackagingType::populateSelect()->get()->pluck('name_sat', 'id');
        $trailer_types = TrailerType::populateSelect()->get()->pluck('name_sat', 'id');
        $yes_nos = ['No' => 'No', 'Sí' => 'Sí'];
        $transport_parts = TransportPart::populateSelect()->get()->pluck('name_sat', 'id');
        $company = $customer_transfer->company; //Empresa
        if(!empty($company->tax_regimen_id2)){
            $tax_regimens = TaxRegimen::populateSelect()->whereIn('id',[$company->tax_regimen_id, $company->tax_regimen_id2])->get()->pluck('name_sat', 'id');
        }else{
            $tax_regimens = collect([]);
        }
        $tax_regimen_customers = TaxRegimen::populateSelect()->get()->pluck('name_sat', 'id');

        $customer_transfer_transfer = CustomerInvoiceTransfer::where('customer_invoice_id', $customer_transfer->id)->first();
        $tax_objects = __('general.text_tax_objects');

        return view('sales.customer_transfers.edit',
            compact('customer_transfer','salespersons', 'branch_offices', 'currencies', 'payment_ways', 'payment_terms', 'payment_methods',
                'cfdi_uses', 'taxes', 'cfdi_relations','list_document_type2','projects','in_out_products','transport_types','station_types','countries', 'countries2', 'traffic_navigations', 'permit_types','setting_transports','packaging_types','yes_nos', 'trailer_types','transport_parts', 'customer_transfer_transfer','tax_regimens','tax_regimen_customers','tax_objects'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Sales\CustomerTransfer  $customer_transfer
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, CustomerTransfer $customer_transfer)
    {
        //Validacion
        $this->validation($request);

        \DB::connection('tenant')->beginTransaction();
        try {

            //Logica
            $request->merge(['created_uid' => \Auth::user()->id]);
            $request->merge(['updated_uid' => \Auth::user()->id]);
            $request->merge(['status' => isset($request->pre_transfer) ? CustomerTransfer::DRAFT : CustomerTransfer::OPEN]);
            if($request->document_type2 != 'invoice'){
                $request->merge(['salesperson_id' => null]);
                $request->merge(['payment_way_id' => null]);
                $request->merge(['payment_method_id' => null]);
            }
            //Ajusta fecha y genera fecha de vencimiento
            $date = Helper::createDateTime($request->date);
            $request->merge(['date' => Helper::dateTimeToSql($date)]);
            $date_due = $date;
            $date_due_fix = $request->date_due;//Fix valida si la fecha de vencimiento esta vacia en caso de error
            if (!empty($request->date_due)) {
                $date_due = Helper::createDate($request->date_due);
            } else {
                $payment_term = PaymentTerm::findOrFail($request->payment_term_id);
                $date_due = $payment_term->days > 0 ? $date->copy()->addDays($payment_term->days) : $date->copy();
            }
            $request->merge(['date_due' => Helper::dateToSql($date_due)]);
            $request->merge(['date_delivery' => null]);

            $request->merge(['valid_amount' => (double)$request->valid_amount]);
            $request->merge(['appraisal_fee' => (double)$request->appraisal_fee]);
            $source_date = Helper::createDateTime($request->source_date);
            $destination_date = Helper::createDateTime($request->destination_date);
            $request->merge(['source_date' => Helper::convertDateTimeToSql($request->source_date)]);
            $request->merge(['destination_date' => Helper::convertDateTimeToSql($request->destination_date)]);
            $request->merge(['distance_traveled' => $request->destination_distance_traveled]);

            $customer_transfer->fill($request->only([
                'updated_uid',
                'date',
                'date_due',
                'reference',
                'customer_id',
                'branch_office_id',
                'payment_term_id',
                'payment_way_id',
                'payment_method_id',
                'cfdi_use_id',
                'salesperson_id',
                'currency_id',
                'currency_value',
                'cfdi_relation_id',
                'comment',
                'status',
                'date_delivery',
                'source_name',
                'source_address',
                'source_delivery_on',
                'destination_name',
                'destination_address',
                'destination_delivery_on',
                'valid_amount',
                'dangerous_material',
                'compensation',
                'driver',
                'vehicle',
                'vehicle_number',
                'vehicle_counter',
                'document_type2',
                'project_id',
                'tax_regimen_id',
                'tax_regimen_customer_id'
            ]));

            //Guardar
            //Registro principal
            $customer_transfer->save();

            $customer_transfer_transfer = CustomerInvoiceTransfer::where('customer_invoice_id',$customer_transfer->id)->first();
            if(!empty($customer_transfer_transfer)){
                $customer_transfer_transfer->fill($request->only([
                    'updated_uid',
                    'transport_int',
                    'distance_traveled',
                    'in_out_product',
                    'in_out_country_id',
                    'transport_type_id',
                    'trans_transport_type_id',
                    'source_station_type_id',
                    'source_distance_traveled',
                    'source_id',
                    'source_taxid',
                    'source_name',
                    'source_tr_country_id',
                    'source_numid',
                    'source_station_id',
                    'source_station',
                    'source_traffic_navigation',
                    'source_date',
                    'source_address_1',
                    'source_address_2',
                    'source_address_3',
                    'source_address_4',
                    'source_address_5',
                    'source_address_6',
                    'source_city_id',
                    'source_state_id',
                    'source_country_id',
                    'source_postcode',
                    'destination_station_type_id',
                    'destination_distance_traveled',
                    'destination_id',
                    'destination_taxid',
                    'destination_name',
                    'destination_tr_country_id',
                    'destination_numid',
                    'destination_station_id',
                    'destination_station',
                    'destination_traffic_navigation',
                    'destination_date',
                    'destination_address_1',
                    'destination_address_2',
                    'destination_address_3',
                    'destination_address_4',
                    'destination_address_5',
                    'destination_address_6',
                    'destination_city_id',
                    'destination_state_id',
                    'destination_country_id',
                    'destination_postcode',
                    'operator_taxid',
                    'operator_license',
                    'operator_name',
                    'operator_tr_country_id',
                    'operator_numid',
                    'gross_weight',
                    'measure_packaging_id',
                    'net_weight',
                    'appraisal_fee',
                    'permit_type_id',
                    'permit_number',
                    'setting_transport_id',
                    'vehicle',
                    'vehicle_number',
                    'name_insurer',
                    'policy_number',
                    'name_insurer2',
                    'policy_number2',
                    'name_insurer3',
                    'policy_number3',
                    'insurance_value',
                    'trailer_type_id_1',
                    'trailer_number_1',
                    'trailer_type_id_2',
                    'trailer_number_2',
                    'trailer_type_id_3',
                    'trailer_number_3',
                    'trailer_type_id_4',
                    'trailer_number_4',
                    'trailer_type_id_5',
                    'trailer_number_5',
                    'owner_taxid',
                    'owner_name',
                    'owner_tr_country_id',
                    'owner_numid',
                    'owner_transport_part_id',
                    'owner_address_1',
                    'owner_address_2',
                    'owner_address_3',
                    'owner_address_4',
                    'owner_address_5',
                    'owner_address_6',
                    'owner_city_id',
                    'owner_state_id',
                    'owner_country_id',
                    'owner_postcode',
                    'lessee_taxid',
                    'lessee_name',
                    'lessee_tr_country_id',
                    'lessee_numid',
                    'lessee_transport_part_id',
                    'lessee_address_1',
                    'lessee_address_2',
                    'lessee_address_3',
                    'lessee_address_4',
                    'lessee_address_5',
                    'lessee_address_6',
                    'lessee_city_id',
                    'lessee_state_id',
                    'lessee_country_id',
                    'lessee_postcode',
                    'notified_taxid',
                    'notified_name',
                    'notified_tr_country_id',
                    'notified_numid',
                ]));

                $customer_transfer_transfer->save();
            }

            //Actualiza folios
            if(!isset($request->pre_transfer)) {
                //Valida que tenga folios disponibles
                if(BaseHelper::getAvailableFolios()<=0){
                    throw new \Exception(__('general.error_available_folios'));
                }

                //Obtiene folio
                $document_type_code = $this->document_type_code;
                if($request->document_type2 == 'invoice'){
                    $document_type_code = 'customer.invoice';
                }
                $document_type = Helper::getNextDocumentTypeByCode($document_type_code, $customer_transfer->company->id, false, $customer_transfer->branch_office_id);
                $customer_transfer->draft = $customer_transfer->name;
                $customer_transfer->document_type_id = $document_type['id'];
                $customer_transfer->name = $document_type['name'];
                $customer_transfer->serie = $document_type['serie'];
                $customer_transfer->folio = $document_type['folio'];
                $customer_transfer->save();
            }

            //Elimina partidas
            if (!empty($request->delete_item_transfer)) {
                foreach ($request->delete_item_transfer as $key => $result) {
                    //Actualizar status
                    $customer_invoice_line_transfer = CustomerInvoiceLineTransfer::findOrFail($result);
                    $customer_invoice_line_transfer->updated_uid = \Auth::user()->id;
                    $customer_invoice_line_transfer->status = 0;
                    $customer_invoice_line_transfer->save();
                }
            }

            //Registro de materiales
            if (!empty($request->item_transfer)) {
                foreach ($request->item_transfer as $key => $result) {
                    //Guardar
                    $data = [
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'customer_invoice_id' => $customer_transfer->id,
                        'sat_product_id' => $result['sat_product_id'],
                        'name' => $result['name'],
                        'quantity' => (double)$result['quantity'],
                        'unit_measure_id' => $result['unit_measure_id'],
                        'weight' => $result['weight'],
                        'packaging_type_id' => $result['packaging_type_id'],
                        'dangerous_product' => $result['dangerous_product'],
                        'dangerous_product_id' => $result['dangerous_product_id'],
                        'dimensions' => $result['dimensions'],
                        'amount_total' => (double)$result['amount_total'],
                        'currency_id' => $result['currency_id'],
                        'tariff_fraction_id' => $result['tariff_fraction_id'],
                        'uuid_ext' => $result['uuid_ext'],
                        'm3' => $result['m3'] ?? 0,
                        'liters' => $result['liters'] ?? 0,
                        'sort_order' => $key,
                        'status' => 1,
                    ];
                    if (!empty($result['id'])) {
                        $customer_transfer_line_transfer = CustomerTransferLineTransfer::findOrFail($result['id']);
                        $customer_transfer_line_transfer->fill(array_only($data, [
                            'updated_uid',
                            'name',
                            'product_id',
                            'weight',
                            'm3',
                            'liters',
                            'packaging',
                            'sort_order',
                            'status',
                            'sat_product_id',
                            'quantity',
                            'unit_measure_id',
                            'dimensions',
                            'dangerous_product',
                            'dangerous_product_id',
                            'packaging_type_id',
                            'amount_total',
                            'currency_id',
                            'tariff_fraction_id',
                            'uuid_ext',
                        ]));
                        $customer_transfer_line_transfer->save();
                    }else{
                        $customer_transfer_line_transfer = CustomerTransferLineTransfer::create($data);
                    }
                }
            }

            //Elimina partidas
            if (!empty($request->delete_item)) {
                foreach ($request->delete_item as $key => $result) {
                    //Actualizar status
                    $customer_transfer_line = CustomerTransferLine::findOrFail($result);
                    $customer_transfer_line->updated_uid = \Auth::user()->id;
                    $customer_transfer_line->status = 0;
                    $customer_transfer_line->save();
                }
            }
            //Registro de lineas
            $amount_discount = 0;
            $amount_untaxed = 0;
            $amount_tax = 0;
            $amount_tax_ret = 0;
            $amount_total = 0;
            $balance = 0;
            $taxes = array();
            //Lineas
            if (!empty($request->item)) {
                foreach ($request->item as $key => $item) {
                    //Logica
                    $item_quantity = (double)$item['quantity'];
                    $item_price_unit = (double)$item['price_unit'];
                    $item_discount = (double)$item['discount'];
                    $item_price_reduce = ($item_price_unit * (100 - $item_discount) / 100);
                    $item_amount_untaxed = round($item_quantity * $item_price_reduce, 2);
                    $item_amount_discount = round($item_quantity * $item_price_unit, 2) - $item_amount_untaxed;
                    $item_amount_tax = 0;
                    $item_amount_tax_ret = 0;
                    if (!empty($item['taxes'])) {
                        foreach ($item['taxes'] as $tax_id) {
                            if (!empty($tax_id)) {
                                $tax = Tax::findOrFail($tax_id);
                                $tmp = 0;
                                if ($tax->factor == 'Tasa') {
                                    $tmp = $item_amount_untaxed * $tax->rate / 100;
                                } elseif ($tax->factor == 'Cuota') {
                                    $tmp = $tax->rate;
                                }
                                $tmp = round($tmp, 2);
                                if ($tax->type == 'R') { //Retenciones
                                    $item_amount_tax_ret += $tmp;
                                } else { //Traslados
                                    $item_amount_tax += $tmp;
                                }

                                //Sumatoria de impuestos
                                $taxes[$tax_id] = array(
                                    'amount_base' => $item_amount_untaxed + (isset($taxes[$tax_id]['amount_base']) ? $taxes[$tax_id]['amount_base'] : 0),
                                    'amount_tax' => $tmp + (isset($taxes[$tax_id]['amount_tax']) ? $taxes[$tax_id]['amount_tax'] : 0),
                                );
                            }
                        }
                    }
                    $item_amount_total = $item_amount_untaxed + $item_amount_tax + $item_amount_tax_ret;
                    //Sumatoria totales
                    $amount_discount += $item_amount_discount;
                    $amount_untaxed += $item_amount_untaxed;
                    $amount_tax += $item_amount_tax;
                    $amount_tax_ret += $item_amount_tax_ret;
                    $amount_total += $item_amount_total;

                    //Guardar linea
                    $data = [
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'customer_invoice_id' => $customer_transfer->id,
                        'name' => $item['name'],
                        'product_id' => $item['product_id'],
                        'sat_product_id' => $item['sat_product_id'],
                        'unit_measure_id' => $item['unit_measure_id'],
                        'quantity' => $item_quantity,
                        'price_unit' => $item_price_unit,
                        'discount' => $item_discount,
                        'price_reduce' => $item_price_reduce,
                        'amount_discount' => $item_amount_discount,
                        'amount_untaxed' => $item_amount_untaxed,
                        'amount_tax' => $item_amount_tax,
                        'amount_tax_ret' => $item_amount_tax_ret,
                        'amount_total' => $item_amount_total,
                        'sort_order' => $key,
                        'status' => 1,
                        'tax_object' => !empty($item['tax_object']) ? $item['tax_object'] : (empty($item['taxes']) ? '01' : '02'),
                    ];
                    if (!empty($item['id'])) {
                        $customer_transfer_line = CustomerTransferLine::findOrFail($item['id']);
                        $customer_transfer_line->fill(array_only($data, [
                            'updated_uid',
                            'name',
                            'product_id',
                            'sat_product_id',
                            'unit_measure_id',
                            'quantity',
                            'price_unit',
                            'discount',
                            'price_reduce',
                            'amount_discount',
                            'amount_untaxed',
                            'amount_tax',
                            'amount_tax_ret',
                            'amount_total',
                            'sort_order',
                            'tax_object'
                        ]));
                        $customer_transfer_line->save();
                    }else{
                        $customer_transfer_line = CustomerTransferLine::create($data);
                    }

                    //Guardar impuestos por linea
                    if (!empty($item['taxes'])) {
                        $customer_transfer_line->taxes()->sync($item['taxes']);
                    } else {
                        $customer_transfer_line->taxes()->sync([]);
                    }
                }
            }

            //Resumen de impuestos
            CustomerTransferTax::where('customer_invoice_id','=',$customer_transfer->id)->delete(); //Borra todo e inserta nuevamente
            if (!empty($taxes)) {
                $i = 0;
                foreach ($taxes as $tax_id => $result) {
                    $tax = Tax::findOrFail($tax_id);
                    $customer_transfer_tax = CustomerTransferTax::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'customer_invoice_id' => $customer_transfer->id,
                        'name' => $tax->name,
                        'tax_id' => $tax_id,
                        'amount_base' => $result['amount_base'],
                        'amount_tax' => $result['amount_tax'],
                        'sort_order' => $i,
                        'status' => 1,
                    ]);
                    $i++;
                }
            }

            //Cfdi relacionados
            CustomerTransferRelation::where('customer_invoice_id','=',$customer_transfer->id)->delete(); //Borra todo e inserta nuevamente
            if (!empty($request->item_relation)) {
                foreach ($request->item_relation as $key => $result) {
                    //Guardar
                    $customer_transfer_relation = CustomerTransferRelation::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'customer_invoice_id' => $customer_transfer->id,
                        'relation_id' => !empty($result['relation_id']) ? $result['relation_id'] : null,
                        'sort_order' => $key,
                        'status' => 1,
                        'uuid_related' => $result['uuid_related'],
                    ]);
                }
            }

            //Valida que tenga exista la clase de facturacion
            $class_cfdi = setting('cfdi_version');
            if (empty($class_cfdi)) {
                throw new \Exception(__('general.error_cfdi_version'));
            }
            if (!method_exists($this, $class_cfdi)) {
                throw new \Exception(__('general.error_cfdi_class_exists'));
            }

            //Registros de cfdi
            CustomerTransferCfdi::where('customer_invoice_id','=',$customer_transfer->id)->delete(); //Borra todo e inserta nuevamente
            $customer_transfer_cfdi = CustomerTransferCfdi::create([
                'created_uid' => \Auth::user()->id,
                'updated_uid' => \Auth::user()->id,
                'customer_invoice_id' => $customer_transfer->id,
                'name' => $customer_transfer->name,
                'cfdi_version' => $class_cfdi,
                'status' => 1,
            ]);

            //Actualiza registro principal con totales
            $customer_transfer->amount_discount = $amount_discount;
            $customer_transfer->amount_untaxed = $amount_untaxed;
            $customer_transfer->amount_tax = $amount_tax;
            $customer_transfer->amount_tax_ret = $amount_tax_ret;
            $customer_transfer->amount_total = $amount_total;
            $customer_transfer->balance = $amount_total;
            $customer_transfer->update();

            //Solo guarda los datos sin timbrar en caso de pre-factura
            if(isset($request->pre_transfer)) {
                //Mensaje
                flash(__('general.text_success_pre_customer_transfer'))->success();

            }else{ //Crear CFDI
                //Valida Empresa y PAC para timbrado
                PacHelper::validateSatActions($customer_transfer->company);

                //Crear XML y timbra
                $tmp = $this->$class_cfdi($customer_transfer);

                //Guardar registros de CFDI
                $customer_transfer_cfdi->fill(array_only($tmp, [
                    'pac_id',
                    'cfdi_version',
                    'uuid',
                    'date',
                    'file_xml',
                    'file_xml_pac',
                ]));
                $customer_transfer_cfdi->save();

                //Disminuye folios
                BaseHelper::decrementFolios();

                //Mensaje
                flash(__('general.text_success_customer_transfer_cfdi'))->success();
            }

            \DB::connection('tenant')->commit();

            if(!isset($request->pre_transfer)) {
                $this->saveCfdiDownloads($customer_transfer, $customer_transfer_cfdi);
            }


        } catch (\Exception $e) {
            //Fix fechas
            $request->merge([
                'date' => Helper::convertSqlToDateTime($request->date),
            ]);
            if (!empty($date_due_fix)) {
                $request->merge([
                    'date_due' => Helper::convertSqlToDate($request->date_due),
                ]);
            }else{
                $request->merge([
                    'date_due' => '',
                ]);
            }
            /*$request->merge([
                'date_delivery' => Helper::convertSqlToDate($request->date_delivery),
            ]);*/

            $request->merge([
                'source_date' => Helper::convertSqlToDateTime($request->source_date),
            ]);
            $request->merge([
                'destination_date' => Helper::convertSqlToDateTime($request->destination_date),
            ]);

            \DB::connection('tenant')->rollback();
            \Log::error('(' . $customer_transfer->company->taxid . ') ' . $e->getMessage());
            flash((isset($request->pre_invoice) ? '' : __('general.error_cfdi_pac').'<br>').$e->getMessage())->error();
            return back()->withInput();
        }

        //Almacenamiento dropbox
        self::dropboxBackup($customer_transfer);

        //Redireccion
        return redirect('/sales/customer-transfers');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Sales\CustomerTransfer  $customer_transfer
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, CustomerTransfer $customer_transfer)
    {
        \DB::connection('tenant')->beginTransaction();
        try {
            //Logica
            if ((int)$customer_transfer->status != CustomerTransfer::CANCEL && $customer_transfer->balance >= $customer_transfer->amount_total) {
                //Actualiza status
                $customer_transfer->updated_uid = \Auth::user()->id;
                $customer_transfer->status = CustomerTransfer::CANCEL;
                //Por autorizar cuando se manda la autorizacion al buzon tributario del SAT
                if($request->cancelable == 2){
                    $customer_transfer->status = CustomerTransfer::CANCEL_PER_AUTHORIZED;
                }
                $customer_transfer->save();

                //Actualiza status CFDI
                $customer_transfer->customerInvoiceCfdi->status = 0;
                $customer_transfer->customerInvoiceCfdi->reason_cancellation_id = $request->reason_cancellation_id;
                $customer_transfer->customerInvoiceCfdi->reason_cancellation_uuid = $request->reason_cancellation_uuid;
                $customer_transfer->customerInvoiceCfdi->save();

                //Cancelacion del timbre fiscal
                if (!empty($customer_transfer->customerInvoiceCfdi->cfdi_version) && !empty($customer_transfer->customerInvoiceCfdi->uuid)) {
                    //Valida Empresa y PAC para cancelar timbrado
                    PacHelper::validateSatCancelActions($customer_transfer->company,$customer_transfer->customerInvoiceCfdi->pac);

                    //Obtener el sellos del CFDI
                    $path_xml = Helper::setDirectory(CustomerTransfer::PATH_XML_FILES_CTR, $customer_transfer->company_id) . '/';
                    if($customer_transfer->document_type2 == 'invoice'){
                        $path_xml = Helper::setDirectory(CustomerTransfer::PATH_XML_FILES_CI, $customer_transfer->company_id) . '/';
                    }
                    $file_xml_pac = $path_xml . $customer_transfer->customerInvoiceCfdi->file_xml_pac;
                    $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($file_xml_pac));
                    $data = Cfdi33Helper::getQuickArrayCfdi($cfdi);

                    //Arreglo temporal para actualizar Customer Invoice CFDI
                    $tmp = [
                        'cancel_date' => Helper::dateTimeToSql(\Date::now()),
                        'cancel_response' => '',
                        'cancel_state' => $request->cancel_state,
                        'rfcR' => $data['cfdi33']->Receptor['Rfc'] ?? $customer_transfer->customer->taxid,
                        'uuid' => $customer_transfer->customerInvoiceCfdi->uuid,
                        'total' => Helper::numberFormat($customer_transfer->amount_total,
                            $customer_transfer->currency->decimal_place, false),
                        'cfdi_type' => $customer_transfer->documentType->cfdiType->code,
                        'cfdi_version' => 'cfdi3.3',
                        'fe' => substr($data['cfdi33']->complemento->timbreFiscalDigital['SelloCFD'],-8), //Los últimos 8 caracteres del Atributo Sello del CFDI
                        'file_xml_pac' => $file_xml_pac,
                        'reason_cancellation_code' => $customer_transfer->customerInvoiceCfdi->reasonCancellation->code ?? '',
                        'reason_cancellation_uuid' => $customer_transfer->customerInvoiceCfdi->reason_cancellation_uuid ?? '',
                    ];

                    //Cancelar Timbrado de XML
                    $class_pac = $customer_transfer->customerInvoiceCfdi->pac->code . 'Cancel';
                    $tmp = PacHelper::$class_pac($tmp,$customer_transfer->company,$customer_transfer->customerInvoiceCfdi->pac);

                    //Guardar registros de CFDI
                    $customer_transfer->customerInvoiceCfdi->fill(array_only($tmp,[
                        'cancel_date',
                        'cancel_response',
                        'cancel_state',
                    ]));
                    $customer_transfer->customerInvoiceCfdi->save();

                    $cfdi_download = CfdiDownload::where('uuid', $customer_transfer->customerInvoiceCfdi->uuid)->where('type', 2)->first();
                    if(!empty($cfdi_download)){
                        $cfdi_download->status = 'Cancelado';
                        $cfdi_download->save();
                    }

                    //Disminuye folios
                    BaseHelper::decrementFolios();

                }
            }
            \DB::connection('tenant')->commit();

            //Mensaje
            flash(__('general.text_form_success_cancel'))->success();


        } catch (\Exception $e) {
            \DB::connection('tenant')->rollback();
            \Log::error('(' . $customer_transfer->company->taxid . ') ' . $e->getMessage());
            flash($e->getMessage())->error();
            return redirect('/sales/customer-transfers');
        }

        //Almacenamiento dropbox
        if ((int)$customer_transfer->status == CustomerTransfer::CANCEL) {
            self::dropboxBackup($customer_transfer,false,true);
        }

        //Redireccion
        return redirect('/sales/customer-transfers');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'customer_id' => 'required|integer',
            'branch_office_id' => 'required|integer',
            'date' => 'required|date|date_format:"'.setting('datetime_format', 'd-m-Y H:i:s').'"',
            'currency_id' => 'required|integer',
            'currency_value' => 'required|numeric|min:0.1',
            'payment_term_id' => 'required|integer',
            'cfdi_use_id' => 'required',
            'cfdi_relation_id' => 'nullable|integer|required_with:item_relation',
            'item' => 'required',
            'item.*.name' => 'required',
            'item.*.unit_measure_id' => 'required',
            'item.*.sat_product_id' => 'required',
            'item.*.quantity' => 'required|numeric|min:0.00001',
            'item.*.price_unit' => 'required|numeric|min:0',
            'item.*.discount' => 'nullable|numeric|min:0',
            'item_relation.*.uuid_related' => 'required',
            'item_transfer' => 'required',
            'item_transfer.*.name' => 'required',
            'source_taxid' => [
                'nullable',
                'regex:/^[A-Z&Ñ]{3,4}[0-9]{2}(0[1-9]|1[012])(0[1-9]|[12][0-9]|3[01])[A-Z0-9]{2}[0-9A]$/i'
            ],
            'destination_taxid' => [
                'nullable',
                'regex:/^[A-Z&Ñ]{3,4}[0-9]{2}(0[1-9]|1[012])(0[1-9]|[12][0-9]|3[01])[A-Z0-9]{2}[0-9A]$/i'
            ],
            'operator_taxid' => [
                'nullable',
                'regex:/^[A-Z&Ñ]{3,4}[0-9]{2}(0[1-9]|1[012])(0[1-9]|[12][0-9]|3[01])[A-Z0-9]{2}[0-9A]$/i'
            ],
            'owner_taxid' => [
                'nullable',
                'regex:/^[A-Z&Ñ]{3,4}[0-9]{2}(0[1-9]|1[012])(0[1-9]|[12][0-9]|3[01])[A-Z0-9]{2}[0-9A]$/i'
            ],
            'lessee_taxid' => [
                'nullable',
                'regex:/^[A-Z&Ñ]{3,4}[0-9]{2}(0[1-9]|1[012])(0[1-9]|[12][0-9]|3[01])[A-Z0-9]{2}[0-9A]$/i'
            ],
        ], [
            'customer_id.*' => __('sales/customer_transfer.error_customer_id'),
            'branch_office_id.*' => __('sales/customer_transfer.error_branch_office_id'),
            'date.required' => __('sales/customer_transfer.error_date'),
            'date.date' => __('sales/customer_transfer.error_date_format'),
            'date.date_format' => __('sales/customer_transfer.error_date_format'),
            'currency_id.*' => __('sales/customer_transfer.error_currency_id'),
            'currency_value.*' => __('sales/customer_transfer.error_currency_value'),
            'payment_term_id.*' => __('sales/customer_transfer.error_payment_term_id'),
            'cfdi_use_id.*' => __('sales/customer_transfer.error_cfdi_use_id'),
            'cfdi_relation_id.*' => __('sales/customer_transfer.error_cfdi_relation_id'),
            'item.required' => __('sales/customer_transfer.error_item'),
            'item.*.name.*' => __('sales/customer_transfer.error_line_name'),
            'item.*.unit_measure_id.*' => __('sales/customer_transfer.error_line_unit_measure_id'),
            'item.*.sat_product_id.*' => __('sales/customer_transfer.error_line_sat_product_id'),
            'item.*.quantity.*' => __('sales/customer_transfer.error_line_quantity'),
            'item.*.price_unit.*' => __('sales/customer_transfer.error_line_price_unit'),
            'item_relation.*.uuid_related.*' => __('sales/customer_transfer.error_relation_uuid_related'),
            'item_transfer.required' => __('sales/customer_transfer.error_item_transfer'),
            'item_transfer.*.name.*' => __('sales/customer_transfer.error_line_transfer_name'),
            'source_taxid.regex' => __('sales/customer_transfer.error_source_taxid_format'),
            'destination_taxid.regex' => __('sales/customer_transfer.error_destination_taxid_format'),
            'operator_taxid.regex' => __('sales/customer_transfer.error_operator_taxid_format'),
            'owner_taxid.regex' => __('sales/customer_transfer.error_owner_taxid_format'),
            'lessee_taxid.regex' => __('sales/customer_transfer.error_lessee_taxid_format'),
        ]);

        //Validaciones manuales
        $validator = \Validator::make([], []);
        if(setting('cfdi_version') == 'cfdi40'){
            if (empty($request->tax_regimen_customer_id)) {
                $validator->after(function ($validator) {
                    $validator->errors()->add('tax_regimen_customer_id', __('sales/customer_transfer.error_tax_regimen_customer_id'));
                });
            }
            $customer = Customer::find($request->customer_id);
            if (empty($customer->postcode) && !in_array($customer->taxid, ['XAXX010101000', 'XEXX010101000'])) {
                $validator->after(function ($validator) {
                    $validator->errors()->add('customer_id', __('sales/customer_transfer.error_postcode_customer'));
                });
            }
        }

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        if($request->document_type2 == 'invoice'){
            $this->validate($request, [
                'date_due' => 'nullable|date|date_format:"'.setting('date_format', 'd-m-Y').'"|after_or_equal:' . Helper::date(Helper::createDateTime($request->date)),
                'payment_way_id' => 'required|integer',
                'payment_method_id' => 'required|integer',
            ], [
                'date_due.date' => __('sales/customer_transfer.error_date_due_format'),
                'date_due.date_format' => __('sales/customer_transfer.error_date_due_format'),
                'date_due.after_or_equal' => __('sales/customer_transfer.error_date_due_after'),
                'payment_way_id.*' => __('sales/customer_transfer.error_payment_way_id'),
                'payment_method_id.*' => __('sales/customer_transfer.error_payment_method_id'),
            ]);
        }

        if($request->document_type2 == 'transfer'){
            $validator = \Validator::make([], []);
            if(setting('cfdi_version') == 'cfdi40'){
                $cfdi_use =CfdiUse::where('id',$request->cfdi_use_id)->where('code','=','S01')->first();
                if (empty($cfdi_use)) {
                    $validator->after(function ($validator) {
                        $validator->errors()->add('cfdi_use_id', __('sales/customer_transfer.error_cfdi_use_id_transfer2'));
                    });
                }
            }else{
                $cfdi_use = CfdiUse::where('id',$request->cfdi_use_id)->where('code','P01')->first();
                if (empty($cfdi_use)) {
                    $validator->after(function ($validator) {
                        $validator->errors()->add('cfdi_use_id', __('sales/customer_transfer.error_cfdi_use_id_transfer'));
                    });
                }
            }
            if ($validator->fails()) {
                throw new ValidationException($validator);
            }
        }

        $validator = \Validator::make([], []);

        if($request->transport_int == 'No'){
            if(!empty($request->in_out_product)){
                $validator->after(function ($validator) {
                    $validator->errors()->add('in_out_product', __('sales/customer_transfer.error_in_out_product_no'));
                });
            }
            if(!empty($request->in_out_country_id)){
                $validator->after(function ($validator) {
                    $validator->errors()->add('in_out_country_id', __('sales/customer_transfer.error_in_out_country_id_no'));
                });
            }
            if(!empty($request->transport_type_id)){
                $validator->after(function ($validator) {
                    $validator->errors()->add('transport_type_id', __('sales/customer_transfer.error_transport_type_id_no'));
                });
            }
        }
        if($request->transport_int == 'Sí'){
            if(empty($request->in_out_product)){
                $validator->after(function ($validator) {
                    $validator->errors()->add('in_out_product', __('sales/customer_transfer.error_in_out_product_si'));
                });
            }
            if(empty($request->in_out_country_id)){
                $validator->after(function ($validator) {
                    $validator->errors()->add('in_out_country_id', __('sales/customer_transfer.error_in_out_country_id_si'));
                });
            }
            if(empty($request->transport_type_id)){
                $validator->after(function ($validator) {
                    $validator->errors()->add('transport_type_id', __('sales/customer_transfer.error_transport_type_id_si'));
                });
            }
        }

        $material_peligroso = false;
        if (!empty($request->item_transfer)) {
            foreach ($request->item_transfer as $key => $item_transfer) {
                if ($item_transfer['dangerous_product'] == 'Sí') {
                    $material_peligroso = true;
                    if(empty($item_transfer['packaging_type_id'])){
                        $validator->after(function ($validator) use ($key) {
                            $validator->errors()->add('item_transfer.'.$key.'.packaging_type_id', __('sales/customer_transfer.error_line_transfer_packaging_type_id'));
                        });
                    }
                    if(empty($item_transfer['dangerous_product_id'])){
                        $validator->after(function ($validator) use ($key) {
                            $validator->errors()->add('item_transfer.'.$key.'.dangerous_product_id', __('sales/customer_transfer.error_line_transfer_dangerous_product_id'));
                        });
                    }
                }
                if (empty($item_transfer['dangerous_product']) || $item_transfer['dangerous_product'] == 'No') {
                    if(!empty($item_transfer['packaging_type_id'])){
                        $validator->after(function ($validator) use ($key) {
                            $validator->errors()->add('item_transfer.'.$key.'.packaging_type_id', __('sales/customer_transfer.error_line_transfer_packaging_type_id_no'));
                        });
                    }
                    if(!empty($item_transfer['dangerous_product_id'])){
                        $validator->after(function ($validator) use ($key) {
                            $validator->errors()->add('item_transfer.'.$key.'.dangerous_product_id', __('sales/customer_transfer.error_line_transfer_dangerous_product_id_no'));
                        });
                    }
                }
            }
        }

        if($material_peligroso){
            if(empty($request->name_insurer2)){
                $validator->after(function ($validator) {
                    $validator->errors()->add('name_insurer2', __('sales/customer_transfer.error_name_insurer2'));
                });
            }
            if(empty($request->policy_number2)){
                $validator->after(function ($validator) {
                    $validator->errors()->add('policy_number2', __('sales/customer_transfer.error_policy_number2'));
                });
            }
        }

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }
    }

    /**
     * Crear XML y enviar a timbrar CFDI 3.3
     *
     * @param CustomerTransfer $customer_transfer
     * @return array|\CfdiUtils\Elements\Cfdi33\Concepto|float|int
     * @throws \Exception
     */
    private function cfdi33(CustomerTransfer $customer_transfer)
    {

        try {
            //Logica
            $pac = Pac::findOrFail(setting('default_pac_id')); //PAC

            //Arreglo CFDI 3.3
            $cfdi33 = [];
            if (!empty($customer_transfer->serie)) {
                $cfdi33['Serie'] = $customer_transfer->serie;
            }
            $cfdi33['Folio'] = $customer_transfer->folio;
            $cfdi33['Fecha'] = \Date::parse($customer_transfer->date)->format('Y-m-d\TH:i:s');
            //$cfdi33['Sello']
            if($customer_transfer->document_type2 == 'invoice'){
                $cfdi33['FormaPago'] = $customer_transfer->paymentWay->code;
            }
            $cfdi33['NoCertificado'] = $customer_transfer->company->certificate_number;
            //$cfdi33['Certificado']
            if($customer_transfer->document_type2 == 'invoice'){
                $cfdi33['CondicionesDePago'] = $customer_transfer->paymentTerm->name;
                $cfdi33['SubTotal'] = Helper::numberFormat($customer_transfer->amount_untaxed + $customer_transfer->amount_discount,
                    $customer_transfer->currency->decimal_place, false);
            }else {
                $cfdi33['SubTotal'] = Helper::numberFormat(0, 0, false);
            }
            if($customer_transfer->amount_discount>0) {
                $cfdi33['Descuento'] = Helper::numberFormat($customer_transfer->amount_discount, $customer_transfer->currency->decimal_place, false);
            }
            $cfdi33['Moneda'] = 'XXX';
            if($customer_transfer->document_type2 == 'invoice'){
                $cfdi33['Moneda'] = $customer_transfer->currency->code;
                if ($customer_transfer->currency->code != 'MXN') {
                    $cfdi33['TipoCambio'] = Helper::numberFormat($customer_transfer->currency_value, 4, false);
                }
            }
            if($customer_transfer->document_type2 == 'invoice'){
                $cfdi33['Total'] = Helper::numberFormat($customer_transfer->amount_total, $customer_transfer->currency->decimal_place, false);
            }else {
                $cfdi33['Total'] = Helper::numberFormat(0, 0, false);
            }

            $cfdi33['TipoDeComprobante'] = $customer_transfer->documentType->cfdiType->code;
            if($customer_transfer->document_type2 == 'invoice'){
                $cfdi33['MetodoPago'] = $customer_transfer->paymentMethod->code;
            }
            $cfdi33['LugarExpedicion'] = $customer_transfer->branchOffice->postcode;
            if (!empty($customer_transfer->confirmacion)) {
                $cfdi33['Confirmacion'] = $customer_transfer->confirmacion;
            }
            //---Cfdi Relacionados
            $cfdi33_relacionados = [];
            $cfdi33_relacionado = [];
            if (!empty($customer_transfer->cfdi_relation_id)) {
                $cfdi33_relacionados['TipoRelacion'] = $customer_transfer->cfdiRelation->code;
                if ($customer_transfer->customerInvoiceRelations->isNotEmpty()) {
                    foreach ($customer_transfer->customerInvoiceRelations as $key => $result) {
                        $cfdi33_relacionado[$key] = [];
                        $cfdi33_relacionado[$key]['UUID'] = $result->uuid_related;
                    }
                }
            }
            //---Emisor
            $cfdi33_emisor = [];
            $cfdi33_emisor['Rfc'] = $customer_transfer->company->taxid;
            $cfdi33_emisor['Nombre'] = trim($customer_transfer->company->name);
            $cfdi33_emisor['RegimenFiscal'] = !empty($customer_transfer->tax_regimen_id) ? $customer_transfer->taxRegimen->code : $customer_transfer->company->taxRegimen->code;
            //---Receptor
            if($customer_transfer->document_type2 == 'invoice'){
                $cfdi33_receptor = [];
                $cfdi33_receptor['Rfc'] = $customer_transfer->customer->taxid;
                $cfdi33_receptor['Nombre'] = trim($customer_transfer->customer->name);
                if ($customer_transfer->customer->taxid == 'XEXX010101000') {
                    $cfdi33_receptor['ResidenciaFiscal'] = $customer_transfer->customer->country->code;
                    $cfdi33_receptor['NumRegIdTrib'] = $customer_transfer->customer->numid;
                }
            }else{
                $cfdi33_receptor = [];
                $cfdi33_receptor['Rfc'] = $customer_transfer->company->taxid;
                $cfdi33_receptor['Nombre'] = trim($customer_transfer->company->name);
            }
            $cfdi33_receptor['UsoCFDI'] = $customer_transfer->cfdiUse->code;
            //---Conceptos
            $cfdi33_conceptos = [];
            $cfdi33_conceptos_traslados = [];
            $cfdi33_conceptos_retenciones = [];
            foreach ($customer_transfer->customerActiveInvoiceLines as $key => $result) {
                $cfdi33_conceptos [$key]['ClaveProdServ'] = $result->satProduct->code;
                if (!empty($result->product->code)) {
                    $cfdi33_conceptos [$key]['NoIdentificacion'] = trim($result->product->code);
                }
                $cfdi33_conceptos [$key]['Cantidad'] = Helper::numberFormat($result->quantity, 6, false);
                $cfdi33_conceptos [$key]['ClaveUnidad'] = trim($result->unitMeasure->code);
                $cfdi33_conceptos [$key]['Unidad'] = str_limit(trim($result->unitMeasure->name),20,'');
                $cfdi33_conceptos [$key]['Descripcion'] = trim($result->name);
                $cfdi33_conceptos [$key]['ValorUnitario'] = Helper::numberFormatRealDecimalPlace($result->price_unit, false);
                $cfdi33_conceptos [$key]['Importe'] = Helper::numberFormat($result->amount_untaxed + $result->amount_discount,
                    2, false);
                if($result->amount_discount>0) {
                    $cfdi33_conceptos [$key]['Descuento'] = Helper::numberFormat($result->amount_discount, 2, false);
                }
                //['InformacionAduanera']
                //['CuentaPredial']
                //['ComplementoConcepto']
                //['Parte']
                //Complemento

                //Impuestos por concepto
                $cfdi33_conceptos_traslados[$key] = [];
                $cfdi33_conceptos_retenciones[$key] = [];
                if ($result->taxes && $customer_transfer->document_type2 == 'invoice') {
                    foreach ($result->taxes as $key2 => $result2) {
                        $tmp = 0;
                        $rate = $result2->rate;
                        if ($result2->factor == 'Tasa') {
                            $rate /= 100;
                            $tmp = $result->amount_untaxed * $rate;
                        } elseif ($result2->factor == 'Cuota') {
                            $tmp = $rate;
                        }
                        $tmp = round($tmp, 2);
                        if ($result2->type == 'R') { //Retenciones
                            $cfdi33_conceptos_retenciones[$key][$key2] = [];
                            $cfdi33_conceptos_retenciones[$key][$key2]['Base'] = Helper::numberFormat($result->amount_untaxed,
                                2, false);
                            $cfdi33_conceptos_retenciones[$key][$key2]['Impuesto'] = $result2->code;
                            $cfdi33_conceptos_retenciones[$key][$key2]['TipoFactor'] = $result2->factor; //Para retenciones no hay exento
                            $cfdi33_conceptos_retenciones[$key][$key2]['TasaOCuota'] = Helper::numberFormat(abs($rate),
                                6, false);
                            $cfdi33_conceptos_retenciones[$key][$key2]['Importe'] = Helper::numberFormat(abs($tmp), 2,
                                false);
                        } else { //Traslados
                            $cfdi33_conceptos_traslados[$key][$key2] = [];
                            $cfdi33_conceptos_traslados[$key][$key2]['Base'] = Helper::numberFormat($result->amount_untaxed,
                                2, false);
                            $cfdi33_conceptos_traslados[$key][$key2]['Impuesto'] = $result2->code;
                            $cfdi33_conceptos_traslados[$key][$key2]['TipoFactor'] = $result2->factor; //Para retenciones no hay exento
                            if ($result2->factor != 'Exento') {
                                $cfdi33_conceptos_traslados[$key][$key2]['TasaOCuota'] = Helper::numberFormat(abs($rate),
                                    6, false);
                                $cfdi33_conceptos_traslados[$key][$key2]['Importe'] = Helper::numberFormat(abs($tmp), 2,
                                    false);
                            }
                        }
                    }
                }
            }
            //Impuestos
            $cfdi33_retenciones = [];
            $cfdi33_traslados = [];
            if ($customer_transfer->customerInvoiceTaxes->isNotEmpty() && $customer_transfer->document_type2 == 'invoice') {
                foreach ($customer_transfer->customerInvoiceTaxes as $key => $result) {
                    $tmp = $result->amount_tax;
                    $rate = $result->tax->rate;
                    if ($result->tax->factor == 'Tasa') {
                        $rate /= 100;
                    }
                    if ($result->tax->type == 'R') { //Retenciones
                        $cfdi33_retenciones[$key] = [];
                        $cfdi33_retenciones[$key]['Impuesto'] = $result->tax->code;
                        $cfdi33_retenciones[$key]['Importe'] = Helper::numberFormat(abs($tmp), $customer_transfer->currency->decimal_place, false);
                    } else { //Traslados
                        if ($result->tax->factor != 'Exento') {
                            $cfdi33_traslados[$key] = [];
                            $cfdi33_traslados[$key]['Impuesto'] = $result->tax->code;
                            $cfdi33_traslados[$key]['TipoFactor'] = $result->tax->factor;
                            $cfdi33_traslados[$key]['TasaOCuota'] = Helper::numberFormat(abs($rate), 6, false);
                            $cfdi33_traslados[$key]['Importe'] = Helper::numberFormat(abs($tmp), 2, false);
                        }
                    }
                }
            }
            $cfdi33_impuestos = [];
            if ((abs($customer_transfer->amount_tax_ret) > 0 || !empty($cfdi33_retenciones)) && $customer_transfer->document_type2 == 'invoice') {
                $cfdi33_impuestos['TotalImpuestosRetenidos'] = Helper::numberFormat(abs($customer_transfer->amount_tax_ret),
                    $customer_transfer->currency->decimal_place, false);
            }
            if ((abs($customer_transfer->amount_tax) > 0 || !empty($cfdi33_traslados)) && $customer_transfer->document_type2 == 'invoice') {
                $cfdi33_impuestos['TotalImpuestosTrasladados'] = Helper::numberFormat(abs($customer_transfer->amount_tax),
                    $customer_transfer->currency->decimal_place, false);
            }

            //Genera XML
            $certificado = new \CfdiUtils\Certificado\Certificado(\Storage::path($customer_transfer->company->pathFileCer()));
            $creator = new \CfdiUtils\CfdiCreator33($cfdi33, $certificado);
            $creator->setXmlResolver(PacHelper::resourcePathCfdiUtils()); //Almacenamiento local
            $comprobante = $creator->comprobante();

            $comprobante->addAttributes([
                'xsi:schemaLocation' => 'http://www.sat.gob.mx/cfd/3 http://www.sat.gob.mx/sitio_internet/cfd/3/cfdv33.xsd http://www.sat.gob.mx/CartaPorte20 http://www.sat.gob.mx/sitio_internet/cfd/CartaPorte/CartaPorte20.xsd',
            ]);
            $comprobante->addAttributes([
                'xmlns:cartaporte20' => 'http://www.sat.gob.mx/CartaPorte20'
            ]);

            if (!empty($cfdi33_relacionados)) {
                $comprobante->addCfdiRelacionados($cfdi33_relacionados);
            }
            if (!empty($cfdi33_relacionado)) {
                foreach ($cfdi33_relacionado as $key => $result) {
                    $comprobante->addCfdiRelacionado($result);
                }
            }
            $comprobante->addEmisor($cfdi33_emisor);
            $comprobante->addReceptor($cfdi33_receptor);
            //Conceptos
            foreach ($cfdi33_conceptos as $key => $result) {
                $concepto = $comprobante->addConcepto($result);
                if (!empty($cfdi33_conceptos_traslados[$key])) {
                    foreach ($cfdi33_conceptos_traslados[$key] as $result2) {
                        $concepto->multiTraslado($result2);
                    }
                }
                if (!empty($cfdi33_conceptos_retenciones[$key])) {
                    foreach ($cfdi33_conceptos_retenciones[$key] as $result2) {
                        $concepto->multiRetencion($result2);
                    }
                }
            }
            //Impuestos
            if(!empty($cfdi33_impuestos)) {
                $comprobante->addImpuestos($cfdi33_impuestos);
                if (!empty($cfdi33_retenciones)) {
                    foreach ($cfdi33_retenciones as $result2) {
                        $comprobante->multiRetencion($result2);
                    }
                }
                if (!empty($cfdi33_traslados)) {
                    foreach ($cfdi33_traslados as $result2) {
                        $comprobante->multiTraslado($result2);
                    }
                }
            }

            //
            $customer_transfer_transfer = $customer_transfer->customerInvoiceTransfer;

            //Complemento carta porte
            $carta_porte = [];
            $carta_porte['Version'] = '2.0';
            $carta_porte['TranspInternac'] = $customer_transfer_transfer->transport_int;
            $carta_porte['EntradaSalidaMerc'] = !empty($customer_transfer_transfer->in_out_product) ? $customer_transfer_transfer->in_out_product : null;
            $carta_porte['PaisOrigenDestino'] = !empty($customer_transfer_transfer->inOutCountry->code) ? $customer_transfer_transfer->inOutCountry->code : null;
            $carta_porte['ViaEntradaSalida'] = !empty($customer_transfer_transfer->transport_type_id) ? $customer_transfer_transfer->transportType->code : null;
            $carta_porte['TotalDistRec'] = !empty($customer_transfer_transfer->distance_traveled) ? $customer_transfer_transfer->distance_traveled : null;
            $nodo_carta_porte = new \CfdiUtils\Nodes\Node(
                'cartaporte20:CartaPorte', // nombre del elemento raíz
                $carta_porte
            );

            //Ubicaciones
            $ubicaciones = [];
            $nodo_ubicaciones = new \CfdiUtils\Nodes\Node(
                'cartaporte20:Ubicaciones', // nombre del elemento raíz
                $ubicaciones
            );
            $nodo_carta_porte->addChild($nodo_ubicaciones);
            //Origen
            $ubicacion_origen = [];
            $ubicacion_origen['TipoUbicacion'] = 'Origen';
            $ubicacion_origen['IDUbicacion'] = !empty($customer_transfer_transfer->source_id) ? $customer_transfer_transfer->source_id : null;
            $ubicacion_origen['RFCRemitenteDestinatario'] = !empty($customer_transfer_transfer->source_taxid) ? $customer_transfer_transfer->source_taxid : null;
            $ubicacion_origen['NombreRemitenteDestinatario'] = !empty($customer_transfer_transfer->source_name) ? $customer_transfer_transfer->source_name : null;
            $ubicacion_origen['NumRegIdTrib'] = !empty($customer_transfer_transfer->source_numid) ? $customer_transfer_transfer->source_numid : null;
            $ubicacion_origen['ResidenciaFiscal'] = !empty($customer_transfer_transfer->sourceTrCountry->code) ? $customer_transfer_transfer->sourceTrCountry->code : null;
            $ubicacion_origen['NumEstacion'] = !empty($customer_transfer_transfer->sourceStation->code) ? $customer_transfer_transfer->sourceStation->code : null;
            $ubicacion_origen['NombreEstacion'] = !empty($customer_transfer_transfer->source_station) ? $customer_transfer_transfer->source_station : (!empty($customer_transfer_transfer->sourceStation->name) ? $customer_transfer_transfer->sourceStation->name : null);
            $ubicacion_origen['NavegacionTrafico'] = !empty($customer_transfer_transfer->source_traffic_navigation) ? $customer_transfer_transfer->source_traffic_navigation : null;
            $ubicacion_origen['FechaHoraSalidaLlegada'] = !empty($customer_transfer_transfer->source_date) ? \Date::parse($customer_transfer_transfer->source_date)->format('Y-m-d\TH:i:s') : null;
            $ubicacion_origen['TipoEstacion'] = !empty($customer_transfer_transfer->sourceStationType->code) ? $customer_transfer_transfer->sourceStationType->code : null;
            $ubicacion_origen['DistanciaRecorrida'] = !empty($customer_transfer_transfer->source_distance_traveled) ? $customer_transfer_transfer->source_distance_traveled : null;
            $nodo_ubicacion_origen = new \CfdiUtils\Nodes\Node(
                'cartaporte20:Ubicacion', // nombre del elemento raíz
                $ubicacion_origen
            );
            $nodo_ubicaciones->addChild($nodo_ubicacion_origen);
            $ubicacion_origen_domicilio = [];
            $ubicacion_origen_domicilio['Calle'] = !empty($customer_transfer_transfer->source_address_1) ? $customer_transfer_transfer->source_address_1 : null;
            $ubicacion_origen_domicilio['NumeroExterior'] = !empty($customer_transfer_transfer->source_address_2) ? $customer_transfer_transfer->source_address_2 : null;
            $ubicacion_origen_domicilio['NumeroInterior'] = !empty($customer_transfer_transfer->source_address_3) ? $customer_transfer_transfer->source_address_3 : null;
            $ubicacion_origen_domicilio['Colonia'] = null;
            $ubicacion_origen_domicilio['Localidad'] = null;
            $ubicacion_origen_domicilio['Referencia'] = !empty($customer_transfer_transfer->source_address_6) ? $customer_transfer_transfer->source_address_6 : null;
            $ubicacion_origen_domicilio['Municipio'] = !empty($customer_transfer_transfer->sourceCity->code) ? $customer_transfer_transfer->sourceCity->code : null;
            $ubicacion_origen_domicilio['Estado'] = !empty($customer_transfer_transfer->sourceState->code) ? $customer_transfer_transfer->sourceState->code : null;
            $ubicacion_origen_domicilio['Pais'] = !empty($customer_transfer_transfer->sourceCountry->code) ? $customer_transfer_transfer->sourceCountry->code : null;
            $ubicacion_origen_domicilio['CodigoPostal'] = !empty($customer_transfer_transfer->source_postcode) ? $customer_transfer_transfer->source_postcode : null;
            $nodoubicacion__origen_domicilio = new \CfdiUtils\Nodes\Node(
                'cartaporte20:Domicilio', // nombre del elemento raíz
                $ubicacion_origen_domicilio
            );
            $nodo_ubicacion_origen->addChild($nodoubicacion__origen_domicilio);
            //Destino
            $ubicacion_destino = [];
            $ubicacion_destino['TipoUbicacion'] = 'Destino';
            $ubicacion_destino['IDUbicacion'] = !empty($customer_transfer_transfer->destination_id) ? $customer_transfer_transfer->destination_id : null;
            $ubicacion_destino['RFCRemitenteDestinatario'] = !empty($customer_transfer_transfer->destination_taxid) ? $customer_transfer_transfer->destination_taxid : null;
            $ubicacion_destino['NombreRemitenteDestinatario'] = !empty($customer_transfer_transfer->destination_name) ? $customer_transfer_transfer->destination_name : null;
            $ubicacion_destino['NumRegIdTrib'] = !empty($customer_transfer_transfer->destination_numid) ? $customer_transfer_transfer->destination_numid : null;
            $ubicacion_destino['ResidenciaFiscal'] = !empty($customer_transfer_transfer->destinationTrCountry->code) ? $customer_transfer_transfer->destinationTrCountry->code : null;
            $ubicacion_destino['NumEstacion'] = !empty($customer_transfer_transfer->destinationStation->code) ? $customer_transfer_transfer->destinationStation->code : null;
            $ubicacion_destino['NombreEstacion'] = !empty($customer_transfer_transfer->destination_station) ? $customer_transfer_transfer->destination_station : (!empty($customer_transfer_transfer->destinationStation->name) ? $customer_transfer_transfer->destinationStation->name : null);
            $ubicacion_destino['NavegacionTrafico'] = !empty($customer_transfer_transfer->destination_traffic_navigation) ? $customer_transfer_transfer->destination_traffic_navigation : null;
            $ubicacion_destino['FechaHoraSalidaLlegada'] = !empty($customer_transfer_transfer->destination_date) ? \Date::parse($customer_transfer_transfer->destination_date)->format('Y-m-d\TH:i:s') : null;
            $ubicacion_destino['TipoEstacion'] = !empty($customer_transfer_transfer->destinationStationType->code) ? $customer_transfer_transfer->destinationStationType->code : null;
            $ubicacion_destino['DistanciaRecorrida'] = !empty($customer_transfer_transfer->destination_distance_traveled) ? $customer_transfer_transfer->destination_distance_traveled : null;
            $nodo_ubicacion_destino = new \CfdiUtils\Nodes\Node(
                'cartaporte20:Ubicacion', // nombre del elemento raíz
                $ubicacion_destino
            );
            $nodo_ubicaciones->addChild($nodo_ubicacion_destino);
            $ubicacion_destino_domicilio = [];
            $ubicacion_destino_domicilio['Calle'] = !empty($customer_transfer_transfer->destination_address_1) ? $customer_transfer_transfer->destination_address_1 : null;
            $ubicacion_destino_domicilio['NumeroExterior'] = !empty($customer_transfer_transfer->destination_address_2) ? $customer_transfer_transfer->destination_address_2 : null;
            $ubicacion_destino_domicilio['NumeroInterior'] = !empty($customer_transfer_transfer->destination_address_3) ? $customer_transfer_transfer->destination_address_3 : null;
            $ubicacion_destino_domicilio['Colonia'] = null;
            $ubicacion_destino_domicilio['Localidad'] = null;
            $ubicacion_destino_domicilio['Referencia'] = !empty($customer_transfer_transfer->destination_address_6) ? $customer_transfer_transfer->destination_address_6 : null;
            $ubicacion_destino_domicilio['Municipio'] = !empty($customer_transfer_transfer->destinationCity->code) ? $customer_transfer_transfer->destinationCity->code : null;
            $ubicacion_destino_domicilio['Estado'] = !empty($customer_transfer_transfer->destinationState->code) ? $customer_transfer_transfer->destinationState->code : null;
            $ubicacion_destino_domicilio['Pais'] = !empty($customer_transfer_transfer->destinationCountry->code) ? $customer_transfer_transfer->destinationCountry->code : null;
            $ubicacion_destino_domicilio['CodigoPostal'] = !empty($customer_transfer_transfer->destination_postcode) ? $customer_transfer_transfer->destination_postcode : null;
            $nodo_ubicacion_destino_domicilio = new \CfdiUtils\Nodes\Node(
                'cartaporte20:Domicilio', // nombre del elemento raíz
                $ubicacion_destino_domicilio
            );
            $nodo_ubicacion_destino->addChild($nodo_ubicacion_destino_domicilio);

            //Mercancias
            $mercancias = [];
            $mercancias['PesoBrutoTotal'] = !empty($customer_transfer_transfer->gross_weight) ? $customer_transfer_transfer->gross_weight : null;
            $mercancias['UnidadPeso'] = !empty($customer_transfer_transfer->measurePackaging->code) ? $customer_transfer_transfer->measurePackaging->code : null;
            $mercancias['PesoNetoTotal'] = !empty($customer_transfer_transfer->net_weight) ? $customer_transfer_transfer->net_weight : null;
            $mercancias['NumTotalMercancias'] = $customer_transfer->customerActiveInvoiceLineTransfers->count();
            $mercancias['CargoPorTasacion'] = (!empty($customer_transfer_transfer->appraisal_fee) && $customer_transfer_transfer->appraisal_fee > 0) ? Helper::numberFormat($customer_transfer_transfer->appraisal_fee, 2, false) : null;
            $nodo_mercancias = new \CfdiUtils\Nodes\Node(
                'cartaporte20:Mercancias', // nombre del elemento raíz
                $mercancias
            );
            $nodo_carta_porte->addChild($nodo_mercancias);

            foreach ($customer_transfer->customerActiveInvoiceLineTransfers as $key => $result) {
                $mercancia = [];
                $mercancia['BienesTransp'] = !empty($result->satProduct->code) ? $result->satProduct->code : null;
                $mercancia['Descripcion'] = !empty($result->name) ? $result->name : null;
                $mercancia['Cantidad'] = !empty($result->quantity) ? round($result->quantity,6) : null;
                $mercancia['ClaveUnidad'] = !empty($result->unitMeasure->code) ? $result->unitMeasure->code : null;
                $mercancia['Unidad'] = null;
                $mercancia['Dimensiones'] = !empty($result->dimensions) ? $result->dimensions : null;
                $mercancia['MaterialPeligroso'] = !empty($result->dangerous_product) ? $result->dangerous_product : null;
                $mercancia['CveMaterialPeligroso'] = !empty($result->dangerousProduct->code) ? $result->dangerousProduct->code : null;
                $mercancia['Embalaje'] = !empty($result->packagingType->code) ? $result->packagingType->code : null;
                $mercancia['DescripEmbalaje'] = !empty($result->packagingType->name) ? $result->packagingType->name : null;
                $mercancia['PesoEnKg'] = !empty($result->weight) ? round($result->weight,6) : null;
                $mercancia['ValorMercancia'] = !empty($result->amount_total) ? round($result->amount_total, 2) : null;
                $mercancia['Moneda'] = !empty($result->currency->code) ? $result->currency->code : null;
                $mercancia['FraccionArancelaria'] = !empty($result->tariffFraction->code) ? $result->tariffFraction->code : null;
                $mercancia['UUIDComercioExt'] = !empty($result->uuid_ext) ? $result->uuid_ext : null;
                $nodo_mercancia = new \CfdiUtils\Nodes\Node(
                    'cartaporte20:Mercancia', // nombre del elemento raíz
                    $mercancia
                );
                $nodo_mercancias->addChild($nodo_mercancia);
            }

            if(!empty($customer_transfer_transfer->trans_transport_type_id) && $customer_transfer_transfer->transTransportType->code == '01'){
                $autotransporte = [];
                $autotransporte['PermSCT'] = !empty($customer_transfer_transfer->permitType->code) ? $customer_transfer_transfer->permitType->code : null;
                $autotransporte['NumPermisoSCT'] = !empty($customer_transfer_transfer->permit_number) ? $customer_transfer_transfer->permit_number : null;
                $nodo_autotransporte = new \CfdiUtils\Nodes\Node(
                    'cartaporte20:Autotransporte', // nombre del elemento raíz
                    $autotransporte
                );
                $nodo_mercancias->addChild($nodo_autotransporte);

                $autotransporte_iden_veh = [];
                $autotransporte_iden_veh['ConfigVehicular'] = !empty($customer_transfer_transfer->settingTransport->code) ? $customer_transfer_transfer->settingTransport->code : null;
                $autotransporte_iden_veh['PlacaVM'] = !empty($customer_transfer_transfer->vehicle_number) ? $customer_transfer_transfer->vehicle_number : null;
                $autotransporte_iden_veh['AnioModeloVM'] = !empty($customer_transfer_transfer->vehicle) ? $customer_transfer_transfer->vehicle : null;
                $nodo_autotransporte_iden_veh = new \CfdiUtils\Nodes\Node(
                    'cartaporte20:IdentificacionVehicular', // nombre del elemento raíz
                    $autotransporte_iden_veh
                );
                $nodo_autotransporte->addChild($nodo_autotransporte_iden_veh);

                $autotransporte_seguros = [];
                $autotransporte_seguros['AseguraRespCivil'] = !empty($customer_transfer_transfer->name_insurer) ? $customer_transfer_transfer->name_insurer : null;
                $autotransporte_seguros['PolizaRespCivil'] = !empty($customer_transfer_transfer->policy_number) ? $customer_transfer_transfer->policy_number : null;
                $autotransporte_seguros['AseguraMedAmbiente'] = !empty($customer_transfer_transfer->name_insurer2) ? $customer_transfer_transfer->name_insurer2 : null;
                $autotransporte_seguros['PolizaMedAmbiente'] = !empty($customer_transfer_transfer->policy_number2) ? $customer_transfer_transfer->policy_number2 : null;
                $autotransporte_seguros['AseguraCarga'] = !empty($customer_transfer_transfer->name_insurer3) ? $customer_transfer_transfer->name_insurer3 : null;
                $autotransporte_seguros['PolizaCarga'] = !empty($customer_transfer_transfer->policy_number3) ? $customer_transfer_transfer->policy_number3 : null;
                $autotransporte_seguros['PrimaSeguro'] = !empty($customer_transfer_transfer->insurance_value) ? $customer_transfer_transfer->insurance_value : null;
                $nodo_autotransporte_seguros = new \CfdiUtils\Nodes\Node(
                    'cartaporte20:Seguros', // nombre del elemento raíz
                    $autotransporte_seguros
                );
                $nodo_autotransporte->addChild($nodo_autotransporte_seguros);

                if(!empty($customer_transfer_transfer->trailer_type_id_1) || !empty($customer_transfer_transfer->trailer_type_id_2) || !empty($customer_transfer_transfer->trailer_type_id_3) || !empty($customer_transfer_transfer->trailer_type_id_4) || !empty($customer_transfer_transfer->trailer_type_id_5)){
                    $nodo_autotransporte_remolques = new \CfdiUtils\Nodes\Node(
                        'cartaporte20:Remolques', // nombre del elemento raíz
                        []
                    );
                    $nodo_autotransporte->addChild($nodo_autotransporte_remolques);
                    if(!empty($customer_transfer_transfer->trailer_type_id_1)){
                        $autotransporte_remolque = [];
                        $autotransporte_remolque['Placa'] = !empty($customer_transfer_transfer->trailer_number_1) ? $customer_transfer_transfer->trailer_number_1 : null;
                        $autotransporte_remolque['SubTipoRem'] = !empty($customer_transfer_transfer->trailerType1->code) ? $customer_transfer_transfer->trailerType1->code : null;
                        $nodo_autotransporte_remolque = new \CfdiUtils\Nodes\Node(
                            'cartaporte20:Remolque', // nombre del elemento raíz
                            $autotransporte_remolque
                        );
                        $nodo_autotransporte_remolques->addChild($nodo_autotransporte_remolque);
                    }
                    if(!empty($customer_transfer_transfer->trailer_type_id_2)){
                        $autotransporte_remolque = [];
                        $autotransporte_remolque['Placa'] = !empty($customer_transfer_transfer->trailer_number_2) ? $customer_transfer_transfer->trailer_number_2 : null;
                        $autotransporte_remolque['SubTipoRem'] = !empty($customer_transfer_transfer->trailerType2->code) ? $customer_transfer_transfer->trailerType2->code : null;
                        $nodo_autotransporte_remolque = new \CfdiUtils\Nodes\Node(
                            'cartaporte20:Remolque', // nombre del elemento raíz
                            $autotransporte_remolque
                        );
                        $nodo_autotransporte_remolques->addChild($nodo_autotransporte_remolque);
                    }
                    if(!empty($customer_transfer_transfer->trailer_type_id_3)){
                        $autotransporte_remolque = [];
                        $autotransporte_remolque['Placa'] = !empty($customer_transfer_transfer->trailer_number_3) ? $customer_transfer_transfer->trailer_number_3 : null;
                        $autotransporte_remolque['SubTipoRem'] = !empty($customer_transfer_transfer->trailerType3->code) ? $customer_transfer_transfer->trailerType3->code : null;
                        $nodo_autotransporte_remolque = new \CfdiUtils\Nodes\Node(
                            'cartaporte20:Remolque', // nombre del elemento raíz
                            $autotransporte_remolque
                        );
                        $nodo_autotransporte_remolques->addChild($nodo_autotransporte_remolque);
                    }
                    if(!empty($customer_transfer_transfer->trailer_type_id_4)){
                        $autotransporte_remolque = [];
                        $autotransporte_remolque['Placa'] = !empty($customer_transfer_transfer->trailer_number_4) ? $customer_transfer_transfer->trailer_number_4 : null;
                        $autotransporte_remolque['SubTipoRem'] = !empty($customer_transfer_transfer->trailerType4->code) ? $customer_transfer_transfer->trailerType4->code : null;
                        $nodo_autotransporte_remolque = new \CfdiUtils\Nodes\Node(
                            'cartaporte20:Remolque', // nombre del elemento raíz
                            $autotransporte_remolque
                        );
                        $nodo_autotransporte_remolques->addChild($nodo_autotransporte_remolque);
                    }
                    if(!empty($customer_transfer_transfer->trailer_type_id_5)){
                        $autotransporte_remolque = [];
                        $autotransporte_remolque['Placa'] = !empty($customer_transfer_transfer->trailer_number_5) ? $customer_transfer_transfer->trailer_number_5 : null;
                        $autotransporte_remolque['SubTipoRem'] = !empty($customer_transfer_transfer->trailerType5->code) ? $customer_transfer_transfer->trailerType5->code : null;
                        $nodo_autotransporte_remolque = new \CfdiUtils\Nodes\Node(
                            'cartaporte20:Remolque', // nombre del elemento raíz
                            $autotransporte_remolque
                        );
                        $nodo_autotransporte_remolques->addChild($nodo_autotransporte_remolque);
                    }
                }
            }

            //FiguraTransporte
            if(!empty($customer_transfer_transfer->operator_name) || !empty($customer_transfer_transfer->operator_taxid) || !empty($customer_transfer_transfer->owner_name) || !empty($customer_transfer_transfer->owner_taxid) || !empty($customer_transfer_transfer->lessee_name) || !empty($customer_transfer_transfer->lessee_taxid) || !empty($customer_transfer_transfer->notified_name) || !empty($customer_transfer_transfer->notified_taxid)){
                $figura_transporte = [];
                $nodo_figura_transporte = new \CfdiUtils\Nodes\Node(
                    'cartaporte20:FiguraTransporte', // nombre del elemento raíz
                    $figura_transporte
                );
                $nodo_carta_porte->addChild($nodo_figura_transporte);

                //Operadores
                if(!empty($customer_transfer_transfer->operator_name) || !empty($customer_transfer_transfer->operator_taxid)){
                    //Operador
                    $operador = [];
                    $operador['TipoFigura'] = '01';
                    $operador['RFCFigura'] = !empty($customer_transfer_transfer->operator_taxid) ? $customer_transfer_transfer->operator_taxid : null;
                    $operador['NumLicencia'] = !empty($customer_transfer_transfer->operator_license) ? $customer_transfer_transfer->operator_license : null;
                    $operador['NombreFigura'] = !empty($customer_transfer_transfer->operator_name) ? $customer_transfer_transfer->operator_name : null;
                    $operador['NumRegIdTribFigura'] = !empty($customer_transfer_transfer->operator_numid) ? $customer_transfer_transfer->operator_numid : null;
                    $operador['ResidenciaFiscalFigura'] = !empty($customer_transfer_transfer->operatorTrCountry->code) ? $customer_transfer_transfer->operatorTrCountry->code : null;
                    $nodo_operador = new \CfdiUtils\Nodes\Node(
                        'cartaporte20:TiposFigura', // nombre del elemento raíz
                        $operador
                    );
                    $nodo_figura_transporte->addChild($nodo_operador);
                }

                //Propietario
                if(!empty($customer_transfer_transfer->owner_name) || !empty($customer_transfer_transfer->owner_taxid)){
                    //Propietario
                    $propietario = [];
                    $propietario['TipoFigura'] = '02';
                    $propietario['RFCFigura'] = !empty($customer_transfer_transfer->owner_taxid) ? $customer_transfer_transfer->owner_taxid : null;
                    $propietario['NumLicencia'] = null;
                    $propietario['NombreFigura'] = !empty($customer_transfer_transfer->owner_name) ? $customer_transfer_transfer->owner_name : null;
                    $propietario['NumRegIdTribFigura'] = !empty($customer_transfer_transfer->owner_numid) ? $customer_transfer_transfer->owner_numid : null;
                    $propietario['ResidenciaFiscalFigura'] = !empty($customer_transfer_transfer->ownerTrCountry->code) ? $customer_transfer_transfer->ownerTrCountry->code : null;
                    $nodo_propietario = new \CfdiUtils\Nodes\Node(
                        'cartaporte20:TiposFigura', // nombre del elemento raíz
                        $propietario
                    );
                    $nodo_figura_transporte->addChild($nodo_propietario);
                    if(!empty($customer_transfer_transfer->owner_transport_part_id)){
                        $propietario_partes = [];
                        $propietario_partes['ParteTransporte'] = !empty($customer_transfer_transfer->ownerTransportPart->code) ? $customer_transfer_transfer->ownerTransportPart->code : null;
                        $nodo_propietario_partes = new \CfdiUtils\Nodes\Node(
                            'cartaporte20:PartesTransporte', // nombre del elemento raíz
                            $propietario_partes
                        );
                        $nodo_propietario->addChild($nodo_propietario_partes);
                        if(!empty($customer_transfer_transfer->owner_postcode)){
                            $propietario_domicilio = [];
                            $propietario_domicilio['Calle'] = !empty($customer_transfer_transfer->owner_address_1) ? $customer_transfer_transfer->owner_address_1 : null;
                            $propietario_domicilio['NumeroExterior'] = !empty($customer_transfer_transfer->owner_address_2) ? $customer_transfer_transfer->owner_address_2 : null;
                            $propietario_domicilio['NumeroInterior'] = !empty($customer_transfer_transfer->owner_address_3) ? $customer_transfer_transfer->owner_address_3 : null;
                            $propietario_domicilio['Colonia'] = null;
                            $propietario_domicilio['Localidad'] = null;
                            $propietario_domicilio['Referencia'] = !empty($customer_transfer_transfer->owner_address_6) ? $customer_transfer_transfer->owner_address_6 : null;
                            $propietario_domicilio['Municipio'] = !empty($customer_transfer_transfer->ownerCity->code) ? $customer_transfer_transfer->ownerCity->code : null;
                            $propietario_domicilio['Estado'] = !empty($customer_transfer_transfer->ownerState->code) ? $customer_transfer_transfer->ownerState->code : null;
                            $propietario_domicilio['Pais'] = !empty($customer_transfer_transfer->ownerCountry->code) ? $customer_transfer_transfer->ownerCountry->code : null;
                            $propietario_domicilio['CodigoPostal'] = !empty($customer_transfer_transfer->owner_postcode) ? $customer_transfer_transfer->owner_postcode : null;
                            $nodo_propietario_domicilio = new \CfdiUtils\Nodes\Node(
                                'cartaporte20:Domicilio', // nombre del elemento raíz
                                $propietario_domicilio
                            );
                            $nodo_propietario->addChild($nodo_propietario_domicilio);
                        }
                    }
                }

                //Arrendatario
                if(!empty($customer_transfer_transfer->lessee_name) || !empty($customer_transfer_transfer->lessee_taxid)){
                    //Arrendatario
                    $arrendatario = [];
                    $arrendatario['TipoFigura'] = '03';
                    $arrendatario['RFCFigura'] = !empty($customer_transfer_transfer->lessee_taxid) ? $customer_transfer_transfer->lessee_taxid : null;
                    $arrendatario['NumLicencia'] = null;
                    $arrendatario['NombreFigura'] = !empty($customer_transfer_transfer->lessee_name) ? $customer_transfer_transfer->lessee_name : null;
                    $arrendatario['NumRegIdTribFigura'] = !empty($customer_transfer_transfer->lessee_numid) ? $customer_transfer_transfer->lessee_numid : null;
                    $arrendatario['ResidenciaFiscalFigura'] = !empty($customer_transfer_transfer->lesseeTrCountry->code) ? $customer_transfer_transfer->lesseeTrCountry->code : null;
                    $nodo_arrendatario = new \CfdiUtils\Nodes\Node(
                        'cartaporte20:TiposFigura', // nombre del elemento raíz
                        $arrendatario
                    );
                    $nodo_figura_transporte->addChild($nodo_arrendatario);
                    if(!empty($customer_transfer_transfer->lessee_transport_part_id)){
                        $arrendatario_partes = [];
                        $arrendatario_partes['ParteTransporte'] = !empty($customer_transfer_transfer->lesseeTransportPart->code) ? $customer_transfer_transfer->lesseeTransportPart->code : null;
                        $nodo_arrendatario_partes = new \CfdiUtils\Nodes\Node(
                            'cartaporte20:PartesTransporte', // nombre del elemento raíz
                            $arrendatario_partes
                        );
                        $nodo_arrendatario->addChild($nodo_arrendatario_partes);
                        if(!empty($customer_transfer_transfer->lessee_postcode)){
                            $arrendatario_domicilio = [];
                            $arrendatario_domicilio['Calle'] = !empty($customer_transfer_transfer->lessee_address_1) ? $customer_transfer_transfer->lessee_address_1 : null;
                            $arrendatario_domicilio['NumeroExterior'] = !empty($customer_transfer_transfer->lessee_address_2) ? $customer_transfer_transfer->lessee_address_2 : null;
                            $arrendatario_domicilio['NumeroInterior'] = !empty($customer_transfer_transfer->lessee_address_3) ? $customer_transfer_transfer->lessee_address_3 : null;
                            $arrendatario_domicilio['Colonia'] = null;
                            $arrendatario_domicilio['Localidad'] = null;
                            $arrendatario_domicilio['Referencia'] = !empty($customer_transfer_transfer->lessee_address_6) ? $customer_transfer_transfer->lessee_address_6 : null;
                            $arrendatario_domicilio['Municipio'] = !empty($customer_transfer_transfer->lesseeCity->code) ? $customer_transfer_transfer->lesseeCity->code : null;
                            $arrendatario_domicilio['Estado'] = !empty($customer_transfer_transfer->lesseeState->code) ? $customer_transfer_transfer->lesseeState->code : null;
                            $arrendatario_domicilio['Pais'] = !empty($customer_transfer_transfer->lesseeCountry->code) ? $customer_transfer_transfer->lesseeCountry->code : null;
                            $arrendatario_domicilio['CodigoPostal'] = !empty($customer_transfer_transfer->lessee_postcode) ? $customer_transfer_transfer->lessee_postcode : null;
                            $nodo_arrendatario_domicilio = new \CfdiUtils\Nodes\Node(
                                'cartaporte20:Domicilio', // nombre del elemento raíz
                                $arrendatario_domicilio
                            );
                            $nodo_arrendatario->addChild($nodo_arrendatario_domicilio);
                        }
                    }
                }

                //Notificado
                if(!empty($customer_transfer_transfer->notified_name) || !empty($customer_transfer_transfer->notified_taxid)){
                    //Notificado
                    $notificado = [];
                    $notificado['TipoFigura'] = '04';
                    $notificado['RFCFigura'] = !empty($customer_transfer_transfer->notified_taxid) ? $customer_transfer_transfer->notified_taxid : null;
                    $notificado['NumLicencia'] = null;
                    $notificado['NombreFigura'] = !empty($customer_transfer_transfer->notified_name) ? $customer_transfer_transfer->notified_name : null;
                    $notificado['NumRegIdTribFigura'] = !empty($customer_transfer_transfer->notified_numid) ? $customer_transfer_transfer->notified_numid : null;
                    $notificado['ResidenciaFiscalFigura'] = !empty($customer_transfer_transfer->notifiedTrCountry->code) ? $customer_transfer_transfer->notifiedTrCountry->code : null;
                    $nodo_notificado = new \CfdiUtils\Nodes\Node(
                        'cartaporte20:TiposFigura', // nombre del elemento raíz
                        $notificado
                    );
                    $nodo_figura_transporte->addChild($nodo_notificado);
                }
            }

            //Agregamos al complemento
            $comprobante->addComplemento($nodo_carta_porte);

            //Método de ayuda para establecer las sumas del comprobante e impuestos con base en la suma de los conceptos y la agrupación de sus impuestos
            //$creator->addSumasConceptos(null, 2);
            //Método de ayuda para generar el sello (obtener la cadena de origen y firmar con la llave privada)
            $creator->addSello('file://' . \Storage::path($customer_transfer->company->pathFileKeyPassPem()), Crypt::decryptString($customer_transfer->company->password_key));
            //Valida la estructura
            //$creator->validate();

            //Guarda XML
            //dd($creator->asXml());
            $path_xml = Helper::setDirectory(CustomerTransfer::PATH_XML_FILES_CTR, $customer_transfer->company_id) . '/';
            if($customer_transfer->document_type2 == 'invoice'){
                $path_xml = Helper::setDirectory(CustomerTransfer::PATH_XML_FILES_CI, $customer_transfer->company_id) . '/';
            }
            $file_xml = Helper::makeDirectoryCfdi($path_xml) . '/' . Str::random(40) . '.xml';
            $creator->saveXml(\Storage::path($path_xml . $file_xml));


            //Arreglo temporal para actualizar Customer Transfer CFDI
            $tmp = [
                'pac_id' => $pac->id,
                'cfdi_version' => setting('cfdi_version'),
                'uuid' => '',
                'date' => '',
                'path_xml' => $path_xml,
                'file_xml' => $file_xml,
                'file_xml_pac' => '',
                'pac' => $pac,
            ];

            //Timbrado de XML
            $class_pac = $pac->code;
            $tmp = PacHelper::$class_pac($tmp, $creator);

            return $tmp;
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Crear XML y enviar a timbrar CFDI 4.0
     *
     * @param CustomerTransfer $customer_transfer
     * @return array|\CfdiUtils\Elements\Cfdi33\Concepto|float|int
     * @throws \Exception
     */
    private function cfdi40(CustomerTransfer $customer_transfer)
    {

        try {
            //Logica
            $pac = Pac::findOrFail(setting('default_pac_id')); //PAC

            //Arreglo CFDI 3.3
            $cfdi33 = [];
            if (!empty($customer_transfer->serie)) {
                $cfdi33['Serie'] = $customer_transfer->serie;
            }
            $cfdi33['Folio'] = $customer_transfer->folio;
            $cfdi33['Fecha'] = \Date::parse($customer_transfer->date)->format('Y-m-d\TH:i:s');
            //$cfdi33['Sello']
            if($customer_transfer->document_type2 == 'invoice'){
                $cfdi33['FormaPago'] = $customer_transfer->paymentWay->code;
            }
            $cfdi33['NoCertificado'] = $customer_transfer->company->certificate_number;
            //$cfdi33['Certificado']
            if($customer_transfer->document_type2 == 'invoice'){
                $cfdi33['CondicionesDePago'] = $customer_transfer->paymentTerm->name;
                $cfdi33['SubTotal'] = Helper::numberFormat($customer_transfer->amount_untaxed + $customer_transfer->amount_discount,
                    $customer_transfer->currency->decimal_place, false);
            }else {
                $cfdi33['SubTotal'] = Helper::numberFormat(0, 0, false);
            }
            if($customer_transfer->amount_discount>0) {
                $cfdi33['Descuento'] = Helper::numberFormat($customer_transfer->amount_discount, $customer_transfer->currency->decimal_place, false);
            }
            $cfdi33['Moneda'] = 'XXX';
            if($customer_transfer->document_type2 == 'invoice'){
                $cfdi33['Moneda'] = $customer_transfer->currency->code;
                if ($customer_transfer->currency->code != 'MXN') {
                    $cfdi33['TipoCambio'] = Helper::numberFormat($customer_transfer->currency_value, 4, false);
                }
            }
            if($customer_transfer->document_type2 == 'invoice'){
                $cfdi33['Total'] = Helper::numberFormat($customer_transfer->amount_total, $customer_transfer->currency->decimal_place, false);
            }else {
                $cfdi33['Total'] = Helper::numberFormat(0, 0, false);
            }

            $cfdi33['TipoDeComprobante'] = $customer_transfer->documentType->cfdiType->code;
            $cfdi33['Exportacion'] = '01';
            if($customer_transfer->document_type2 == 'invoice'){
                $cfdi33['MetodoPago'] = $customer_transfer->paymentMethod->code;
            }
            $cfdi33['LugarExpedicion'] = $customer_transfer->branchOffice->postcode;
            if (!empty($customer_transfer->confirmacion)) {
                $cfdi33['Confirmacion'] = $customer_transfer->confirmacion;
            }
            //---Informacion global
            //---Cfdi Relacionados
            $cfdi33_relacionados = [];
            $cfdi33_relacionado = [];
            if (!empty($customer_transfer->cfdi_relation_id)) {
                $cfdi33_relacionados['TipoRelacion'] = $customer_transfer->cfdiRelation->code;
                if ($customer_transfer->customerInvoiceRelations->isNotEmpty()) {
                    foreach ($customer_transfer->customerInvoiceRelations as $key => $result) {
                        $cfdi33_relacionado[$key] = [];
                        $cfdi33_relacionado[$key]['UUID'] = $result->uuid_related;
                    }
                }
            }
            //---Emisor
            $cfdi33_emisor = [];
            $cfdi33_emisor['Rfc'] = $customer_transfer->company->taxid;
            $cfdi33_emisor['Nombre'] = trim($customer_transfer->company->name);
            $cfdi33_emisor['RegimenFiscal'] = !empty($customer_transfer->tax_regimen_id) ? $customer_transfer->taxRegimen->code : $customer_transfer->company->taxRegimen->code;
            $cfdi33_emisor['FacAtrAdquirente'] = null;
            //---Receptor
            if($customer_transfer->document_type2 == 'invoice'){
                $cfdi33_receptor = [];
                $cfdi33_receptor['Rfc'] = $customer_transfer->customer->taxid;
                $cfdi33_receptor['Nombre'] = trim($customer_transfer->customer->name);
                if (!in_array($customer_transfer->customer->taxid, ['XAXX010101000','XEXX010101000'])) {
                    $cfdi33_receptor['DomicilioFiscalReceptor'] = $customer_transfer->customer->postcode;
                }else{
                    $cfdi33_receptor['DomicilioFiscalReceptor'] = $customer_transfer->branchOffice->postcode;
                }
                if ($customer_transfer->customer->taxid == 'XEXX010101000') {
                    $cfdi33_receptor['ResidenciaFiscal'] = $customer_transfer->customer->country->code;
                    $cfdi33_receptor['NumRegIdTrib'] = $customer_transfer->customer->numid;
                }
            }else{
                $cfdi33_receptor = [];
                $cfdi33_receptor['Rfc'] = $customer_transfer->company->taxid;
                $cfdi33_receptor['Nombre'] = trim($customer_transfer->company->name);
                $cfdi33_receptor['DomicilioFiscalReceptor'] = $customer_transfer->company->postcode;
            }
            $cfdi33_receptor['RegimenFiscalReceptor'] = !empty($customer_transfer->tax_regimen_customer_id) ? $customer_transfer->taxRegimenCustomer->code : $customer_transfer->customer->taxRegimen->code;
            $cfdi33_receptor['UsoCFDI'] = $customer_transfer->cfdiUse->code;
            //---Conceptos
            $cfdi33_conceptos = [];
            $cfdi33_conceptos_traslados = [];
            $cfdi33_conceptos_retenciones = [];
            foreach ($customer_transfer->customerActiveInvoiceLines as $key => $result) {
                $cfdi33_conceptos [$key]['ClaveProdServ'] = $result->satProduct->code;
                if (!empty($result->product->code)) {
                    $cfdi33_conceptos [$key]['NoIdentificacion'] = trim($result->product->code);
                }
                $cfdi33_conceptos [$key]['Cantidad'] = Helper::numberFormat($result->quantity, 6, false);
                $cfdi33_conceptos [$key]['ClaveUnidad'] = trim($result->unitMeasure->code);
                $cfdi33_conceptos [$key]['Unidad'] = str_limit(trim($result->unitMeasure->name),20,'');
                $cfdi33_conceptos [$key]['Descripcion'] = trim($result->name);
                $cfdi33_conceptos [$key]['ValorUnitario'] = Helper::numberFormatRealDecimalPlace($result->price_unit, false);
                $cfdi33_conceptos [$key]['Importe'] = Helper::numberFormat($result->amount_untaxed + $result->amount_discount,
                    2, false);
                if($result->amount_discount>0) {
                    $cfdi33_conceptos [$key]['Descuento'] = Helper::numberFormat($result->amount_discount, 2, false);
                }
                //['InformacionAduanera']
                //['CuentaPredial']
                //['ComplementoConcepto']
                //['Parte']
                //Complemento

                //Impuestos por concepto
                $cfdi33_conceptos_traslados[$key] = [];
                $cfdi33_conceptos_retenciones[$key] = [];
                if ($result->taxes && $customer_transfer->document_type2 == 'invoice') {
                    foreach ($result->taxes as $key2 => $result2) {
                        $tmp = 0;
                        $rate = $result2->rate;
                        if ($result2->factor == 'Tasa') {
                            $rate /= 100;
                            $tmp = $result->amount_untaxed * $rate;
                        } elseif ($result2->factor == 'Cuota') {
                            $tmp = $rate;
                        }
                        $tmp = round($tmp, 2);
                        if ($result2->type == 'R') { //Retenciones
                            $cfdi33_conceptos_retenciones[$key][$key2] = [];
                            $cfdi33_conceptos_retenciones[$key][$key2]['Base'] = Helper::numberFormat($result->amount_untaxed,
                                2, false);
                            $cfdi33_conceptos_retenciones[$key][$key2]['Impuesto'] = $result2->code;
                            $cfdi33_conceptos_retenciones[$key][$key2]['TipoFactor'] = $result2->factor; //Para retenciones no hay exento
                            $cfdi33_conceptos_retenciones[$key][$key2]['TasaOCuota'] = Helper::numberFormat(abs($rate),
                                6, false);
                            $cfdi33_conceptos_retenciones[$key][$key2]['Importe'] = Helper::numberFormat(abs($tmp), 2,
                                false);
                        } else { //Traslados
                            $cfdi33_conceptos_traslados[$key][$key2] = [];
                            $cfdi33_conceptos_traslados[$key][$key2]['Base'] = Helper::numberFormat($result->amount_untaxed,
                                2, false);
                            $cfdi33_conceptos_traslados[$key][$key2]['Impuesto'] = $result2->code;
                            $cfdi33_conceptos_traslados[$key][$key2]['TipoFactor'] = $result2->factor; //Para retenciones no hay exento
                            if ($result2->factor != 'Exento') {
                                $cfdi33_conceptos_traslados[$key][$key2]['TasaOCuota'] = Helper::numberFormat(abs($rate),
                                    6, false);
                                $cfdi33_conceptos_traslados[$key][$key2]['Importe'] = Helper::numberFormat(abs($tmp), 2,
                                    false);
                            }
                        }
                    }
                }
                $cfdi33_conceptos [$key]['ObjetoImp'] = !empty($result->tax_object) ? $result->tax_object  : ((empty($cfdi33_conceptos_traslados[$key]) && empty($cfdi33_conceptos_retenciones[$key])) ? '01' : '02');
            }
            //Impuestos
            $cfdi33_retenciones = [];
            $cfdi33_traslados = [];
            $cfdi33_exento = null;
            if ($customer_transfer->customerInvoiceTaxes->isNotEmpty() && $customer_transfer->document_type2 == 'invoice') {
                foreach ($customer_transfer->customerInvoiceTaxes as $key => $result) {
                    $tmp = $result->amount_tax;
                    $rate = $result->tax->rate;
                    if ($result->tax->factor == 'Tasa') {
                        $rate /= 100;
                    }
                    if ($result->tax->type == 'R') { //Retenciones
                        $cfdi33_retenciones[$key] = [];
                        $cfdi33_retenciones[$key]['Impuesto'] = $result->tax->code;
                        $cfdi33_retenciones[$key]['Importe'] = Helper::numberFormat(abs($tmp), $customer_transfer->currency->decimal_place, false);
                    } else { //Traslados

                        $cfdi33_traslados[$key] = [];
                        $cfdi33_traslados[$key]['Base'] = Helper::numberFormat(abs($result->amount_base), 2, false);
                        $cfdi33_traslados[$key]['Impuesto'] = $result->tax->code;
                        $cfdi33_traslados[$key]['TipoFactor'] = $result->tax->factor;
                        if ($result->tax->factor != 'Exento') {
                            $cfdi33_traslados[$key]['TasaOCuota'] = Helper::numberFormat(abs($rate), 6, false);
                            $cfdi33_traslados[$key]['Importe'] = Helper::numberFormat(abs($tmp), 2, false);
                        }
                        if($result->tax->factor == 'Exento' && ($cfdi33_exento || is_null($cfdi33_exento)))
                        {
                            $cfdi33_exento = true;
                        }
                        if($result->tax->factor != 'Exento')
                        {
                            $cfdi33_exento = false;
                        }
                    }
                }
            }
            $cfdi33_impuestos = [];
            if ((abs($customer_transfer->amount_tax_ret) > 0 || !empty($cfdi33_retenciones)) && $customer_transfer->document_type2 == 'invoice') {
                $cfdi33_impuestos['TotalImpuestosRetenidos'] = Helper::numberFormat(abs($customer_transfer->amount_tax_ret),
                    $customer_transfer->currency->decimal_place, false);
            }
            if ((abs($customer_transfer->amount_tax) > 0 || !empty($cfdi33_traslados)) && $customer_transfer->document_type2 == 'invoice') {
                $cfdi33_impuestos['TotalImpuestosTrasladados'] = Helper::numberFormat(abs($customer_transfer->amount_tax),
                    $customer_transfer->currency->decimal_place, false);
            }
            if($cfdi33_exento){
                $cfdi33_impuestos['TotalImpuestosTrasladados'] = null;
            }

            //Genera XML
            $certificado = new \CfdiUtils\Certificado\Certificado(\Storage::path($customer_transfer->company->pathFileCer()));
            $creator = new \CfdiUtils\CfdiCreator40($cfdi33, $certificado);
            $creator->setXmlResolver(PacHelper::resourcePathCfdiUtils()); //Almacenamiento local
            $comprobante = $creator->comprobante();

            $comprobante->addAttributes([
                'xsi:schemaLocation' => 'http://www.sat.gob.mx/cfd/4 http://www.sat.gob.mx/sitio_internet/cfd/4/cfdv40.xsd http://www.sat.gob.mx/CartaPorte20 http://www.sat.gob.mx/sitio_internet/cfd/CartaPorte/CartaPorte20.xsd',
            ]);
            $comprobante->addAttributes([
                'xmlns:cartaporte20' => 'http://www.sat.gob.mx/CartaPorte20'
            ]);

            if (!empty($cfdi33_relacionados)) {
                $relacionados = $comprobante->addCfdiRelacionados($cfdi33_relacionados);
                if (!empty($cfdi33_relacionado)) {
                    foreach ($cfdi33_relacionado as $key => $result) {
                        $relacionados->addCfdiRelacionado($result);
                    }
                }
            }
            $comprobante->addEmisor($cfdi33_emisor);
            $comprobante->addReceptor($cfdi33_receptor);
            //Conceptos
            foreach ($cfdi33_conceptos as $key => $result) {
                $concepto = $comprobante->addConcepto($result);
                if (!empty($cfdi33_conceptos_traslados[$key])) {
                    foreach ($cfdi33_conceptos_traslados[$key] as $result2) {
                        $concepto->multiTraslado($result2);
                    }
                }
                if (!empty($cfdi33_conceptos_retenciones[$key])) {
                    foreach ($cfdi33_conceptos_retenciones[$key] as $result2) {
                        $concepto->multiRetencion($result2);
                    }
                }
            }
            //Impuestos
            if(!empty($cfdi33_impuestos)) {
                $comprobante->addImpuestos($cfdi33_impuestos);
                if (!empty($cfdi33_retenciones)) {
                    foreach ($cfdi33_retenciones as $result2) {
                        $comprobante->multiRetencion($result2);
                    }
                }
                if (!empty($cfdi33_traslados)) {
                    foreach ($cfdi33_traslados as $result2) {
                        $comprobante->multiTraslado($result2);
                    }
                }
            }

            //
            $customer_transfer_transfer = $customer_transfer->customerInvoiceTransfer;

            //Complemento carta porte
            $carta_porte = [];
            $carta_porte['Version'] = '2.0';
            $carta_porte['TranspInternac'] = $customer_transfer_transfer->transport_int;
            $carta_porte['EntradaSalidaMerc'] = !empty($customer_transfer_transfer->in_out_product) ? $customer_transfer_transfer->in_out_product : null;
            $carta_porte['PaisOrigenDestino'] = !empty($customer_transfer_transfer->inOutCountry->code) ? $customer_transfer_transfer->inOutCountry->code : null;
            $carta_porte['ViaEntradaSalida'] = !empty($customer_transfer_transfer->transport_type_id) ? $customer_transfer_transfer->transportType->code : null;
            $carta_porte['TotalDistRec'] = !empty($customer_transfer_transfer->distance_traveled) ? $customer_transfer_transfer->distance_traveled : null;
            $nodo_carta_porte = new \CfdiUtils\Nodes\Node(
                'cartaporte20:CartaPorte', // nombre del elemento raíz
                $carta_porte
            );

            //Ubicaciones
            $ubicaciones = [];
            $nodo_ubicaciones = new \CfdiUtils\Nodes\Node(
                'cartaporte20:Ubicaciones', // nombre del elemento raíz
                $ubicaciones
            );
            $nodo_carta_porte->addChild($nodo_ubicaciones);
            //Origen
            $ubicacion_origen = [];
            $ubicacion_origen['TipoUbicacion'] = 'Origen';
            $ubicacion_origen['IDUbicacion'] = !empty($customer_transfer_transfer->source_id) ? $customer_transfer_transfer->source_id : null;
            $ubicacion_origen['RFCRemitenteDestinatario'] = !empty($customer_transfer_transfer->source_taxid) ? $customer_transfer_transfer->source_taxid : null;
            $ubicacion_origen['NombreRemitenteDestinatario'] = !empty($customer_transfer_transfer->source_name) ? $customer_transfer_transfer->source_name : null;
            $ubicacion_origen['NumRegIdTrib'] = !empty($customer_transfer_transfer->source_numid) ? $customer_transfer_transfer->source_numid : null;
            $ubicacion_origen['ResidenciaFiscal'] = !empty($customer_transfer_transfer->sourceTrCountry->code) ? $customer_transfer_transfer->sourceTrCountry->code : null;
            $ubicacion_origen['NumEstacion'] = !empty($customer_transfer_transfer->sourceStation->code) ? $customer_transfer_transfer->sourceStation->code : null;
            $ubicacion_origen['NombreEstacion'] = !empty($customer_transfer_transfer->source_station) ? $customer_transfer_transfer->source_station : (!empty($customer_transfer_transfer->sourceStation->name) ? $customer_transfer_transfer->sourceStation->name : null);
            $ubicacion_origen['NavegacionTrafico'] = !empty($customer_transfer_transfer->source_traffic_navigation) ? $customer_transfer_transfer->source_traffic_navigation : null;
            $ubicacion_origen['FechaHoraSalidaLlegada'] = !empty($customer_transfer_transfer->source_date) ? \Date::parse($customer_transfer_transfer->source_date)->format('Y-m-d\TH:i:s') : null;
            $ubicacion_origen['TipoEstacion'] = !empty($customer_transfer_transfer->sourceStationType->code) ? $customer_transfer_transfer->sourceStationType->code : null;
            $ubicacion_origen['DistanciaRecorrida'] = !empty($customer_transfer_transfer->source_distance_traveled) ? $customer_transfer_transfer->source_distance_traveled : null;
            $nodo_ubicacion_origen = new \CfdiUtils\Nodes\Node(
                'cartaporte20:Ubicacion', // nombre del elemento raíz
                $ubicacion_origen
            );
            $nodo_ubicaciones->addChild($nodo_ubicacion_origen);
            $ubicacion_origen_domicilio = [];
            $ubicacion_origen_domicilio['Calle'] = !empty($customer_transfer_transfer->source_address_1) ? $customer_transfer_transfer->source_address_1 : null;
            $ubicacion_origen_domicilio['NumeroExterior'] = !empty($customer_transfer_transfer->source_address_2) ? $customer_transfer_transfer->source_address_2 : null;
            $ubicacion_origen_domicilio['NumeroInterior'] = !empty($customer_transfer_transfer->source_address_3) ? $customer_transfer_transfer->source_address_3 : null;
            $ubicacion_origen_domicilio['Colonia'] = null;
            $ubicacion_origen_domicilio['Localidad'] = null;
            $ubicacion_origen_domicilio['Referencia'] = !empty($customer_transfer_transfer->source_address_6) ? $customer_transfer_transfer->source_address_6 : null;
            $ubicacion_origen_domicilio['Municipio'] = !empty($customer_transfer_transfer->sourceCity->code) ? $customer_transfer_transfer->sourceCity->code : null;
            $ubicacion_origen_domicilio['Estado'] = !empty($customer_transfer_transfer->sourceState->code) ? $customer_transfer_transfer->sourceState->code : null;
            $ubicacion_origen_domicilio['Pais'] = !empty($customer_transfer_transfer->sourceCountry->code) ? $customer_transfer_transfer->sourceCountry->code : null;
            $ubicacion_origen_domicilio['CodigoPostal'] = !empty($customer_transfer_transfer->source_postcode) ? $customer_transfer_transfer->source_postcode : null;
            $nodoubicacion__origen_domicilio = new \CfdiUtils\Nodes\Node(
                'cartaporte20:Domicilio', // nombre del elemento raíz
                $ubicacion_origen_domicilio
            );
            $nodo_ubicacion_origen->addChild($nodoubicacion__origen_domicilio);
            //Destino
            $ubicacion_destino = [];
            $ubicacion_destino['TipoUbicacion'] = 'Destino';
            $ubicacion_destino['IDUbicacion'] = !empty($customer_transfer_transfer->destination_id) ? $customer_transfer_transfer->destination_id : null;
            $ubicacion_destino['RFCRemitenteDestinatario'] = !empty($customer_transfer_transfer->destination_taxid) ? $customer_transfer_transfer->destination_taxid : null;
            $ubicacion_destino['NombreRemitenteDestinatario'] = !empty($customer_transfer_transfer->destination_name) ? $customer_transfer_transfer->destination_name : null;
            $ubicacion_destino['NumRegIdTrib'] = !empty($customer_transfer_transfer->destination_numid) ? $customer_transfer_transfer->destination_numid : null;
            $ubicacion_destino['ResidenciaFiscal'] = !empty($customer_transfer_transfer->destinationTrCountry->code) ? $customer_transfer_transfer->destinationTrCountry->code : null;
            $ubicacion_destino['NumEstacion'] = !empty($customer_transfer_transfer->destinationStation->code) ? $customer_transfer_transfer->destinationStation->code : null;
            $ubicacion_destino['NombreEstacion'] = !empty($customer_transfer_transfer->destination_station) ? $customer_transfer_transfer->destination_station : (!empty($customer_transfer_transfer->destinationStation->name) ? $customer_transfer_transfer->destinationStation->name : null);
            $ubicacion_destino['NavegacionTrafico'] = !empty($customer_transfer_transfer->destination_traffic_navigation) ? $customer_transfer_transfer->destination_traffic_navigation : null;
            $ubicacion_destino['FechaHoraSalidaLlegada'] = !empty($customer_transfer_transfer->destination_date) ? \Date::parse($customer_transfer_transfer->destination_date)->format('Y-m-d\TH:i:s') : null;
            $ubicacion_destino['TipoEstacion'] = !empty($customer_transfer_transfer->destinationStationType->code) ? $customer_transfer_transfer->destinationStationType->code : null;
            $ubicacion_destino['DistanciaRecorrida'] = !empty($customer_transfer_transfer->destination_distance_traveled) ? $customer_transfer_transfer->destination_distance_traveled : null;
            $nodo_ubicacion_destino = new \CfdiUtils\Nodes\Node(
                'cartaporte20:Ubicacion', // nombre del elemento raíz
                $ubicacion_destino
            );
            $nodo_ubicaciones->addChild($nodo_ubicacion_destino);
            $ubicacion_destino_domicilio = [];
            $ubicacion_destino_domicilio['Calle'] = !empty($customer_transfer_transfer->destination_address_1) ? $customer_transfer_transfer->destination_address_1 : null;
            $ubicacion_destino_domicilio['NumeroExterior'] = !empty($customer_transfer_transfer->destination_address_2) ? $customer_transfer_transfer->destination_address_2 : null;
            $ubicacion_destino_domicilio['NumeroInterior'] = !empty($customer_transfer_transfer->destination_address_3) ? $customer_transfer_transfer->destination_address_3 : null;
            $ubicacion_destino_domicilio['Colonia'] = null;
            $ubicacion_destino_domicilio['Localidad'] = null;
            $ubicacion_destino_domicilio['Referencia'] = !empty($customer_transfer_transfer->destination_address_6) ? $customer_transfer_transfer->destination_address_6 : null;
            $ubicacion_destino_domicilio['Municipio'] = !empty($customer_transfer_transfer->destinationCity->code) ? $customer_transfer_transfer->destinationCity->code : null;
            $ubicacion_destino_domicilio['Estado'] = !empty($customer_transfer_transfer->destinationState->code) ? $customer_transfer_transfer->destinationState->code : null;
            $ubicacion_destino_domicilio['Pais'] = !empty($customer_transfer_transfer->destinationCountry->code) ? $customer_transfer_transfer->destinationCountry->code : null;
            $ubicacion_destino_domicilio['CodigoPostal'] = !empty($customer_transfer_transfer->destination_postcode) ? $customer_transfer_transfer->destination_postcode : null;
            $nodo_ubicacion_destino_domicilio = new \CfdiUtils\Nodes\Node(
                'cartaporte20:Domicilio', // nombre del elemento raíz
                $ubicacion_destino_domicilio
            );
            $nodo_ubicacion_destino->addChild($nodo_ubicacion_destino_domicilio);

            //Mercancias
            $mercancias = [];
            $mercancias['PesoBrutoTotal'] = !empty($customer_transfer_transfer->gross_weight) ? $customer_transfer_transfer->gross_weight : null;
            $mercancias['UnidadPeso'] = !empty($customer_transfer_transfer->measurePackaging->code) ? $customer_transfer_transfer->measurePackaging->code : null;
            $mercancias['PesoNetoTotal'] = !empty($customer_transfer_transfer->net_weight) ? $customer_transfer_transfer->net_weight : null;
            $mercancias['NumTotalMercancias'] = $customer_transfer->customerActiveInvoiceLineTransfers->count();
            $mercancias['CargoPorTasacion'] = (!empty($customer_transfer_transfer->appraisal_fee) && $customer_transfer_transfer->appraisal_fee > 0) ? Helper::numberFormat($customer_transfer_transfer->appraisal_fee, 2, false) : null;
            $nodo_mercancias = new \CfdiUtils\Nodes\Node(
                'cartaporte20:Mercancias', // nombre del elemento raíz
                $mercancias
            );
            $nodo_carta_porte->addChild($nodo_mercancias);

            foreach ($customer_transfer->customerActiveInvoiceLineTransfers as $key => $result) {
                $mercancia = [];
                $mercancia['BienesTransp'] = !empty($result->satProduct->code) ? $result->satProduct->code : null;
                $mercancia['Descripcion'] = !empty($result->name) ? $result->name : null;
                $mercancia['Cantidad'] = !empty($result->quantity) ? round($result->quantity,6) : null;
                $mercancia['ClaveUnidad'] = !empty($result->unitMeasure->code) ? $result->unitMeasure->code : null;
                $mercancia['Unidad'] = null;
                $mercancia['Dimensiones'] = !empty($result->dimensions) ? $result->dimensions : null;
                $mercancia['MaterialPeligroso'] = !empty($result->dangerous_product) ? $result->dangerous_product : null;
                $mercancia['CveMaterialPeligroso'] = !empty($result->dangerousProduct->code) ? $result->dangerousProduct->code : null;
                $mercancia['Embalaje'] = !empty($result->packagingType->code) ? $result->packagingType->code : null;
                $mercancia['DescripEmbalaje'] = !empty($result->packagingType->name) ? $result->packagingType->name : null;
                $mercancia['PesoEnKg'] = !empty($result->weight) ? round($result->weight,6) : null;
                $mercancia['ValorMercancia'] = !empty($result->amount_total) ? round($result->amount_total, 2) : null;
                $mercancia['Moneda'] = !empty($result->currency->code) ? $result->currency->code : null;
                $mercancia['FraccionArancelaria'] = !empty($result->tariffFraction->code) ? $result->tariffFraction->code : null;
                $mercancia['UUIDComercioExt'] = !empty($result->uuid_ext) ? $result->uuid_ext : null;
                $nodo_mercancia = new \CfdiUtils\Nodes\Node(
                    'cartaporte20:Mercancia', // nombre del elemento raíz
                    $mercancia
                );
                $nodo_mercancias->addChild($nodo_mercancia);
            }

            if(!empty($customer_transfer_transfer->trans_transport_type_id) && $customer_transfer_transfer->transTransportType->code == '01'){
                $autotransporte = [];
                $autotransporte['PermSCT'] = !empty($customer_transfer_transfer->permitType->code) ? $customer_transfer_transfer->permitType->code : null;
                $autotransporte['NumPermisoSCT'] = !empty($customer_transfer_transfer->permit_number) ? $customer_transfer_transfer->permit_number : null;
                $nodo_autotransporte = new \CfdiUtils\Nodes\Node(
                    'cartaporte20:Autotransporte', // nombre del elemento raíz
                    $autotransporte
                );
                $nodo_mercancias->addChild($nodo_autotransporte);

                $autotransporte_iden_veh = [];
                $autotransporte_iden_veh['ConfigVehicular'] = !empty($customer_transfer_transfer->settingTransport->code) ? $customer_transfer_transfer->settingTransport->code : null;
                $autotransporte_iden_veh['PlacaVM'] = !empty($customer_transfer_transfer->vehicle_number) ? $customer_transfer_transfer->vehicle_number : null;
                $autotransporte_iden_veh['AnioModeloVM'] = !empty($customer_transfer_transfer->vehicle) ? $customer_transfer_transfer->vehicle : null;
                $nodo_autotransporte_iden_veh = new \CfdiUtils\Nodes\Node(
                    'cartaporte20:IdentificacionVehicular', // nombre del elemento raíz
                    $autotransporte_iden_veh
                );
                $nodo_autotransporte->addChild($nodo_autotransporte_iden_veh);

                $autotransporte_seguros = [];
                $autotransporte_seguros['AseguraRespCivil'] = !empty($customer_transfer_transfer->name_insurer) ? $customer_transfer_transfer->name_insurer : null;
                $autotransporte_seguros['PolizaRespCivil'] = !empty($customer_transfer_transfer->policy_number) ? $customer_transfer_transfer->policy_number : null;
                $autotransporte_seguros['AseguraMedAmbiente'] = !empty($customer_transfer_transfer->name_insurer2) ? $customer_transfer_transfer->name_insurer2 : null;
                $autotransporte_seguros['PolizaMedAmbiente'] = !empty($customer_transfer_transfer->policy_number2) ? $customer_transfer_transfer->policy_number2 : null;
                $autotransporte_seguros['AseguraCarga'] = !empty($customer_transfer_transfer->name_insurer3) ? $customer_transfer_transfer->name_insurer3 : null;
                $autotransporte_seguros['PolizaCarga'] = !empty($customer_transfer_transfer->policy_number3) ? $customer_transfer_transfer->policy_number3 : null;
                $autotransporte_seguros['PrimaSeguro'] = !empty($customer_transfer_transfer->insurance_value) ? $customer_transfer_transfer->insurance_value : null;
                $nodo_autotransporte_seguros = new \CfdiUtils\Nodes\Node(
                    'cartaporte20:Seguros', // nombre del elemento raíz
                    $autotransporte_seguros
                );
                $nodo_autotransporte->addChild($nodo_autotransporte_seguros);

                if(!empty($customer_transfer_transfer->trailer_type_id_1) || !empty($customer_transfer_transfer->trailer_type_id_2) || !empty($customer_transfer_transfer->trailer_type_id_3) || !empty($customer_transfer_transfer->trailer_type_id_4) || !empty($customer_transfer_transfer->trailer_type_id_5)){
                    $nodo_autotransporte_remolques = new \CfdiUtils\Nodes\Node(
                        'cartaporte20:Remolques', // nombre del elemento raíz
                        []
                    );
                    $nodo_autotransporte->addChild($nodo_autotransporte_remolques);
                    if(!empty($customer_transfer_transfer->trailer_type_id_1)){
                        $autotransporte_remolque = [];
                        $autotransporte_remolque['Placa'] = !empty($customer_transfer_transfer->trailer_number_1) ? $customer_transfer_transfer->trailer_number_1 : null;
                        $autotransporte_remolque['SubTipoRem'] = !empty($customer_transfer_transfer->trailerType1->code) ? $customer_transfer_transfer->trailerType1->code : null;
                        $nodo_autotransporte_remolque = new \CfdiUtils\Nodes\Node(
                            'cartaporte20:Remolque', // nombre del elemento raíz
                            $autotransporte_remolque
                        );
                        $nodo_autotransporte_remolques->addChild($nodo_autotransporte_remolque);
                    }
                    if(!empty($customer_transfer_transfer->trailer_type_id_2)){
                        $autotransporte_remolque = [];
                        $autotransporte_remolque['Placa'] = !empty($customer_transfer_transfer->trailer_number_2) ? $customer_transfer_transfer->trailer_number_2 : null;
                        $autotransporte_remolque['SubTipoRem'] = !empty($customer_transfer_transfer->trailerType2->code) ? $customer_transfer_transfer->trailerType2->code : null;
                        $nodo_autotransporte_remolque = new \CfdiUtils\Nodes\Node(
                            'cartaporte20:Remolque', // nombre del elemento raíz
                            $autotransporte_remolque
                        );
                        $nodo_autotransporte_remolques->addChild($nodo_autotransporte_remolque);
                    }
                    if(!empty($customer_transfer_transfer->trailer_type_id_3)){
                        $autotransporte_remolque = [];
                        $autotransporte_remolque['Placa'] = !empty($customer_transfer_transfer->trailer_number_3) ? $customer_transfer_transfer->trailer_number_3 : null;
                        $autotransporte_remolque['SubTipoRem'] = !empty($customer_transfer_transfer->trailerType3->code) ? $customer_transfer_transfer->trailerType3->code : null;
                        $nodo_autotransporte_remolque = new \CfdiUtils\Nodes\Node(
                            'cartaporte20:Remolque', // nombre del elemento raíz
                            $autotransporte_remolque
                        );
                        $nodo_autotransporte_remolques->addChild($nodo_autotransporte_remolque);
                    }
                    if(!empty($customer_transfer_transfer->trailer_type_id_4)){
                        $autotransporte_remolque = [];
                        $autotransporte_remolque['Placa'] = !empty($customer_transfer_transfer->trailer_number_4) ? $customer_transfer_transfer->trailer_number_4 : null;
                        $autotransporte_remolque['SubTipoRem'] = !empty($customer_transfer_transfer->trailerType4->code) ? $customer_transfer_transfer->trailerType4->code : null;
                        $nodo_autotransporte_remolque = new \CfdiUtils\Nodes\Node(
                            'cartaporte20:Remolque', // nombre del elemento raíz
                            $autotransporte_remolque
                        );
                        $nodo_autotransporte_remolques->addChild($nodo_autotransporte_remolque);
                    }
                    if(!empty($customer_transfer_transfer->trailer_type_id_5)){
                        $autotransporte_remolque = [];
                        $autotransporte_remolque['Placa'] = !empty($customer_transfer_transfer->trailer_number_5) ? $customer_transfer_transfer->trailer_number_5 : null;
                        $autotransporte_remolque['SubTipoRem'] = !empty($customer_transfer_transfer->trailerType5->code) ? $customer_transfer_transfer->trailerType5->code : null;
                        $nodo_autotransporte_remolque = new \CfdiUtils\Nodes\Node(
                            'cartaporte20:Remolque', // nombre del elemento raíz
                            $autotransporte_remolque
                        );
                        $nodo_autotransporte_remolques->addChild($nodo_autotransporte_remolque);
                    }
                }
            }

            //FiguraTransporte
            if(!empty($customer_transfer_transfer->operator_name) || !empty($customer_transfer_transfer->operator_taxid) || !empty($customer_transfer_transfer->owner_name) || !empty($customer_transfer_transfer->owner_taxid) || !empty($customer_transfer_transfer->lessee_name) || !empty($customer_transfer_transfer->lessee_taxid) || !empty($customer_transfer_transfer->notified_name) || !empty($customer_transfer_transfer->notified_taxid)){
                $figura_transporte = [];
                $nodo_figura_transporte = new \CfdiUtils\Nodes\Node(
                    'cartaporte20:FiguraTransporte', // nombre del elemento raíz
                    $figura_transporte
                );
                $nodo_carta_porte->addChild($nodo_figura_transporte);

                //Operadores
                if(!empty($customer_transfer_transfer->operator_name) || !empty($customer_transfer_transfer->operator_taxid)){
                    //Operador
                    $operador = [];
                    $operador['TipoFigura'] = '01';
                    $operador['RFCFigura'] = !empty($customer_transfer_transfer->operator_taxid) ? $customer_transfer_transfer->operator_taxid : null;
                    $operador['NumLicencia'] = !empty($customer_transfer_transfer->operator_license) ? $customer_transfer_transfer->operator_license : null;
                    $operador['NombreFigura'] = !empty($customer_transfer_transfer->operator_name) ? $customer_transfer_transfer->operator_name : null;
                    $operador['NumRegIdTribFigura'] = !empty($customer_transfer_transfer->operator_numid) ? $customer_transfer_transfer->operator_numid : null;
                    $operador['ResidenciaFiscalFigura'] = !empty($customer_transfer_transfer->operatorTrCountry->code) ? $customer_transfer_transfer->operatorTrCountry->code : null;
                    $nodo_operador = new \CfdiUtils\Nodes\Node(
                        'cartaporte20:TiposFigura', // nombre del elemento raíz
                        $operador
                    );
                    $nodo_figura_transporte->addChild($nodo_operador);
                }

                //Propietario
                if(!empty($customer_transfer_transfer->owner_name) || !empty($customer_transfer_transfer->owner_taxid)){
                    //Propietario
                    $propietario = [];
                    $propietario['TipoFigura'] = '02';
                    $propietario['RFCFigura'] = !empty($customer_transfer_transfer->owner_taxid) ? $customer_transfer_transfer->owner_taxid : null;
                    $propietario['NumLicencia'] = null;
                    $propietario['NombreFigura'] = !empty($customer_transfer_transfer->owner_name) ? $customer_transfer_transfer->owner_name : null;
                    $propietario['NumRegIdTribFigura'] = !empty($customer_transfer_transfer->owner_numid) ? $customer_transfer_transfer->owner_numid : null;
                    $propietario['ResidenciaFiscalFigura'] = !empty($customer_transfer_transfer->ownerTrCountry->code) ? $customer_transfer_transfer->ownerTrCountry->code : null;
                    $nodo_propietario = new \CfdiUtils\Nodes\Node(
                        'cartaporte20:TiposFigura', // nombre del elemento raíz
                        $propietario
                    );
                    $nodo_figura_transporte->addChild($nodo_propietario);
                    if(!empty($customer_transfer_transfer->owner_transport_part_id)){
                        $propietario_partes = [];
                        $propietario_partes['ParteTransporte'] = !empty($customer_transfer_transfer->ownerTransportPart->code) ? $customer_transfer_transfer->ownerTransportPart->code : null;
                        $nodo_propietario_partes = new \CfdiUtils\Nodes\Node(
                            'cartaporte20:PartesTransporte', // nombre del elemento raíz
                            $propietario_partes
                        );
                        $nodo_propietario->addChild($nodo_propietario_partes);
                        if(!empty($customer_transfer_transfer->owner_postcode)){
                            $propietario_domicilio = [];
                            $propietario_domicilio['Calle'] = !empty($customer_transfer_transfer->owner_address_1) ? $customer_transfer_transfer->owner_address_1 : null;
                            $propietario_domicilio['NumeroExterior'] = !empty($customer_transfer_transfer->owner_address_2) ? $customer_transfer_transfer->owner_address_2 : null;
                            $propietario_domicilio['NumeroInterior'] = !empty($customer_transfer_transfer->owner_address_3) ? $customer_transfer_transfer->owner_address_3 : null;
                            $propietario_domicilio['Colonia'] = null;
                            $propietario_domicilio['Localidad'] = null;
                            $propietario_domicilio['Referencia'] = !empty($customer_transfer_transfer->owner_address_6) ? $customer_transfer_transfer->owner_address_6 : null;
                            $propietario_domicilio['Municipio'] = !empty($customer_transfer_transfer->ownerCity->code) ? $customer_transfer_transfer->ownerCity->code : null;
                            $propietario_domicilio['Estado'] = !empty($customer_transfer_transfer->ownerState->code) ? $customer_transfer_transfer->ownerState->code : null;
                            $propietario_domicilio['Pais'] = !empty($customer_transfer_transfer->ownerCountry->code) ? $customer_transfer_transfer->ownerCountry->code : null;
                            $propietario_domicilio['CodigoPostal'] = !empty($customer_transfer_transfer->owner_postcode) ? $customer_transfer_transfer->owner_postcode : null;
                            $nodo_propietario_domicilio = new \CfdiUtils\Nodes\Node(
                                'cartaporte20:Domicilio', // nombre del elemento raíz
                                $propietario_domicilio
                            );
                            $nodo_propietario->addChild($nodo_propietario_domicilio);
                        }
                    }
                }

                //Arrendatario
                if(!empty($customer_transfer_transfer->lessee_name) || !empty($customer_transfer_transfer->lessee_taxid)){
                    //Arrendatario
                    $arrendatario = [];
                    $arrendatario['TipoFigura'] = '03';
                    $arrendatario['RFCFigura'] = !empty($customer_transfer_transfer->lessee_taxid) ? $customer_transfer_transfer->lessee_taxid : null;
                    $arrendatario['NumLicencia'] = null;
                    $arrendatario['NombreFigura'] = !empty($customer_transfer_transfer->lessee_name) ? $customer_transfer_transfer->lessee_name : null;
                    $arrendatario['NumRegIdTribFigura'] = !empty($customer_transfer_transfer->lessee_numid) ? $customer_transfer_transfer->lessee_numid : null;
                    $arrendatario['ResidenciaFiscalFigura'] = !empty($customer_transfer_transfer->lesseeTrCountry->code) ? $customer_transfer_transfer->lesseeTrCountry->code : null;
                    $nodo_arrendatario = new \CfdiUtils\Nodes\Node(
                        'cartaporte20:TiposFigura', // nombre del elemento raíz
                        $arrendatario
                    );
                    $nodo_figura_transporte->addChild($nodo_arrendatario);
                    if(!empty($customer_transfer_transfer->lessee_transport_part_id)){
                        $arrendatario_partes = [];
                        $arrendatario_partes['ParteTransporte'] = !empty($customer_transfer_transfer->lesseeTransportPart->code) ? $customer_transfer_transfer->lesseeTransportPart->code : null;
                        $nodo_arrendatario_partes = new \CfdiUtils\Nodes\Node(
                            'cartaporte20:PartesTransporte', // nombre del elemento raíz
                            $arrendatario_partes
                        );
                        $nodo_arrendatario->addChild($nodo_arrendatario_partes);
                        if(!empty($customer_transfer_transfer->lessee_postcode)){
                            $arrendatario_domicilio = [];
                            $arrendatario_domicilio['Calle'] = !empty($customer_transfer_transfer->lessee_address_1) ? $customer_transfer_transfer->lessee_address_1 : null;
                            $arrendatario_domicilio['NumeroExterior'] = !empty($customer_transfer_transfer->lessee_address_2) ? $customer_transfer_transfer->lessee_address_2 : null;
                            $arrendatario_domicilio['NumeroInterior'] = !empty($customer_transfer_transfer->lessee_address_3) ? $customer_transfer_transfer->lessee_address_3 : null;
                            $arrendatario_domicilio['Colonia'] = null;
                            $arrendatario_domicilio['Localidad'] = null;
                            $arrendatario_domicilio['Referencia'] = !empty($customer_transfer_transfer->lessee_address_6) ? $customer_transfer_transfer->lessee_address_6 : null;
                            $arrendatario_domicilio['Municipio'] = !empty($customer_transfer_transfer->lesseeCity->code) ? $customer_transfer_transfer->lesseeCity->code : null;
                            $arrendatario_domicilio['Estado'] = !empty($customer_transfer_transfer->lesseeState->code) ? $customer_transfer_transfer->lesseeState->code : null;
                            $arrendatario_domicilio['Pais'] = !empty($customer_transfer_transfer->lesseeCountry->code) ? $customer_transfer_transfer->lesseeCountry->code : null;
                            $arrendatario_domicilio['CodigoPostal'] = !empty($customer_transfer_transfer->lessee_postcode) ? $customer_transfer_transfer->lessee_postcode : null;
                            $nodo_arrendatario_domicilio = new \CfdiUtils\Nodes\Node(
                                'cartaporte20:Domicilio', // nombre del elemento raíz
                                $arrendatario_domicilio
                            );
                            $nodo_arrendatario->addChild($nodo_arrendatario_domicilio);
                        }
                    }
                }

                //Notificado
                if(!empty($customer_transfer_transfer->notified_name) || !empty($customer_transfer_transfer->notified_taxid)){
                    //Notificado
                    $notificado = [];
                    $notificado['TipoFigura'] = '04';
                    $notificado['RFCFigura'] = !empty($customer_transfer_transfer->notified_taxid) ? $customer_transfer_transfer->notified_taxid : null;
                    $notificado['NumLicencia'] = null;
                    $notificado['NombreFigura'] = !empty($customer_transfer_transfer->notified_name) ? $customer_transfer_transfer->notified_name : null;
                    $notificado['NumRegIdTribFigura'] = !empty($customer_transfer_transfer->notified_numid) ? $customer_transfer_transfer->notified_numid : null;
                    $notificado['ResidenciaFiscalFigura'] = !empty($customer_transfer_transfer->notifiedTrCountry->code) ? $customer_transfer_transfer->notifiedTrCountry->code : null;
                    $nodo_notificado = new \CfdiUtils\Nodes\Node(
                        'cartaporte20:TiposFigura', // nombre del elemento raíz
                        $notificado
                    );
                    $nodo_figura_transporte->addChild($nodo_notificado);
                }
            }

            //Agregamos al complemento
            $comprobante->addComplemento($nodo_carta_porte);

            //Método de ayuda para establecer las sumas del comprobante e impuestos con base en la suma de los conceptos y la agrupación de sus impuestos
            //$creator->addSumasConceptos(null, 2);
            //Método de ayuda para generar el sello (obtener la cadena de origen y firmar con la llave privada)
            $creator->addSello('file://' . \Storage::path($customer_transfer->company->pathFileKeyPassPem()), Crypt::decryptString($customer_transfer->company->password_key));
            //Valida la estructura
            //$creator->validate();

            //Guarda XML
            //dd($creator->asXml());
            $path_xml = Helper::setDirectory(CustomerTransfer::PATH_XML_FILES_CTR, $customer_transfer->company_id) . '/';
            if($customer_transfer->document_type2 == 'invoice'){
                $path_xml = Helper::setDirectory(CustomerTransfer::PATH_XML_FILES_CI, $customer_transfer->company_id) . '/';
            }
            $file_xml = Helper::makeDirectoryCfdi($path_xml) . '/' . Str::random(40) . '.xml';
            $creator->saveXml(\Storage::path($path_xml . $file_xml));


            //Arreglo temporal para actualizar Customer Transfer CFDI
            $tmp = [
                'pac_id' => $pac->id,
                'cfdi_version' => setting('cfdi_version'),
                'uuid' => '',
                'date' => '',
                'path_xml' => $path_xml,
                'file_xml' => $file_xml,
                'file_xml_pac' => '',
                'pac' => $pac,
            ];

            //Timbrado de XML
            $class_pac = $pac->code;
            $tmp = PacHelper::$class_pac($tmp, $creator);

            return $tmp;
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Descarga de archivo XML
     *
     * @param Request $request
     * @param CustomerTransfer $customer_transfer
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector|\Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function downloadXml(Request $request, CustomerTransfer $customer_transfer)
    {
        //Ruta y validacion del XML
        $path_xml = Helper::setDirectory(CustomerTransfer::PATH_XML_FILES_CTR, $customer_transfer->company_id) . '/';
        if($customer_transfer->document_type2 == 'invoice'){
            $path_xml = Helper::setDirectory(CustomerTransfer::PATH_XML_FILES_CI, $customer_transfer->company_id) . '/';
        }
        $file_xml_pac = $path_xml . $customer_transfer->customerInvoiceCfdi->file_xml_pac;
        if (!empty($customer_transfer->customerInvoiceCfdi->file_xml_pac) && \Storage::exists($file_xml_pac)) {
            while (ob_get_level()) ob_end_clean();
            ob_start();

            return response()->download(\Storage::path($file_xml_pac), str_replace('/','',$customer_transfer->name) . '.xml',['Cache-Control' => 'no-cache, must-revalidate']);
        }

        //Mensaje
        flash(__('sales/customer_transfer.error_download_xml'))->error();

        //Redireccion
        return redirect('/sales/customer-transfers');
    }

    /**
     * Descarga de archivo PDF
     *
     * @param Request $request
     * @param CustomerTransfer $customer_transfer
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector|\Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function downloadPdf(Request $request, CustomerTransfer $customer_transfer)
    {
        //Descarga archivo
        return $this->print($customer_transfer, true);
    }

    /**
     * Cambiar estatus a abierta
     *
     * @param CustomerTransfer $customer_transfer
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function markOpen(CustomerTransfer $customer_transfer)
    {
        //Logica
        if ((int)$customer_transfer->status == CustomerTransfer::TO_PAY || (int)$customer_transfer->status == CustomerTransfer::PAID) {
            $customer_transfer->updated_uid = \Auth::user()->id;
            $customer_transfer->status = CustomerTransfer::OPEN;
            $customer_transfer->save();

            //Cancelacion del timbre fiscal

            //Mensaje
            flash(__('general.text_form_success_edit'))->success();
        } else {
            //Mensaje
            flash(__('sales/customer_transfer.error_change_status'))->error();
        }

        //Redireccion
        return redirect('/sales/customer-transfers');
    }

    /**
     * Cambiar estatus a abierta
     *
     * @param CustomerTransfer $customer_fee
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function markToPay(CustomerTransfer $customer_transfer)
    {
        //Logica
        if ((int)$customer_transfer->status == CustomerTransfer::OPEN || (int)$customer_transfer->status == CustomerTransfer::PAID) {
            $customer_transfer->updated_uid = \Auth::user()->id;
            $customer_transfer->status = CustomerTransfer::TO_PAY;
            $customer_transfer->save();

            //Cancelacion del timbre fiscal

            //Mensaje
            flash(__('general.text_form_success_edit'))->success();
        } else {
            //Mensaje
            flash(__('sales/customer_transfer.error_change_status'))->error();
        }

        //Redireccion
        return redirect('/sales/customer-transfers');
    }

    /**
     * Cambiar estatus a enviada
     *
     * @param CustomerTransfer $customer_transfer
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function markSent(CustomerTransfer $customer_transfer)
    {
        //Logica
        $customer_transfer->updated_uid = \Auth::user()->id;
        $customer_transfer->mail_sent = 1;
        $customer_transfer->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/sales/customer-transfers');
    }

    /**
     * Cambiar estatus a pagada
     *
     * @param CustomerTransfer $customer_transfer
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function markPaid(CustomerTransfer $customer_transfer)
    {

        //Logica
        if ((int)$customer_transfer->status == CustomerTransfer::OPEN || (int)$customer_transfer->status == CustomerTransfer::TO_PAY) {
            $customer_transfer->updated_uid = \Auth::user()->id;
            $customer_transfer->status = CustomerTransfer::PAID;
            $customer_transfer->save();

            //Mensaje
            flash(__('general.text_form_success_edit'))->success();
        } else {
            //Mensaje
            flash(__('sales/customer_transfer.error_change_status'))->error();
        }

        //Redireccion
        return redirect('/sales/customer-transfers');
    }

    /**
     * Clase generica de impresion
     *
     * @param CustomerTransfer $customer_transfer
     * @return mixed
     */
    public function print(CustomerTransfer $customer_transfer, $download = false, $save = false)
    {
        try {
            //Logica
            $tmp = 'default';
            if (!empty($customer_transfer->customerInvoiceCfdi->cfdi_version)) {
                $tmp = $customer_transfer->customerInvoiceCfdi->cfdi_version;
            }
            $class_print = 'print' . ucfirst($tmp);
            return $this->$class_print($customer_transfer,$download,$save);

        } catch (\Exception $e) {
            flash($e->getMessage())->error();
            return redirect('/sales/customer-transfers');
        }
    }

    /**
     * Impresion default
     *
     * @param $customer_transfer
     * @return mixed
     */
    private function printDefault($customer_transfer, $download = false, $save = false)
    {
        //Datos
        $list_document_type2 = $this->list_document_type2;

        //PDF
        $pdf = \PDF::loadView('sales.customer_transfers.pdf_default', compact('customer_transfer','list_document_type2'));

        //Guardar
        if ($save) {
            return $pdf->output();
        }

        //Redireccion
        if($download){
            return $pdf->download($this->list_document_type2[$customer_transfer->document_type2] . '_' . str_replace('/','',$customer_transfer->name) . '.pdf');
        }
        return $pdf->stream($this->list_document_type2[$customer_transfer->document_type2] . '_' . str_replace('/','',$customer_transfer->name) . '.pdf');
    }

    /**
     * Impresion CFDI 3.3
     *
     * @param $customer_transfer
     * @param bool $download
     * @return mixed
     * @throws \Exception
     */
    private function printCfdi33($customer_transfer, $download = false, $save = false)
    {
        //Datos
        $list_document_type2 = $this->list_document_type2;

        //Datos del XML timbrado
        $path_xml = Helper::setDirectory(CustomerTransfer::PATH_XML_FILES_CTR, $customer_transfer->company_id) . '/';
        if($customer_transfer->document_type2 == 'invoice'){
            $path_xml = Helper::setDirectory(CustomerTransfer::PATH_XML_FILES_CI, $customer_transfer->company_id) . '/';
        }
        $file_xml_pac = $path_xml . $customer_transfer->customerInvoiceCfdi->file_xml_pac;
        $data = [];

        //Valida que el archivo exista
        if (!empty($customer_transfer->customerInvoiceCfdi->file_xml_pac) && \Storage::exists($file_xml_pac)) {
            //Crea arreglo
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($file_xml_pac));
            $data = Cfdi33Helper::getQuickArrayCfdi($cfdi);

            //Genera codigo QR
            $image = QrCode::format('png')->size(150)->margin(0)->generate($data['qr_cadena']);
            $data['qr'] = 'data:image/png;base64,' . base64_encode($image);

            //Regimen fiscal
            $data['tax_regimen'] = TaxRegimen::where('code', $data['cfdi33']->emisor['RegimenFiscal'])->first()->name_sat;
        }

        //PDF
        $pdf_template = \App\Helpers\Helper::companyPdfTemplate($customer_transfer->company_id);//Plantilla PDF
        if(!empty($customer_transfer->customerInvoiceTransfer->transport_int)){
            $customer_transfer_transfer = $customer_transfer->customerInvoiceTransfer;
            $pdf = \PDF::loadView('sales.customer_transfers.pdf_cfdi33_v2_default', compact('customer_transfer', 'data','list_document_type2', 'customer_transfer_transfer'));
        }else{
            $pdf = \PDF::loadView('sales.customer_transfers.pdf_cfdi33_' . $pdf_template, compact('customer_transfer', 'data','list_document_type2'));
        }

        //Guardar
        if ($save) {
            return $pdf->output();
        }

        //Redireccion
        if($download){
            return $pdf->download(str_replace('/','',$customer_transfer->name) . '.pdf');
        }
        return $pdf->stream(str_replace('/','',$customer_transfer->name) . '.pdf');
    }

    /**
     * Impresion CFDI 4.0
     *
     * @param $customer_transfer
     * @param bool $download
     * @return mixed
     * @throws \Exception
     */
    private function printCfdi40($customer_transfer, $download = false, $save = false)
    {
        //Datos
        $list_document_type2 = $this->list_document_type2;

        //Datos del XML timbrado
        $path_xml = Helper::setDirectory(CustomerTransfer::PATH_XML_FILES_CTR, $customer_transfer->company_id) . '/';
        if($customer_transfer->document_type2 == 'invoice'){
            $path_xml = Helper::setDirectory(CustomerTransfer::PATH_XML_FILES_CI, $customer_transfer->company_id) . '/';
        }
        $file_xml_pac = $path_xml . $customer_transfer->customerInvoiceCfdi->file_xml_pac;
        $data = [];

        //Valida que el archivo exista
        if (!empty($customer_transfer->customerInvoiceCfdi->file_xml_pac) && \Storage::exists($file_xml_pac)) {
            //Crea arreglo
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($file_xml_pac));
            $data = Cfdi33Helper::getQuickArrayCfdi($cfdi);

            //Genera codigo QR
            $image = QrCode::format('png')->size(150)->margin(0)->generate($data['qr_cadena']);
            $data['qr'] = 'data:image/png;base64,' . base64_encode($image);

            //Regimen fiscal
            $data['tax_regimen'] = TaxRegimen::where('code', $data['cfdi33']->emisor['RegimenFiscal'])->first()->name_sat;
            $data['tax_regimen_customer'] = TaxRegimen::where('code', $data['cfdi33']->receptor['RegimenFiscalReceptor'])->first()->name_sat;
        }

        //PDF
        $pdf_template = \App\Helpers\Helper::companyPdfTemplate($customer_transfer->company_id);//Plantilla PDF
        if(!empty($customer_transfer->customerInvoiceTransfer->transport_int)){
            $customer_transfer_transfer = $customer_transfer->customerInvoiceTransfer;
            $pdf = \PDF::loadView('sales.customer_transfers.pdf_cfdi40_v2_default', compact('customer_transfer', 'data','list_document_type2', 'customer_transfer_transfer'));
        }else{
            $pdf = \PDF::loadView('sales.customer_transfers.pdf_cfdi40_' . $pdf_template, compact('customer_transfer', 'data','list_document_type2'));
        }

        //Guardar
        if ($save) {
            return $pdf->output();
        }

        //Redireccion
        if($download){
            return $pdf->download(str_replace('/','',$customer_transfer->name) . '.pdf');
        }
        return $pdf->stream(str_replace('/','',$customer_transfer->name) . '.pdf');
    }

    /**
     * Modal para envio de correo de factura
     *
     * @param Request $request
     * @param CustomerTransfer $customer_transfer
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function modalSendMail(Request $request, CustomerTransfer $customer_transfer)
    {
        //Variables
        $id = $request->id;

        //Logica
        if ($request->ajax() && !empty($id)) {
            //Correo default del cliente
            $reply = Helper::replyMails();
            $to = [];
            $to_selected = [];
            if (!empty($customer_transfer->customer->email)) {
                $tmp = explode(';',$customer_transfer->customer->email);
                if(!empty($tmp)) {
                    foreach($tmp as $email) {
                        $email = trim($email);
                        $to[$email] = $email;
                        $to_selected [] = $email;
                    }
                }
            }
            //Etiquetas solo son demostrativas
            if($customer_transfer->status != CustomerTransfer::DRAFT) {
                $files = [
                    'pdf' => str_replace('/','',$customer_transfer->name) . '.pdf',
                    'xml' => str_replace('/','',$customer_transfer->name) . '.xml'
                ];
            }else{
                $files = [
                    'pdf' => str_replace('/','',$customer_transfer->name) . '.pdf',
                ];
            }
            $files_selected = array_keys($files);

            //modal de mensaje
            $html = view('layouts.partials.customer_transfers.modal_send_mail',
                compact('customer_transfer', 'to', 'to_selected', 'files', 'files_selected', 'reply'))->render();

            //Mensaje predefinido
            $custom_message = view('layouts.partials.customer_transfers.message_send_mail',
                compact('customer_transfer'))->render();

            return response()->json(['html' => $html, 'custom_message' => $custom_message]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Envio de factura por correo
     *
     * @param Request $request
     * @param CustomerTransfer $customer_transfer
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function sendMail(Request $request, CustomerTransfer $customer_transfer)
    {
        //Validaciones
        $validator = \Validator::make($request->all(), [
            'subject' => 'required',
            'to' => 'required',
            'to.*' => 'email',
            'message' => 'required'
        ], [
            'subject.*' => __('general.error_mail_subject'),
            'to.required' => __('general.error_mail_to'),
            'to.*.email' => __('general.error_mail_to_format'),
            'message.*' => __('general.error_mail_message'),
        ]);
        if ($validator->fails()) {
            $errors = '<ul>';
            foreach ($validator->errors()->all() as $message) {
                $errors .= '<li>'.$message . '</li>';
            }
            $errors .= '</ul>';
            return response()->json(['error' => $errors], 422);
        }

        //Creamos PDF en stream
        $pdf = $this->print($customer_transfer);
        //Ruta del XML
        $path_xml = Helper::setDirectory(CustomerTransfer::PATH_XML_FILES_CTR, $customer_transfer->company_id) . '/';
        if($customer_transfer->document_type2 == 'invoice'){
            $path_xml = Helper::setDirectory(CustomerTransfer::PATH_XML_FILES_CI, $customer_transfer->company_id) . '/';
        }
        $file_xml_pac = $path_xml . $customer_transfer->customerInvoiceCfdi->file_xml_pac;

        //Envio de correo
        $to = $request->to;
        $reply = $request->reply;
        \Mail::to($to)->send(new SendCustomerTransfer($customer_transfer, $request->subject, $request->message, $pdf,
            $file_xml_pac, $reply));

        //Actualiza campo de enviado
        $customer_transfer->updated_uid = \Auth::user()->id;
        $customer_transfer->mail_sent = 1;
        $customer_transfer->save();

        //Mensaje
        return response()->json([
            'success' => sprintf(__('sales/customer_transfer.text_success_send_mail'), $customer_transfer->name)
        ]);
    }

    /**
     * Obtener registro
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCustomerTransfer(Request $request)
    {
        //Variables
        $id = $request->id;

        //Logica
        if ($request->ajax() && !empty($id)) {
            $customer_transfer = CustomerTransfer::findOrFail($id);
            $customer_transfer->uuid = $customer_transfer->customerInvoiceCfdi->uuid ?? '';
            return response()->json($customer_transfer, 200);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Autoacompletar select2 de facturas solo activas del SAT
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function autocompleteCfdi(Request $request)
    {
        //Variables
        $term = $request->term;
        $customer_id = !empty($request->customer_id) ? $request->customer_id : '111111111111'; //Filtra las facturas por cliente

        //Logica
        if ($request->ajax() && !empty($term)) {
            $tmp = CustomerTransfer::filter([
                'filter_search_cfdi_select2' => $term,
                'filter_customer_id' => $customer_id,
                'filter_document_type_code' => [$this->document_type_code,'customer.invoice'],
            ])->sortable('date')->limit(16)->get();
            $results = [];
            if ($tmp->isNotEmpty()) {
                foreach ($tmp as $result) {
                    $results[] = [
                        'id' => $result->id,
                        'text' => $result->text_select2,
                        'description' => $result->description_select2
                    ];
                }
            }
            return response()->json($results, 200);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Calcula el total de las lineas
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function totalLines(Request $request)
    {

        //Variables
        $json = new \stdClass;
        $input_items = $request->item;
        $currency_id = $request->currency_id;
        $currency_code = 'MXN';
        $taxes = [];

        if ($request->ajax()) {
            //Datos de moneda
            if (!empty($currency_id)) {
                $currency = Currency::findOrFail($currency_id);
                $currency_code = $currency->code;
            }
            //Variables de totales
            $amount_discount = 0;
            $amount_untaxed = 0;
            $amount_tax = 0;
            $amount_tax_ret = 0;
            $amount_total = 0;
            $balance = 0;
            $items = [];
            if (!empty($input_items)) {
                foreach ($input_items as $key => $item) {
                    //Logica
                    $item_quantity = (double)$item['quantity'];
                    $item_price_unit = (double)$item['price_unit'];
                    $item_discount = (double)$item['discount'];
                    $item_price_reduce = ($item_price_unit * (100 - $item_discount) / 100);
                    $item_amount_untaxed = round($item_quantity * $item_price_reduce, 2);
                    $item_amount_discount = round($item_quantity * $item_price_unit, 2) - $item_amount_untaxed;
                    $item_amount_tax = 0;
                    $item_amount_tax_ret = 0;
                    if (!empty($item['taxes'])) {
                        foreach ($item['taxes'] as $tax_id) {
                            if (!empty($tax_id)) {
                                $tax = Tax::findOrFail($tax_id);
                                $tmp = 0;
                                if ($tax->factor == 'Tasa') {
                                    $tmp = $item_amount_untaxed * $tax->rate / 100;
                                } elseif ($tax->factor == 'Cuota') {
                                    $tmp = $tax->rate;
                                }
                                $tmp = round($tmp, 2);
                                if ($tax->type == 'R') { //Retenciones
                                    $item_amount_tax_ret += $tmp;
                                } else { //Traslados
                                    $item_amount_tax += $tmp;
                                }

                                //Sumatoria de impuestos
                                $taxes[$tax_id] = array(
                                    'amount_base' => $item_amount_untaxed + (isset($taxes[$tax_id]['amount_base']) ? $taxes[$tax_id]['amount_base'] : 0),
                                    'amount_tax' => $tmp + (isset($taxes[$tax_id]['amount_tax']) ? $taxes[$tax_id]['amount_tax'] : 0),
                                );
                            }
                        }
                    }
                    $item_amount_total = $item_amount_untaxed + $item_amount_tax + $item_amount_tax_ret;
                    //Sumatoria totales
                    $amount_discount += $item_amount_discount;
                    $amount_untaxed += $item_amount_untaxed;
                    $amount_tax += $item_amount_tax;
                    $amount_tax_ret += $item_amount_tax_ret;
                    $amount_total += $item_amount_total;
                    //Subtotales por cada item
                    $items[$key] = money($item_amount_untaxed, $currency_code, true)->format();
                }
            }
            //Respuesta
            $taxes_tmp = [];
            if (!empty($taxes)) {
                foreach ($taxes as $tax_id => $result) {
                    $tax = Tax::findOrFail($tax_id);
                    $taxes_tmp[] = [
                        'name' => $tax->name,
                        'tax_id' => $tax_id,
                        'amount_base' => money($result['amount_base'], $currency_code, true)->format(),
                        'amount_tax' => money($result['amount_tax'], $currency_code, true)->format(),
                    ];
                }
            }

            //
            $json->items = $items;
            $json->taxes = $taxes_tmp;
            $json->amount_discount = money($amount_discount, $currency_code, true)->format();
            $json->amount_untaxed = money($amount_untaxed, $currency_code, true)->format();
            $json->amount_tax = money($amount_tax + $amount_tax_ret, $currency_code, true)->format();
            $json->amount_total = money($amount_total, $currency_code, true)->format();
            $json->amount_total_tmp = $amount_total;
            return response()->json($json);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Modal para cancelar factura
     *
     * @param Request $request
     * @param CustomerTransfer $customer_transfer
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function modalCancel(Request $request, CustomerTransfer $customer_transfer)
    {
        //Variables
        $id = $request->id;

        //Logica
        if ($request->ajax() && !empty($id)) {
            //Obtener informacion de estatus
            $data_status_sat = [
                'cancelable' => 1,
                'pac_is_cancelable' => ''
            ];
            $reason_cancellations = ReasonCancellation::populateSelect()->get()->pluck('name_sat', 'id');
            if (!empty($customer_transfer->customerInvoiceCfdi->cfdi_version) && !empty($customer_transfer->customerInvoiceCfdi->uuid)) {

                //Obtener el sellos del CFDI
                $path_xml = Helper::setDirectory(CustomerTransfer::PATH_XML_FILES_CTR, $customer_transfer->company_id) . '/';
                if($customer_transfer->document_type2 == 'invoice'){
                    $path_xml = Helper::setDirectory(CustomerTransfer::PATH_XML_FILES_CI, $customer_transfer->company_id) . '/';
                }
                $file_xml_pac = $path_xml . $customer_transfer->customerInvoiceCfdi->file_xml_pac;
                $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($file_xml_pac));
                $data = Cfdi33Helper::getQuickArrayCfdi($cfdi);

                $tmp = [
                    'rfcR' => $data['cfdi33']->Receptor['Rfc'] ?? $customer_transfer->customer->taxid,
                    'uuid' => $customer_transfer->customerInvoiceCfdi->uuid,
                    'total' => Helper::numberFormat($customer_transfer->amount_total, $customer_transfer->currency->decimal_place, false),
                    'fe' => substr($data['cfdi33']->complemento->timbreFiscalDigital['SelloCFD'],-8), //Los últimos 8 caracteres del Atributo Sello del CFDI
                    'file_xml_pac' => $file_xml_pac
                ];
                $class_pac = $customer_transfer->customerInvoiceCfdi->pac->code . 'Status';
                $data_status_sat = PacHelper::$class_pac($tmp,$customer_transfer->company,$customer_transfer->customerInvoiceCfdi->pac);
            }
            $is_cancelable = true;
            if($data_status_sat['cancelable'] == 3){
                $is_cancelable = false;
            }

            //modal de cancelar
            $html = view('layouts.partials.customer_transfers.modal_cancel', compact('customer_transfer','data_status_sat','is_cancelable', 'reason_cancellations'))->render();

            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Exportar datos a excel
     *
     * @param Request $request
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function exportToExcel(Request $request)
    {
        while (ob_get_level()) ob_end_clean();
        ob_start();

        return Excel::download(new CustomerTransfersExport($request),
            __('sales/customer_transfer.document_title') . '-' . config('app.name') . '.xlsx');
    }

    /**
     * Funcion para timbrar factura con estatus de prefactura
     *
     * @param CustomerTransfer $customer_transfer
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function stamp(CustomerTransfer $customer_transfer){

        \DB::connection('tenant')->beginTransaction();
        try {
            $invoiced = false;
            if((int) $customer_transfer->status == CustomerTransfer::DRAFT) {
                //Valida que tenga exista la clase de facturacion
                $class_cfdi = setting('cfdi_version');
                if (empty($class_cfdi)) {
                    throw new \Exception(__('general.error_cfdi_version'));
                }
                if (!method_exists($this, $class_cfdi)) {
                    throw new \Exception(__('general.error_cfdi_class_exists'));
                }

                //Valida que tenga folios disponibles
                if(BaseHelper::getAvailableFolios()<=0){
                    throw new \Exception(__('general.error_available_folios'));
                }

                //CFDI
                $customer_transfer_cfdi = $customer_transfer->customerInvoiceCfdi;

                //Actualiza registr principal
                $customer_transfer->updated_uid = \Auth::user()->id;
                $customer_transfer->date = Helper::dateTimeToSql(BaseHelper::getDateTimeBranchOffice($customer_transfer->branch_office_id));
                $customer_transfer->status = CustomerTransfer::OPEN;
                $customer_transfer->save();

                //Actualiza folios
                //Obtiene folio
                $document_type_code = $this->document_type_code;
                if($customer_transfer->document_type2 == 'invoice'){
                    $document_type_code = 'customer.invoice';
                }
                $document_type = Helper::getNextDocumentTypeByCode($document_type_code, $customer_transfer->company->id,false, $customer_transfer->branch_office_id);
                $customer_transfer->draft = $customer_transfer->name;
                $customer_transfer->name = $document_type['name'];
                $customer_transfer->serie = $document_type['serie'];
                $customer_transfer->folio = $document_type['folio'];
                $customer_transfer->save();
                $customer_transfer_cfdi->name = $customer_transfer->name;
                $customer_transfer_cfdi->save();

                //Valida Empresa y PAC para timbrado
                PacHelper::validateSatActions($customer_transfer->company);

                //Crear XML y timbra
                $tmp = $this->$class_cfdi($customer_transfer);

                //Guardar registros de CFDI
                $customer_transfer_cfdi->fill(array_only($tmp, [
                    'pac_id',
                    'cfdi_version',
                    'uuid',
                    'date',
                    'file_xml',
                    'file_xml_pac',
                ]));
                $customer_transfer_cfdi->save();

                //Disminuye folios
                BaseHelper::decrementFolios();

                //Mensaje
                flash(__('general.text_success_customer_transfer_cfdi'))->success();

                $invoiced = true;
            }

            \DB::connection('tenant')->commit();

            if($invoiced) {
                $this->saveCfdiDownloads($customer_transfer, $customer_transfer_cfdi);
            }


        } catch (\Exception $e) {
            \DB::connection('tenant')->rollback();
            flash( __('general.error_cfdi_pac').'<br>'.$e->getMessage())->error();
            return back()->withInput();
        }

        //Almacenamiento dropbox
        self::dropboxBackup($customer_transfer);

        //Redireccion
        return redirect('/sales/customer-transfers');
    }

    /**
     * Funcion para autorizar la cancelacion
     *
     * @param CustomerTransfer $customer_transfer
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function cancelAuthorized(CustomerTransfer $customer_transfer){

        \DB::connection('tenant')->beginTransaction();
        try {

            if((int) $customer_transfer->status == CustomerTransfer::CANCEL_PER_AUTHORIZED) {

                //Actualiza registro principal
                $customer_transfer->updated_uid = \Auth::user()->id;
                $customer_transfer->status = CustomerTransfer::CANCEL;
                $customer_transfer->save();

                //Mensaje
                flash(__('general.text_form_success_cancel'))->success();
            }

            \DB::connection('tenant')->commit();

            //Redireccion
            return redirect('/sales/customer-transfers');
        } catch (\Exception $e) {
            \DB::connection('tenant')->rollback();
            flash($e->getMessage())->error();
            return back()->withInput();
        }
    }

    /**
     * Funcion para cancelar la cancelacion
     *
     * @param CustomerTransfer $customer_transfer
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function cancelRejected(CustomerTransfer $customer_transfer){

        \DB::connection('tenant')->beginTransaction();
        try {

            if((int) $customer_transfer->status == CustomerTransfer::CANCEL_PER_AUTHORIZED) {

                //Actualiza registro principal
                $customer_transfer->updated_uid = \Auth::user()->id;
                $customer_transfer->status = CustomerTransfer::OPEN;
                $customer_transfer->save();

                //Actualiza el cfdi
                $customer_transfer->customerInvoiceCfdi->cancel_date = null;
                $customer_transfer->customerInvoiceCfdi->cancel_response = 'Cancelación rechazada - ' . $customer_transfer->customerInvoiceCfdi->cancel_response;
                //$customer_transfer->customerInvoiceCfdi->cancel_state = null;
                $customer_transfer->customerInvoiceCfdi->status = 1;
                $customer_transfer->customerInvoiceCfdi->save();

                $cfdi_download = CfdiDownload::where('uuid', $customer_transfer->customerInvoiceCfdi->uuid)->where('type', 2)->first();
                if(!empty($cfdi_download)){
                    $cfdi_download->status = 'Vigente';
                    $cfdi_download->save();
                }

                //Mensaje
                flash(__('general.text_success_customer_transfer_rejected'))->success();
            }

            \DB::connection('tenant')->commit();

            //Redireccion
            return redirect('/sales/customer-transfers');
        } catch (\Exception $e) {
            \DB::connection('tenant')->rollback();
            flash($e->getMessage())->error();
            return back()->withInput();
        }
    }

    /**
     * Modal para estatus de CFDI
     *
     * @param Request $request
     * @param CustomerTransfer $customer_transfer
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function modalStatusSat(Request $request, CustomerTransfer $customer_transfer)
    {
        //Variables
        $id = $request->id;

        //Logica
        if ($request->ajax() && !empty($id)) {
            //Obtener informacion de estatus
            $data_status_sat = [
                'cancelable' => 1,
                'pac_is_cancelable' => ''
            ];
            if (!empty($customer_transfer->customerInvoiceCfdi->cfdi_version) && !empty($customer_transfer->customerInvoiceCfdi->uuid)) {

                //Obtener el sellos del CFDI
                $path_xml = Helper::setDirectory(CustomerTransfer::PATH_XML_FILES_CTR, $customer_transfer->company_id) . '/';
                if($customer_transfer->document_type2 == 'invoice'){
                    $path_xml = Helper::setDirectory(CustomerTransfer::PATH_XML_FILES_CI, $customer_transfer->company_id) . '/';
                }
                $file_xml_pac = $path_xml . $customer_transfer->customerInvoiceCfdi->file_xml_pac;
                $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($file_xml_pac));
                $data = Cfdi33Helper::getQuickArrayCfdi($cfdi);

                $tmp = [
                    'rfcR' => $data['cfdi33']->Receptor['Rfc'] ?? $customer_transfer->customer->taxid,
                    'uuid' => $customer_transfer->customerInvoiceCfdi->uuid,
                    'total' => Helper::numberFormat($customer_transfer->amount_total, $customer_transfer->currency->decimal_place, false),
                    'fe' => substr($data['cfdi33']->complemento->timbreFiscalDigital['SelloCFD'],-8), //Los últimos 8 caracteres del Atributo Sello del CFDI
                    'file_xml_pac' => $file_xml_pac
                ];
                $class_pac = $customer_transfer->customerInvoiceCfdi->pac->code . 'Status';
                $data_status_sat = PacHelper::$class_pac($tmp,$customer_transfer->company,$customer_transfer->customerInvoiceCfdi->pac);
            }
            $is_cancelable = true;
            if($data_status_sat['cancelable'] == 3){
                $is_cancelable = false;
            }

            //modal de visualizar estatus en el SAT
            $html = view('layouts.partials.customer_transfers.modal_status_sat', compact('customer_transfer','data_status_sat','is_cancelable'))->render();

            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Respaldo en dropbox si esta activo
     *
     * @param $customer_transfer
     */
    public function dropboxBackup($customer_transfer, $save_xml = true, $save_pdf = true){
        try{
            if(!empty($customer_transfer) && !empty(setting('dropbox_backup')) && !empty(setting('dropbox_access_token'))){
                $path_xml = Helper::setDirectory(CustomerTransfer::PATH_XML_FILES_CTR, $customer_transfer->company_id) . '/';
                if($customer_transfer->document_type2 == 'invoice'){
                    $path_xml = Helper::setDirectory(CustomerTransfer::PATH_XML_FILES_CI, $customer_transfer->company_id) . '/';
                }
                $file_xml_pac = $path_xml . $customer_transfer->customerInvoiceCfdi->file_xml_pac;
                if (!empty($customer_transfer->customerInvoiceCfdi->file_xml_pac) && \Storage::exists($file_xml_pac)) {

                    $folder = !empty(setting('dropbox_folder','')) ? setting('dropbox_folder','') . '/' : setting('dropbox_folder','');
                    $folder .= strtoupper(\Date::parse($customer_transfer->date)->format('F Y'));
                    $folder .= '/' .preg_replace("/[^a-zA-Z0-9\s]/", '', $customer_transfer->company->name);

                    //Guarda XML
                    if($save_xml) {
                        \Storage::disk('dropbox')->putFileAs($folder,
                            new \Illuminate\Http\File(\Storage::path($file_xml_pac)),
                            str_replace('/','',$customer_transfer->name) . '.xml');
                    }

                    //Guarda PDF
                    if($save_pdf) {
                        $pdf = $this->print($customer_transfer, false, true);
                        \Storage::disk('dropbox')->put($folder . '/' . str_replace('/','',$customer_transfer->name) . '.pdf', $pdf);
                    }
                }


            }
        }catch (\Exception $e){
            flash($e->getMessage())->error();
        }
    }

    private function saveCfdiDownloads($customer_transfer, $customer_transfer_cfdi){
        //Directorio de XML
        $path_files = Helper::setDirectory(CfdiDownload::PATH_FILES, $customer_transfer->company->id);
        $tmp_path = str_replace(['files/', '/xml'], '', CustomerInvoice::PATH_XML_FILES_CTR);
        if($customer_transfer->document_type2 == 'invoice'){
            $tmp_path = str_replace(['files/', '/xml'], '', CustomerInvoice::PATH_XML_FILES_CI);
        }
        if (!\Storage::exists($path_files . '/' . $tmp_path)) {
            \Storage::makeDirectory($path_files . '/' . $tmp_path, 0777, true, true);
        }

        //Genera PDF
        $pdf = $this->print($customer_transfer, false, true);
        $path_xml = Helper::setDirectory(CustomerInvoice::PATH_XML_FILES_CTR, $customer_transfer->company_id) . '/';
        if($customer_transfer->document_type2 == 'invoice'){
            $path_xml = Helper::setDirectory(CustomerInvoice::PATH_XML_FILES_CI, $customer_transfer->company_id) . '/';
        }
        $file_xml = $tmp_path . '/' . $customer_transfer_cfdi->uuid . '.xml';
        $file_pdf = str_replace('.xml', '.pdf', $file_xml);
        \Storage::put($path_files . '/' . $file_pdf, $pdf);

        if(!empty($customer_transfer_cfdi->file_xml_pac)){
            \Storage::copy($path_xml . $customer_transfer_cfdi->file_xml_pac, $path_files . '/' . $file_xml);
        }

        //Guarda registro en CFDIS descargados
        $cfdi_download = CfdiDownload::create([
            'created_uid' => \Auth::user()->id,
            'updated_uid' => \Auth::user()->id,
            'type' => 2,
            'uuid' => $customer_transfer_cfdi->uuid,
            'file_xml' => $file_xml,
            'file_pdf' => $file_pdf,
            'file_acuse' => null,
            'status' => 'Vigente',
            'is_cancelable' => 'Cancelable sin aceptación',
            'date_cancel' => null,
            'company_id' => $customer_transfer->company_id,
        ]);
        $data_xml = Helper::parseXmlToArrayCfdi33($path_xml . '/' . $customer_transfer_cfdi->file_xml_pac);
        $data_xml['data']['customer_id'] = $customer_transfer->customer_id;
        $data_xml['data']['caccounting_type_id'] = $customer_transfer->customer->caccounting_type_sale_id ?? null;
        $cfdi_download->fill($data_xml['data']);
        $cfdi_download->save();
    }

    public function downloadTemplateTransferImport()
    {
        //Descargar archivo
        while (ob_get_level()) ob_end_clean();
        ob_start();

        return Excel::download(new CustomerTransferTemplateTransferImportExport(), __('sales/customer_transfer.text_template_transfer_import') . '.xlsx',\Maatwebsite\Excel\Excel::XLSX);
    }

    public function transferImport(Request $request){
        //Validaciones
        //Validacion de archivos por extension
        if ($request->hasFile('file_transfer_import')) {
            $request->merge(['file_transfer_import_ext' => request()->file('file_transfer_import')->getClientOriginalExtension()]);
        }
        $validator = \Validator::make($request->all(), [
            'file_transfer_import' => 'required|max:2048',
            'file_transfer_import_ext' => 'nullable|in:xls,xlsx'
        ], [
            'file_transfer_import.*' => __('sales/customer_transfer.error_file_transfer_import'),
            'file_transfer_import_ext.*' => __('sales/customer_transfer.error_file_transfer_import'),
        ]);
        //Errores
        if ($validator->fails()) {
            $errors = '';
            foreach ($validator->errors()->all() as $message) {
                $errors = $message;
            }
            return response()->json(['error' => $errors], 422);
        }

        $import = new CustomerTransferTemplateTransferImportImport;
        Excel::import($import, request()->file('file_transfer_import'));

        $rows = $import->rows;
        //Validacion
        foreach ($rows as $key => $row) {
            $num_row = $key + 2;

            \Validator::make($row->toArray(), [
                'bienes_transportados' => 'required',
                'cantidad' => 'required|numeric|min:0.00001',
                'u_m' => 'required',
                'peso_en_kg' => 'required|numeric|min:0.00001',
                'moneda' => 'required',
                'valor' => 'required|numeric|min:0.00001',
            ], [
                'bienes_transportados.*' => __('sales/customer_transfer.error_transfer_sat_product') . sprintf(__('sales/customer_transfer.error_row'), $num_row,$row['bienes_transportados']),
                'cantidad.*' => __('sales/customer_transfer.error_transfer_quantity') . sprintf(__('sales/customer_transfer.error_row'), $num_row,$row['cantidad']),
                'u_m.*' => __('sales/customer_transfer.error_transfer_unit_measure_id') . sprintf(__('sales/customer_transfer.error_row'), $num_row,$row['u_m']),
                'peso_en_kg.*' => __('sales/customer_transfer.error_transfer_weight') . sprintf(__('sales/customer_transfer.error_row'), $num_row,$row['peso_en_kg']),
                'moneda.*' => __('sales/customer_transfer.error_transfer_currency_id') . sprintf(__('sales/customer_transfer.error_row'), $num_row,$row['moneda']),
                'valor.*' => __('sales/customer_transfer.error_transfer_weight') . sprintf(__('sales/customer_transfer.error_row'), $num_row,$row['valor']),
            ])->validate();
        }

        $results = [];
        foreach ($rows as $key => $row) {

            $unit_measure = null;
            $sat_product = null;
            $currency = null;
            $packaging_type = null;
            $dangerous_product = null;
            $tariff_fraction = null;

            try{
                if (!empty($row['u_m'])) {
                    $tpm = explode('-',$row['u_m']);
                    $unit_measure = UnitMeasure::where('code', '=', $tpm[0])->first();
                    if (empty($unit_measure)) {
                        throw new \Exception(__('sales/customer_transfer.error_transfer_unit_measure_id2') . sprintf(__('sales/customer_transfer.error_row'), $num_row,$tpm[0]));
                    }
                }

                if (!empty($row['bienes_transportados'])) {
                    $sat_product = SatProduct::where('code', '=', $row['bienes_transportados'])->first();
                    if (empty($sat_product)) {
                        throw new \Exception(__('sales/customer_transfer.error_transfer_sat_product_id2') . sprintf(__('sales/customer_transfer.error_row'), $num_row,$row['bienes_transportados']));
                    }
                }

                if (!empty($row['moneda'])) {
                    $currency = Currency::where('code', '=', $row['moneda'])->first();
                    if (empty($currency)) {
                        throw new \Exception(__('sales/customer_transfer.error_transfer_currency_id2') . sprintf(__('sales/customer_transfer.error_row'), $num_row,$row['moneda']));
                    }
                }

                if (!empty($row['embalaje'])) {
                    $tpm = explode('-',$row['embalaje']);
                    $packaging_type = PackagingType::where('code', '=', $tpm[0])->first();
                    if (empty($packaging_type)) {
                        throw new \Exception(__('sales/customer_transfer.error_transfer_packaging_type_id2') . sprintf(__('sales/customer_transfer.error_row'), $num_row,$tpm[0]));
                    }
                }

                if (!empty($row['tipo_material_peligroso'])) {
                    $tpm = explode('-',$row['tipo_material_peligroso']);
                    $dangerous_product = DangerousProduct::where('code', '=', $tpm[0])->first();
                    if (empty($dangerous_product)) {
                        throw new \Exception(__('sales/customer_transfer.error_transfer_dangerous_product_id2') . sprintf(__('sales/customer_transfer.error_row'), $num_row,$tpm[0]));
                    }
                }

                if (!empty($row['fracc_arancelaria'])) {
                    $tpm = explode('-',$row['fracc_arancelaria']);
                    $tariff_fraction = TariffFraction::where('code', '=', $tpm[0])->first();
                    if (empty($tariff_fraction)) {
                        throw new \Exception(__('sales/customer_transfer.error_transfer_tariff_fraction_id2') . sprintf(__('sales/customer_transfer.error_row'), $num_row,$tpm[0]));
                    }
                }
            }catch(\Exception $e){
                return response()->json(['error' => $e->getMessage()], 422);
            }

            $results[] = [
                'sat_product_id' => $sat_product->id,
                'sat_product' => $sat_product->name_sat,
                'name' => !empty($row['descripcion']) ? $row['descripcion'] : $sat_product->name_sat,
                'quantity' => (float)$row['cantidad'],
                'unit_measure_id' => $unit_measure->id,
                'unit_measure' => $unit_measure->name_sat,
                'weight' => (float)$row['peso_en_kg'],
                'dangerous_product' => !empty($row['material_peligroso']) ? ucfirst(mb_strtolower($row['material_peligroso'])) : '',
                'packaging_type_id' => $packaging_type->id ?? null,
                'dangerous_product_id' => $dangerous_product->id ?? null,
                'dangerous_product2' => $dangerous_product->name_sat ?? null,
                'dimensions' => $row['dimensiones'],
                'amount_total' => (float)$row['valor'],
                'currency_id' => $currency->id,
                'tariff_fraction_id' => $tariff_fraction->id ?? null,
                'tariff_fraction' => $tariff_fraction->name_sat ?? null,
                'uuid_ext' => $row['uuid'],
            ];
        }

        //Mensaje
        return response()->json([
            'success' => 'ok',
            'results' => $results,
        ]);
    }

}
