<?php

namespace App\Http\Controllers\Sales;

use App\Helpers\Helper;
use App\Models\Base\Pac;
use App\Helpers\PacHelper;
use App\Helpers\BaseHelper;
use Illuminate\Support\Str;
use App\Models\Catalogs\Tax;
use Illuminate\Http\Request;
use App\Helpers\Cfdi33Helper;
use App\Models\Sales\Customer;
use App\Models\Catalogs\CfdiUse;
use App\Models\Catalogs\Product;
use App\Models\Catalogs\Project;
use App\Models\Base\BranchOffice;
use App\Models\Catalogs\Currency;
use App\Models\Sales\Salesperson;
use App\Models\Sales\CustomerIedu;
use App\Models\Catalogs\PaymentWay;
use App\Models\Catalogs\SatProduct;
use App\Models\Catalogs\TaxRegimen;
use App\Http\Controllers\Controller;
use App\Models\Catalogs\PaymentTerm;
use App\Models\Catalogs\UnitMeasure;
use Illuminate\Support\Facades\Crypt;
use App\Models\Catalogs\PaymentMethod;
use App\Jobs\ProcessRecurringCustomerInvoice;
use App\Models\Sales\RecurringCustomerInvoice;
use Illuminate\Validation\ValidationException;
use App\Models\Sales\RecurringCustomerInvoiceTax;
use App\Models\Sales\RecurringCustomerInvoiceLine;
use App\Models\Sales\RecurringCustomerInvoiceLineComplement;

class RecurringCustomerInvoiceController extends Controller
{

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);
        $salespersons = Salesperson::populateSelect()->pluck('name', 'id');
        $branch_offices = BranchOffice::populateSelect()->pluck('name', 'id');

        //Consulta
        $results = RecurringCustomerInvoice::filter($request->all())
            ->with('customer')
            ->with('salesperson')
            ->with('currency')
            ->sortable(['date_next' => 'desc'])->paginate($limit);

        //Vista
        return view('sales.recurring_customer_invoices.index',
            compact('results', 'salespersons', 'branch_offices'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        $salespersons = Salesperson::populateSelect()->pluck('name', 'id');
        $branch_offices = BranchOffice::populateSelect()->pluck('name', 'id');
        $currencies = Currency::populateSelect()->get()->pluck('name_sat', 'id');
        $payment_terms = PaymentTerm::populateSelect()->pluck('name', 'id');
        $payment_ways = PaymentWay::populateSelect()->get()->pluck('name_sat', 'id');
        $payment_methods = PaymentMethod::populateSelect()->get()->pluck('name_sat', 'id');
        $cfdi_uses = CfdiUse::populateSelect()->get()->pluck('name_sat', 'id');
        $taxes = Tax::populateSelect()->pluck('name', 'id');
        $projects = Project::populateSelect()->pluck('name', 'id');
        $frequencies = config('app.frequencies');
        $company = Helper::defaultCompany(); //Empresa
        if(!empty($company->tax_regimen_id2)){
            $tax_regimens = TaxRegimen::populateSelect()->whereIn('id',[$company->tax_regimen_id, $company->tax_regimen_id2])->get()->pluck('name_sat', 'id');
        }else{
            $tax_regimens = collect([]);
        }
        $tax_regimen_customers = TaxRegimen::populateSelect()->get()->pluck('name_sat', 'id');

        $duplicate_ci = null;
        if($request->duplicate_id){
            $duplicate_ci = RecurringCustomerInvoice::findOrFail($request->duplicate_id);
        }
        $tax_objects = __('general.text_tax_objects');

        return view('sales.recurring_customer_invoices.create',
            compact('salespersons', 'branch_offices', 'currencies', 'payment_ways', 'payment_terms', 'payment_methods',
                'cfdi_uses', 'taxes','duplicate_ci','projects','frequencies','tax_regimens','tax_regimen_customers','tax_objects'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {

        //Validacion
        $this->validation($request);


        \DB::connection('tenant')->beginTransaction();
        try {
            //Logica
            $company = Helper::defaultCompany(); //Empresa
            $request->merge(['created_uid' => \Auth::user()->id]);
            $request->merge(['updated_uid' => \Auth::user()->id]);
            $request->merge(['status' => 1]);
            $request->merge(['company_id' => $company->id]);
            $request->merge(['date_start' => Helper::convertDateToSql($request->date_start)]);
            $date_end = $request->date_end;
            $request->merge(['date_end' => !empty($request->date_end) ? Helper::convertDateToSql($request->date_end) : null]);
            $request->merge(['date_next' => Helper::convertDateToSql($request->date_next)]);
            $request->merge(['mail_sent' => !empty($request->mail_sent) ? 1 : 0]);

            //Guardar
            //Registro principal
            $recurring_customer_invoice = RecurringCustomerInvoice::create($request->input());

            //Registro de lineas
            $amount_discount = 0;
            $amount_untaxed = 0;
            $amount_tax = 0;
            $amount_tax_ret = 0;
            $amount_total = 0;
            $balance = 0;
            $taxes = array();
            //Lineas
            if (!empty($request->item)) {
                foreach ($request->item as $key => $item) {
                    //Logica
                    $item_quantity = (double)$item['quantity'];
                    $item_price_unit = (double)$item['price_unit'];
                    $item_quota_ieps = (double)$item['quota_ieps'];
                    //Ajuste para cuando tiene IVA incluido
                    if (!empty($item['taxes']) && !empty($item['includes_iva'])) {
                        foreach ($item['taxes'] as $tax_id) {
                            if (!empty($tax_id)) {
                                $tax = Tax::findOrFail($tax_id);
                                if($tax->code == '002' && $tax->rate>0) {
                                    if ($tax->factor == 'Tasa') {
                                        $item_price_unit = $item_price_unit / (1+($tax->rate/100));
                                    } elseif ($tax->factor == 'Cuota') {
                                        $item_price_unit -= $tax->rate;
                                    }
                                    $item_price_unit = round($item_price_unit, \App\Helpers\Helper::companyProductPriceDecimalPlace());
                                    $input_items[$key]['price_unit'] = $item_price_unit;
                                }
                            }
                            break;
                        }
                    }
                    $item_discount = (double)$item['discount'];
                    $item_price_reduce = ($item_price_unit * (100 - $item_discount) / 100);
                    $item_amount_untaxed = round($item_quantity * $item_price_reduce, 2);
                    if($item_quota_ieps > 0){
                        $item_amount_untaxed_per_taxes = round($item_quantity * (($item_price_unit * (100 - $item_discount) / 100) - $item_quota_ieps), 2);
                    }else{
                        $item_amount_untaxed_per_taxes = $item_amount_untaxed;
                    }
                    $item_amount_discount = round($item_quantity * $item_price_unit, 2) - $item_amount_untaxed;
                    $item_amount_tax = 0;
                    $item_amount_tax_ret = 0;
                    if (!empty($item['taxes'])) {
                        foreach ($item['taxes'] as $tax_id) {
                            if (!empty($tax_id)) {
                                $tax = Tax::findOrFail($tax_id);
                                $tmp = 0;
                                if ($tax->factor == 'Tasa') {
                                    $tmp = $item_amount_untaxed_per_taxes * $tax->rate / 100;
                                } elseif ($tax->factor == 'Cuota') {
                                    $tmp = $tax->rate;
                                }
                                $tmp = round($tmp, 2);
                                if ($tax->type == 'R') { //Retenciones
                                    $item_amount_tax_ret += $tmp;
                                } else { //Traslados
                                    $item_amount_tax += $tmp;
                                }

                                //Sumatoria de impuestos
                                $taxes[$tax_id] = array(
                                    'amount_base' => $item_amount_untaxed_per_taxes + (isset($taxes[$tax_id]['amount_base']) ? $taxes[$tax_id]['amount_base'] : 0),
                                    'amount_tax' => $tmp + (isset($taxes[$tax_id]['amount_tax']) ? $taxes[$tax_id]['amount_tax'] : 0),
                                );
                            }
                        }
                    }
                    $item_amount_total = $item_amount_untaxed + $item_amount_tax + $item_amount_tax_ret;
                    //Sumatoria totales
                    $amount_discount += $item_amount_discount;
                    $amount_untaxed += $item_amount_untaxed;
                    $amount_tax += $item_amount_tax;
                    $amount_tax_ret += $item_amount_tax_ret;
                    $amount_total += $item_amount_total;

                    //Guardar linea
                    $recurring_customer_invoice_line = RecurringCustomerInvoiceLine::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'recurring_customer_invoice_id' => $recurring_customer_invoice->id,
                        'name' => $item['name'],
                        'product_id' => $item['product_id'],
                        'sat_product_id' => $item['sat_product_id'],
                        'unit_measure_id' => $item['unit_measure_id'],
                        'quantity' => $item_quantity,
                        'price_unit' => $item_price_unit,
                        'discount' => $item_discount,
                        'price_reduce' => $item_price_reduce,
                        'amount_discount' => $item_amount_discount,
                        'amount_untaxed' => $item_amount_untaxed,
                        'amount_tax' => $item_amount_tax,
                        'amount_tax_ret' => $item_amount_tax_ret,
                        'amount_total' => $item_amount_total,
                        'sort_order' => $key,
                        'status' => 1,
                        'quota_ieps' => $item_quota_ieps,
                        'tax_object' => !empty($item['tax_object']) ? $item['tax_object'] : (empty($item['taxes']) ? '01' : '02'),
                    ]);

                    //Guardar impuestos por linea
                    if (!empty($item['taxes'])) {
                        $recurring_customer_invoice_line->taxes()->sync($item['taxes']);
                    } else {
                        $recurring_customer_invoice_line->taxes()->sync([]);
                    }

                    //Datos de complementos IEDU
                    if(!empty($item['customer_iedu_id'])){
                        $customer_iedu = CustomerIedu::findOrFail($item['customer_iedu_id']);
                        $item['iedu_nombre_alumno'] = $customer_iedu->nombre_alumno;
                        $item['iedu_curp'] = $customer_iedu->curp;
                        $item['iedu_nivel_educativo'] = $customer_iedu->ieduNivelEducativo->code;
                        $item['iedu_aut_rvoe'] = $customer_iedu->ieduNivelEducativo->aut_rvoe;
                    }
                    $recurring_customer_invoice_line_complement = RecurringCustomerInvoiceLineComplement::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'recurring_customer_invoice_line_id' => $recurring_customer_invoice_line->id,
                        'name' => $item['name'],
                        'customer_iedu_id' => !empty($item['customer_iedu_id']) ? $item['customer_iedu_id'] : null,
                        'iedu_nombre_alumno' => !empty($item['iedu_nombre_alumno']) ? $item['iedu_nombre_alumno'] : '',
                        'iedu_curp' => !empty($item['iedu_curp']) ? $item['iedu_curp'] : '',
                        'iedu_nivel_educativo' => !empty($item['iedu_nivel_educativo']) ? $item['iedu_nivel_educativo'] : '',
                        'iedu_aut_rvoe' => !empty($item['iedu_aut_rvoe']) ? $item['iedu_aut_rvoe'] : '',
                        'iedu_rfc_pago' => !empty($item['iedu_rfc_pago']) ? $item['iedu_rfc_pago'] : '',
                        'sort_order' => $key,
                        'status' => 1,
                    ]);
                }
            }

            //Resumen de impuestos
            if (!empty($taxes)) {
                $i = 0;
                foreach ($taxes as $tax_id => $result) {
                    $tax = Tax::findOrFail($tax_id);
                    $recurring_customer_invoice_tax = RecurringCustomerInvoiceTax::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'recurring_customer_invoice_id' => $recurring_customer_invoice->id,
                        'name' => $tax->name,
                        'tax_id' => $tax_id,
                        'amount_base' => $result['amount_base'],
                        'amount_tax' => $result['amount_tax'],
                        'sort_order' => $i,
                        'status' => 1,
                    ]);
                    $i++;
                }
            }

            //Actualiza registro principal con totales
            $recurring_customer_invoice->amount_discount = $amount_discount;
            $recurring_customer_invoice->amount_untaxed = $amount_untaxed;
            $recurring_customer_invoice->amount_tax = $amount_tax;
            $recurring_customer_invoice->amount_tax_ret = $amount_tax_ret;
            $recurring_customer_invoice->amount_total = $amount_total;
            $recurring_customer_invoice->name = $recurring_customer_invoice->id;
            $recurring_customer_invoice->update();

            //Mensaje
            flash(__('general.text_success_recurring_customer_invoice'))->success();

            \DB::connection('tenant')->commit();

        } catch (\Exception $e) {
            //Fix fechas
            $request->merge([
                'date_start' => Helper::convertSqlToDate($request->date_start),
            ]);
            if (!empty($date_end)) {
                $request->merge([
                    'date_end' => $date_end,
                ]);
            }
            $request->merge([
                'date_next' => Helper::convertSqlToDate($request->date_next),
            ]);

            \DB::connection('tenant')->rollback();
            flash($e->getMessage())->error();
            return back()->withInput();
        }

        //Redireccion
        return redirect('/sales/recurring-customer-invoices');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Sales\RecurringCustomerInvoice $recurring_customer_invoice
     * @return \Illuminate\Http\Response
     */
    public function show(RecurringCustomerInvoice $recurring_customer_invoice)
    {
        //Datos
        $data = [];

        return view('sales.recurring_customer_invoices.show', compact('recurring_customer_invoice','data'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Sales\RecurringCustomerInvoice $recurring_customer_invoice
     * @return \Illuminate\Http\Response
     */
    public function edit(RecurringCustomerInvoice $recurring_customer_invoice)
    {
        $salespersons = Salesperson::populateSelect()->pluck('name', 'id');
        $branch_offices = BranchOffice::populateSelect()->pluck('name', 'id');
        $currencies = Currency::populateSelect()->get()->pluck('name_sat', 'id');
        $payment_terms = PaymentTerm::populateSelect()->pluck('name', 'id');
        $payment_ways = PaymentWay::populateSelect()->get()->pluck('name_sat', 'id');
        $payment_methods = PaymentMethod::populateSelect()->get()->pluck('name_sat', 'id');
        $cfdi_uses = CfdiUse::populateSelect()->get()->pluck('name_sat', 'id');
        $taxes = Tax::populateSelect()->pluck('name', 'id');
        $projects = Project::populateSelect()->pluck('name', 'id');
        $frequencies = config('app.frequencies');
        $company = Helper::defaultCompany(); //Empresa
        if(!empty($company->tax_regimen_id2)){
            $tax_regimens = TaxRegimen::populateSelect()->whereIn('id',[$company->tax_regimen_id, $company->tax_regimen_id2])->get()->pluck('name_sat', 'id');
        }else{
            $tax_regimens = collect([]);
        }
        $tax_regimen_customers = TaxRegimen::populateSelect()->get()->pluck('name_sat', 'id');
        $tax_objects = __('general.text_tax_objects');

        return view('sales.recurring_customer_invoices.edit',
            compact('recurring_customer_invoice','salespersons', 'branch_offices', 'currencies', 'payment_ways', 'payment_terms', 'payment_methods',
                'cfdi_uses', 'taxes','projects','frequencies','tax_regimens','tax_regimen_customers','tax_objects'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @param  \App\Models\Sales\RecurringCustomerInvoice $recurring_customer_invoice
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, RecurringCustomerInvoice $recurring_customer_invoice)
    {
        //Validacion
        $this->validation($request);

        \DB::connection('tenant')->beginTransaction();
        try {

            //Logica
            $request->merge(['created_uid' => \Auth::user()->id]);
            $request->merge(['updated_uid' => \Auth::user()->id]);
            $request->merge(['date_start' => Helper::convertDateToSql($request->date_start)]);
            $date_end = $request->date_end;
            $request->merge(['date_end' => !empty($request->date_end) ? Helper::convertDateToSql($request->date_end) : null]);
            $request->merge(['date_next' => Helper::convertDateToSql($request->date_next)]);
            $request->merge(['mail_sent' => !empty($request->mail_sent) ? 1 : 0]);
            $recurring_customer_invoice->fill($request->only([
                'updated_uid',
                'date_start',
                'date_end',
                'date_next',
                'reference',
                'customer_id',
                'branch_office_id',
                'payment_term_id',
                'payment_way_id',
                'payment_method_id',
                'cfdi_use_id',
                'salesperson_id',
                'currency_id',
                'currency_value',
                'comment',
                'title',
                'project_id',
                'mail_sent',
                'tax_regimen_id',
                'tax_regimen_customer_id'
            ]));

            //Guardar
            //Registro principal
            $recurring_customer_invoice->save();

            //Elimina partidas
            if (!empty($request->delete_item)) {
                foreach ($request->delete_item as $key => $result) {
                    //Actualizar status
                    $recurring_customer_invoice_line = RecurringCustomerInvoiceLine::findOrFail($result);
                    $recurring_customer_invoice_line->updated_uid = \Auth::user()->id;
                    $recurring_customer_invoice_line->status = 0;
                    $recurring_customer_invoice_line->save();
                }
            }
            //Registro de lineas
            $amount_discount = 0;
            $amount_untaxed = 0;
            $amount_tax = 0;
            $amount_tax_ret = 0;
            $amount_total = 0;
            $balance = 0;
            $taxes = array();
            //Lineas
            if (!empty($request->item)) {
                foreach ($request->item as $key => $item) {
                    //Logica
                    $item_quantity = (double)$item['quantity'];
                    $item_price_unit = (double)$item['price_unit'];
                    $item_quota_ieps = (double)$item['quota_ieps'];
                    //Ajuste para cuando tiene IVA incluido
                    if (!empty($item['taxes']) && !empty($item['includes_iva'])) {
                        foreach ($item['taxes'] as $tax_id) {
                            if (!empty($tax_id)) {
                                $tax = Tax::findOrFail($tax_id);
                                if($tax->code == '002' && $tax->rate>0) {
                                    if ($tax->factor == 'Tasa') {
                                        $item_price_unit = $item_price_unit / (1+($tax->rate/100));
                                    } elseif ($tax->factor == 'Cuota') {
                                        $item_price_unit -= $tax->rate;
                                    }
                                    $item_price_unit = round($item_price_unit, \App\Helpers\Helper::companyProductPriceDecimalPlace());
                                    $input_items[$key]['price_unit'] = $item_price_unit;
                                }
                            }
                            break;
                        }
                    }
                    $item_discount = (double)$item['discount'];
                    $item_price_reduce = ($item_price_unit * (100 - $item_discount) / 100);
                    $item_amount_untaxed = round($item_quantity * $item_price_reduce, 2);
                    if($item_quota_ieps > 0){
                        $item_amount_untaxed_per_taxes = round($item_quantity * (($item_price_unit * (100 - $item_discount) / 100) - $item_quota_ieps), 2);
                    }else{
                        $item_amount_untaxed_per_taxes = $item_amount_untaxed;
                    }
                    $item_amount_discount = round($item_quantity * $item_price_unit, 2) - $item_amount_untaxed;
                    $item_amount_tax = 0;
                    $item_amount_tax_ret = 0;
                    if (!empty($item['taxes'])) {
                        foreach ($item['taxes'] as $tax_id) {
                            if (!empty($tax_id)) {
                                $tax = Tax::findOrFail($tax_id);
                                $tmp = 0;
                                if ($tax->factor == 'Tasa') {
                                    $tmp = $item_amount_untaxed_per_taxes * $tax->rate / 100;
                                } elseif ($tax->factor == 'Cuota') {
                                    $tmp = $tax->rate;
                                }
                                $tmp = round($tmp, 2);
                                if ($tax->type == 'R') { //Retenciones
                                    $item_amount_tax_ret += $tmp;
                                } else { //Traslados
                                    $item_amount_tax += $tmp;
                                }

                                //Sumatoria de impuestos
                                $taxes[$tax_id] = array(
                                    'amount_base' => $item_amount_untaxed_per_taxes + (isset($taxes[$tax_id]['amount_base']) ? $taxes[$tax_id]['amount_base'] : 0),
                                    'amount_tax' => $tmp + (isset($taxes[$tax_id]['amount_tax']) ? $taxes[$tax_id]['amount_tax'] : 0),
                                );
                            }
                        }
                    }
                    $item_amount_total = $item_amount_untaxed + $item_amount_tax + $item_amount_tax_ret;
                    //Sumatoria totales
                    $amount_discount += $item_amount_discount;
                    $amount_untaxed += $item_amount_untaxed;
                    $amount_tax += $item_amount_tax;
                    $amount_tax_ret += $item_amount_tax_ret;
                    $amount_total += $item_amount_total;

                    //Guardar linea
                    $data = [
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'recurring_customer_invoice_id' => $recurring_customer_invoice->id,
                        'name' => $item['name'],
                        'product_id' => $item['product_id'],
                        'sat_product_id' => $item['sat_product_id'],
                        'unit_measure_id' => $item['unit_measure_id'],
                        'quantity' => $item_quantity,
                        'price_unit' => $item_price_unit,
                        'discount' => $item_discount,
                        'price_reduce' => $item_price_reduce,
                        'amount_discount' => $item_amount_discount,
                        'amount_untaxed' => $item_amount_untaxed,
                        'amount_tax' => $item_amount_tax,
                        'amount_tax_ret' => $item_amount_tax_ret,
                        'amount_total' => $item_amount_total,
                        'sort_order' => $key,
                        'status' => 1,
                        'quota_ieps' => $item_quota_ieps,
                        'tax_object' => !empty($item['tax_object']) ? $item['tax_object'] : (empty($item['taxes']) ? '01' : '02'),
                    ];
                    if (!empty($item['id'])) {
                        $recurring_customer_invoice_line = RecurringCustomerInvoiceLine::findOrFail($item['id']);
                        $recurring_customer_invoice_line->fill(array_only($data, [
                            'updated_uid',
                            'name',
                            'product_id',
                            'sat_product_id',
                            'unit_measure_id',
                            'quantity',
                            'price_unit',
                            'discount',
                            'price_reduce',
                            'amount_discount',
                            'amount_untaxed',
                            'amount_tax',
                            'amount_tax_ret',
                            'amount_total',
                            'sort_order',
                            'quota_ieps',
                            'tax_object'
                        ]));
                        $recurring_customer_invoice_line->save();
                    }else{
                        $recurring_customer_invoice_line = RecurringCustomerInvoiceLine::create($data);
                    }

                    //Guardar impuestos por linea
                    if (!empty($item['taxes'])) {
                        $recurring_customer_invoice_line->taxes()->sync($item['taxes']);
                    } else {
                        $recurring_customer_invoice_line->taxes()->sync([]);
                    }

                    //Datos de complementos IEDU
                    if(!empty($item['customer_iedu_id'])){
                        $customer_iedu = CustomerIedu::findOrFail($item['customer_iedu_id']);
                        $item['iedu_nombre_alumno'] = $customer_iedu->nombre_alumno;
                        $item['iedu_curp'] = $customer_iedu->curp;
                        $item['iedu_nivel_educativo'] = $customer_iedu->ieduNivelEducativo->code;
                        $item['iedu_aut_rvoe'] = $customer_iedu->ieduNivelEducativo->aut_rvoe;
                    }
                    $recurring_customer_invoice_line_complement = RecurringCustomerInvoiceLineComplement::where('recurring_customer_invoice_line_id','=',$recurring_customer_invoice_line->id)->first();
                    $data_complement = [
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'recurring_customer_invoice_line_id' => $recurring_customer_invoice_line->id,
                        'name' => $item['name'],
                        'customer_iedu_id' => !empty($item['customer_iedu_id']) ? $item['customer_iedu_id'] : null,
                        'iedu_nombre_alumno' => !empty($item['iedu_nombre_alumno']) ? $item['iedu_nombre_alumno'] : '',
                        'iedu_curp' => !empty($item['iedu_curp']) ? $item['iedu_curp'] : '',
                        'iedu_nivel_educativo' => !empty($item['iedu_nivel_educativo']) ? $item['iedu_nivel_educativo'] : '',
                        'iedu_aut_rvoe' => !empty($item['iedu_aut_rvoe']) ? $item['iedu_aut_rvoe'] : '',
                        'iedu_rfc_pago' => !empty($item['iedu_rfc_pago']) ? $item['iedu_rfc_pago'] : '',
                        'sort_order' => $key,
                        'status' => 1,
                    ];
                    if(!empty($recurring_customer_invoice_line_complement)){
                        $recurring_customer_invoice_line_complement->fill(array_only($data_complement, [
                            'updated_uid',
                            'name',
                            'customer_iedu_id',
                            'iedu_nombre_alumno',
                            'iedu_curp',
                            'iedu_nivel_educativo',
                            'iedu_aut_rvoe',
                            'iedu_rfc_pago',
                            'sort_order',
                        ]));
                        $recurring_customer_invoice_line_complement->save();
                    }else{
                        $recurring_customer_invoice_line_complement = RecurringCustomerInvoiceLineComplement::create($data_complement);
                    }
                }
            }

            //Resumen de impuestos
            RecurringCustomerInvoiceTax::where('recurring_customer_invoice_id','=',$recurring_customer_invoice->id)->delete(); //Borra todo e inserta nuevamente
            if (!empty($taxes)) {
                $i = 0;
                foreach ($taxes as $tax_id => $result) {
                    $tax = Tax::findOrFail($tax_id);
                    $recurring_customer_invoice_tax = RecurringCustomerInvoiceTax::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'recurring_customer_invoice_id' => $recurring_customer_invoice->id,
                        'name' => $tax->name,
                        'tax_id' => $tax_id,
                        'amount_base' => $result['amount_base'],
                        'amount_tax' => $result['amount_tax'],
                        'sort_order' => $i,
                        'status' => 1,
                    ]);
                    $i++;
                }
            }

            //Actualiza registro principal con totales
            $recurring_customer_invoice->amount_discount = $amount_discount;
            $recurring_customer_invoice->amount_untaxed = $amount_untaxed;
            $recurring_customer_invoice->amount_tax = $amount_tax;
            $recurring_customer_invoice->amount_tax_ret = $amount_tax_ret;
            $recurring_customer_invoice->amount_total = $amount_total;
            $recurring_customer_invoice->update();

            //Mensaje
            flash(__('general.text_success_recurring_customer_invoice'))->success();

            \DB::connection('tenant')->commit();

        } catch (\Exception $e) {
            //Fix fechas
            $request->merge([
                'date_start' => Helper::convertSqlToDate($request->date_start),
            ]);
            if (!empty($date_end)) {
                $request->merge([
                    'date_end' => $date_end,
                ]);
            }
            $request->merge([
                'date_next' => Helper::convertSqlToDate($request->date_next),
            ]);

            \DB::connection('tenant')->rollback();
            flash($e->getMessage())->error();
            return back()->withInput();
        }

        //Redireccion
        return redirect('/sales/recurring-customer-invoices');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Sales\RecurringCustomerInvoice $recurring_customer_invoice
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, RecurringCustomerInvoice $recurring_customer_invoice)
    {
        //Logica
        $recurring_customer_invoice->updated_uid = \Auth::user()->id;
        $recurring_customer_invoice->status = 0;
        $recurring_customer_invoice->save();

        //Mensaje
        flash(__('general.text_form_success_cancel'))->success();

        //Redireccion
        return redirect('/sales/recurring-customer-invoices');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {

        $this->validate($request, [
            'customer_id' => 'required|integer',
            'branch_office_id' => 'required|integer',
            'date_start' => 'required|date|date_format:"'.setting('date_format', 'd-m-Y H:i:s').'"',
            'date_end' => 'nullable|date|date_format:"'.setting('date_format', 'd-m-Y').'"',
            'date_next' => 'required|date|date_format:"'.setting('date_format', 'd-m-Y H:i:s').'"',
            'frequency' => 'required',
            'currency_id' => 'required|integer',
            'currency_value' => 'required|numeric|min:0.1',
            'payment_term_id' => 'required|integer',
            'payment_way_id' => 'required|integer',
            'payment_method_id' => 'required|integer',
            'cfdi_use_id' => 'required|integer',
            'item' => 'required',
            'item.*.name' => 'required',
            'item.*.unit_measure_id' => 'required',
            'item.*.sat_product_id' => 'required',
            'item.*.quantity' => 'required|numeric|min:0.00001',
            'item.*.price_unit' => 'required|numeric|min:0.00001',
            'item.*.discount' => 'nullable|numeric|min:0',
        ], [
            'customer_id.*' => __('sales/recurring_customer_invoice.error_customer_id'),
            'branch_office_id.*' => __('sales/recurring_customer_invoice.error_branch_office_id'),
            'date_start.required' => __('sales/recurring_customer_invoice.error_date_start'),
            'date_start.date' => __('sales/recurring_customer_invoice.error_date_start_format'),
            'date_start.date_format' => __('sales/recurring_customer_invoice.error_date_start_format'),
            'date_end.date' => __('sales/recurring_customer_invoice.error_date_end_format'),
            'date_end.date_format' => __('sales/recurring_customer_invoice.error_date_end_format'),
            'date_next.required' => __('sales/recurring_customer_invoice.error_date_next'),
            'date_next.date' => __('sales/recurring_customer_invoice.error_date_next_format'),
            'date_next.date_format' => __('sales/recurring_customer_invoice.error_date_next_format'),
            'frequency.*' => __('sales/recurring_customer_invoice.error_frequency'),
            'currency_id.*' => __('sales/recurring_customer_invoice.error_currency_id'),
            'currency_value.*' => __('sales/recurring_customer_invoice.error_currency_value'),
            'payment_term_id.*' => __('sales/recurring_customer_invoice.error_payment_term_id'),
            'payment_way_id.*' => __('sales/recurring_customer_invoice.error_payment_way_id'),
            'payment_method_id.*' => __('sales/recurring_customer_invoice.error_payment_method_id'),
            'cfdi_use_id.*' => __('sales/recurring_customer_invoice.error_cfdi_use_id'),
            'item.required' => __('sales/recurring_customer_invoice.error_item'),
            'item.*.name.*' => __('sales/recurring_customer_invoice.error_line_name'),
            'item.*.unit_measure_id.*' => __('sales/recurring_customer_invoice.error_line_unit_measure_id'),
            'item.*.sat_product_id.*' => __('sales/recurring_customer_invoice.error_line_sat_product_id'),
            'item.*.quantity.*' => __('sales/recurring_customer_invoice.error_line_quantity'),
            'item.*.price_unit.*' => __('sales/recurring_customer_invoice.error_line_price_unit'),
        ]);

        //Validacion complemento IEDU
        if(\App\Helpers\Helper::companyComplementoCfdiIedu()){
            $this->validate($request, [
                'item.*.customer_iedu_id' => 'required',
                'item.*.iedu_rfc_pago' => ['nullable', 'regex:/^[A-Z&Ñ]{3,4}[0-9]{2}(0[1-9]|1[012])(0[1-9]|[12][0-9]|3[01])[A-Z0-9]{2}[0-9A]$/i'],
            ], [
                'item.*.customer_iedu_id.*' => __('sales/recurring_customer_invoice.error_line_iedu_customer_iedu_id'),
                'item.*.iedu_rfc_pago.*' => __('sales/recurring_customer_invoice.error_line_iedu_rfc_pago_format'),
            ]);
        }

        //Validaciones manuales
        $validator = \Validator::make([], []);
        if(setting('cfdi_version') == 'cfdi40'){
            if (empty($request->tax_regimen_customer_id)) {
                $validator->after(function ($validator) {
                    $validator->errors()->add('tax_regimen_customer_id', __('sales/recurring_customer_invoice.error_tax_regimen_customer_id'));
                });
            }
            $customer = Customer::find($request->customer_id);
            if (empty($customer->postcode) && !in_array($customer->taxid, ['XAXX010101000', 'XEXX010101000'])) {
                $validator->after(function ($validator) {
                    $validator->errors()->add('customer_id', __('sales/recurring_customer_invoice.error_postcode_customer'));
                });
            }
        }

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }
    }

    /**
     * Clase generica de impresion
     *
     * @param RecurringCustomerInvoice $recurring_customer_invoice
     * @return mixed
     */
    public function print(RecurringCustomerInvoice $recurring_customer_invoice, $download = false, $save = false)
    {
        try {
            //Logica
            $tmp = 'default';
            $class_print = 'print' . ucfirst($tmp);
            return $this->$class_print($recurring_customer_invoice, $download, $save);

        } catch (\Exception $e) {
            flash($e->getMessage())->error();
            return redirect('/sales/recurring-customer-invoices');
        }
    }

    /**
     * Impresion default
     *
     * @param $recurring_customer_invoice
     * @return mixed
     */
    private function printDefault($recurring_customer_invoice, $download = false, $save = false)
    {
        //PDF
        $pdf = \PDF::loadView('sales.recurring_customer_invoices.pdf_default', compact('recurring_customer_invoice'));

        //Guardar
        if ($save) {
            return $pdf->output();
        }

        //Descarga
        if($download){
            return $pdf->download(__('sales/recurring_customer_invoice.document_title') . '_' . str_replace('/','',$recurring_customer_invoice->name) . '.pdf');
        }

        //Redireccion
        return $pdf->stream(__('sales/recurring_customer_invoice.document_title') . '_' . str_replace('/','',$recurring_customer_invoice->name) . '.pdf');
    }

    /**
     * Calcula el total de las lineas
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function totalLines(Request $request)
    {

        //Variables
        $json = new \stdClass;
        $input_items = $request->item;
        $currency_id = $request->currency_id;
        $currency_code = 'MXN';
        $taxes = [];

        if ($request->ajax()) {
            //Datos de moneda
            if (!empty($currency_id)) {
                $currency = Currency::findOrFail($currency_id);
                $currency_code = $currency->code;
            }
            //Variables de totales
            $amount_discount = 0;
            $amount_untaxed = 0;
            $amount_tax = 0;
            $amount_tax_ret = 0;
            $amount_total = 0;
            $balance = 0;
            $items = [];
            if (!empty($input_items)) {
                foreach ($input_items as $key => $item) {
                    //Logica
                    $item_quantity = (double)$item['quantity'];
                    $item_price_unit = (double)$item['price_unit'];
                    $item_quota_ieps = (double)$item['quota_ieps'];
                    //Ajuste para cuando tiene IVA incluido
                    if (!empty($item['taxes']) && !empty($item['includes_iva'])) {
                        foreach ($item['taxes'] as $tax_id) {
                            if (!empty($tax_id)) {
                                $tax = Tax::findOrFail($tax_id);
                                if($tax->code == '002' && $tax->rate>0) {
                                    if ($tax->factor == 'Tasa') {
                                        $item_price_unit = $item_price_unit / (1+($tax->rate/100));
                                    } elseif ($tax->factor == 'Cuota') {
                                        $item_price_unit -= $tax->rate;
                                    }
                                    $item_price_unit = round($item_price_unit, \App\Helpers\Helper::companyProductPriceDecimalPlace());
                                    $input_items[$key]['price_unit'] = $item_price_unit;
                                }
                            }
                            break;
                        }
                    }
                    $item_discount = (double)$item['discount'];
                    $item_price_reduce = ($item_price_unit * (100 - $item_discount) / 100);
                    $item_amount_untaxed = round($item_quantity * $item_price_reduce, 2);
                    if($item_quota_ieps > 0){
                        $item_amount_untaxed_per_taxes = round($item_quantity * (($item_price_unit * (100 - $item_discount) / 100) - $item_quota_ieps), 2);
                    }else{
                        $item_amount_untaxed_per_taxes = $item_amount_untaxed;
                    }
                    $item_amount_discount = round($item_quantity * $item_price_unit, 2) - $item_amount_untaxed;
                    $item_amount_tax = 0;
                    $item_amount_tax_ret = 0;

                    if (!empty($item['taxes'])) {
                        foreach ($item['taxes'] as $tax_id) {
                            if (!empty($tax_id)) {
                                $tax = Tax::findOrFail($tax_id);
                                $tmp = 0;
                                if ($tax->factor == 'Tasa') {
                                    $tmp = $item_amount_untaxed_per_taxes * $tax->rate / 100;
                                } elseif ($tax->factor == 'Cuota') {
                                    $tmp = $tax->rate;
                                }
                                $tmp = round($tmp, 2);
                                if ($tax->type == 'R') { //Retenciones
                                    $item_amount_tax_ret += $tmp;
                                } else { //Traslados
                                    $item_amount_tax += $tmp;
                                }

                                //Sumatoria de impuestos
                                $taxes[$tax_id] = array(
                                    'amount_base' => $item_amount_untaxed_per_taxes + (isset($taxes[$tax_id]['amount_base']) ? $taxes[$tax_id]['amount_base'] : 0),
                                    'amount_tax' => $tmp + (isset($taxes[$tax_id]['amount_tax']) ? $taxes[$tax_id]['amount_tax'] : 0),
                                );
                            }
                        }
                    }
                    $item_amount_total = $item_amount_untaxed + $item_amount_tax + $item_amount_tax_ret;
                    //Sumatoria totales
                    $amount_discount += $item_amount_discount;
                    $amount_untaxed += $item_amount_untaxed;
                    $amount_tax += $item_amount_tax;
                    $amount_tax_ret += $item_amount_tax_ret;
                    $amount_total += $item_amount_total;
                    //Subtotales por cada item
                    $items[$key] = money($item_amount_untaxed, $currency_code, true)->format();
                }
            }
            //Respuesta
            $taxes_tmp = [];
            if (!empty($taxes)) {
                foreach ($taxes as $tax_id => $result) {
                    $tax = Tax::findOrFail($tax_id);
                    $taxes_tmp[] = [
                        'name' => $tax->name,
                        'tax_id' => $tax_id,
                        'amount_base' => money($result['amount_base'], $currency_code, true)->format(),
                        'amount_tax' => money($result['amount_tax'], $currency_code, true)->format(),
                    ];
                }
            }

            //
            $json->items = $items;
            $json->taxes = $taxes_tmp;
            $json->amount_discount = money($amount_discount, $currency_code, true)->format();
            $json->amount_untaxed = money($amount_untaxed, $currency_code, true)->format();
            $json->amount_tax = money($amount_tax + $amount_tax_ret, $currency_code, true)->format();
            $json->amount_total = money($amount_total, $currency_code, true)->format();
            $json->amount_total_tmp = $amount_total;
            return response()->json($json);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

}
