<?php

namespace App\Http\Controllers\Sales;

use App\Models\Sales\Salesperson;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class SalespersonController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results = Salesperson::filter($request->all())->sortable('name')->paginate($limit);

        //Vista
        return view('sales.salespersons.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('sales.salespersons.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'email' => 'nullable|email|unique:tenant.salespersons',
        ], [
            'email.email' => __('sales/salesperson.error_email'),
            'email.unique' => __('sales/salesperson.error_email_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $salesperon = Salesperson::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/sales/salespersons');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Sales\Salesperson  $salesperson
     * @return \Illuminate\Http\Response
     */
    public function show(Salesperson $salesperson)
    {
        return redirect('/sales/salespersons');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Sales\Salesperson  $salesperson
     * @return \Illuminate\Http\Response
     */
    public function edit(Salesperson $salesperson)
    {
        return view('sales.salespersons.edit',compact('salesperson'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Sales\Salesperson  $salesperson
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Salesperson $salesperson)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'email' => [
                'nullable',
                'email',
                Rule::unique('tenant.salespersons')->ignore($salesperson->id)
            ],
        ], [
            'email.email' => __('sales/salesperson.error_email'),
            'email.unique' => __('sales/salesperson.error_email_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => 0]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $salesperson->fill($request->only([
            'updated_uid',
            'name',
            'email',
            'phone',
            'phone_mobile',
            'comission_percent',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $salesperson->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/sales/salespersons');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Sales\Salesperson  $salesperson
     * @return \Illuminate\Http\Response
     */
    public function destroy(Salesperson $salesperson)
    {
        //Logica
        $salesperson->updated_uid = \Auth::user()->id;
        $salesperson->status = 0;
        $salesperson->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/sales/salespersons');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required',
        ], [
            'name.*' => __('sales/salesperson.error_name'),
        ]);
    }
}
