<?php

namespace App\Http\Controllers\Shop;

use App\Helpers\Helper;
use App\Mail\SendActiveFolios;
use App\Mail\SendCartSuccess;
use App\Models\Base\Folio;
use App\Models\Catalogs\Currency;
use App\Models\Shop\ShopProduct;
use App\Resolvers\PaymentPlatformResolver;
use App\Models\System\SysListPriceDistributor;
use App\Models\System\SysSalesOrder;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;

class CartController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        //
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        try {
            $shop_product = ShopProduct::findOrFail($request->id);

            //El carrito solo puede guardar un paquete a la vez
            \Cart::session(\Auth::user()->id)->clear();
            \Cart::session(\Auth::user()->id);

            //Agrega los impuetos en caso de tener
            $conditions = [];
            if ($shop_product->taxes) {
                foreach ($shop_product->taxes as $result2) {
                    $value = '';
                    $rate = $result2->rate;
                    if ($result2->factor == 'Tasa') {
                        $value = ($rate > 0 ? '+' : '') . $rate . '%';
                    } elseif ($result2->factor == 'Cuota') {
                        $value = ($rate > 0 ? '+' : '') . $rate;
                    }

                    $tmp = new \Darryldecode\Cart\CartCondition(array(
                        'name' => $result2->name,
                        'type' => 'tax',
                        'target' => 'subtotal',
                        'value' => $value,
                    ));
                    $conditions[] = $tmp;
                }
            }

            //Agrega al carrito
            //Obtiene precio distribuidor
            $price = $shop_product->price;
            if(!empty(\Auth::user()->sysCustomer->distributor_id)){
                $sys_list_price_distributor = SysListPriceDistributor::active()
                    ->where('distributor_id','=',\Auth::user()->sysCustomer->distributor_id)
                    ->where('shop_product_id','=',$shop_product->id)
                    ->get()
                    ->first();
                if(!empty($sys_list_price_distributor)){
                    if($sys_list_price_distributor->price > 0){
                        $price = $sys_list_price_distributor->price;
                    }
                }
            }
            \Cart::session(\Auth::user()->id)->add($shop_product->id, $shop_product->description, $price, 1,
                [
                    'code' => $shop_product->code,
                    'slug' => $shop_product->slug,
                    'description' => $shop_product->description,
                    'quantity_folio' => $shop_product->quantity_folio,
                ],
                $conditions
            );

            //Actualiza icono con el producto agregado
            $icon_shopping_cart_html = view('layouts.partials.cart.icon_shopping_cart')->render();
            return response()->json([
                'success' => __('shop/cart.text_form_success_add_cart'),
                'icon_shopping_cart_html' => $icon_shopping_cart_html
            ]);
        } catch (\Exception $e) {
            return response()->json($e->getMessage(), $e->getCode());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @param  int $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
    }

    /**
     * Carrito antes de procesar
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function checkout(Request $request)
    {
        //Valida que se haya agregado un paquete
        if (\Cart::session(\Auth::user()->id)->isEmpty()) {
            flash(__('shop/cart.error_cart_empty'))->error();
            return redirect()->route('shop');
        }

        //En caso de cancelar el pago se cancela el regitro que se habia realizado
        $paypal_cancel = !empty($request->pp_cancel) ? true : false;
        $folio_id = !empty($request->folio_id) ? $request->folio_id : null;
        if($paypal_cancel && !empty($folio_id)) {
            $folio = Folio::findOrFail($folio_id);
            if(!empty($folio)) {
                if ($folio->status == Folio::PER_ACTIVE) {
                    $folio->status = Folio::CANCELED;
                    $folio->save();

                    //Cancela pedido
                    $sys_customer = \Auth::user()->sysCustomer;
                    if (!empty($sys_customer)) {
                        $sys_sales_order = SysSalesOrder::where('sys_customer_id', '=', $sys_customer->id)
                            ->where('folio_id', '=', $folio->id)
                            ->update([
                                'status' => SysSalesOrder::CANCELED,
                            ]);
                    }

                    flash(__('shop/cart.text_success_help4'))->error();
                }
            }
        }

        //Variables
        $payment_platforms = collect(config('app.payment_platforms'));

        return view('shop.cart.checkout', compact('payment_platforms'));
    }

    /**
     * Procesamos el carrito de compra
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function process(Request $request)
    {
        \DB::beginTransaction();
        \DB::connection('tenant')->beginTransaction();
        try {

            //validamos los datos obtenidos antes de guardar
            //Valida que seleccione un metodo de pago
            if (empty($request->payment_platform)) {
                throw new \Exception(__('shop/cart.error_payment_platform'));
            }
            if($request->payment_platform == 'stripe' && empty($request->stripe_payment_method)){
                throw new \Exception(__('shop/cart.error_stripe_payment_method'));
            }

            //Validamos que tenga por lo menos un producto en el carrito
            if (\Cart::session(\Auth::user()->id)->isEmpty()) {
                throw new \Exception(__('shop/cart.error_cart_empty'));
            }

            //Carrito
            $cart = \Cart::session(\Auth::user()->id)->getContent()->first();
            $subtotal = \Cart::session(\Auth::user()->id)->getSubTotalWithoutConditions(false);
            $total = \Cart::session(\Auth::user()->id)->getTotal(false);
            $taxes = $total-$subtotal;

            //Otros datos
            $currency = Currency::where('code','=','MXN')->get()->first();

            //Guardamos registro en folios
            $request->merge(['created_uid' => \Auth::user()->id]);
            $request->merge(['updated_uid' => \Auth::user()->id]);
            $request->merge(['name' => 'XXX' . random_int(1000, 9999)]);
            $request->merge(['product' => $cart->attributes->description]);
            $request->merge(['product_code' => $cart->attributes->code]);
            $request->merge(['quantity' => $cart->quantity]);
            $request->merge(['price_unit' => $cart->price]);
            $request->merge(['discount' => 0]);
            $request->merge(['price_reduce' => $cart->price]);
            $request->merge(['payment_way_id' => null]);
            $request->merge(['currency_id' => !empty($currency) ? $currency->id : null]);
            $request->merge(['currency_value' => !empty($currency) ? $currency->rate : null]);
            $request->merge(['amount_discount' => 0]);
            $request->merge(['amount_untaxed' => $subtotal]);
            $request->merge(['amount_tax' => $taxes]);
            $request->merge(['amount_tax_ret' => 0]);
            $request->merge(['amount_total' => $total]);
            $request->merge(['expiry_date' => \Date::now()->addDays(365)]);
            $request->merge(['quantity_folio' => $cart->attributes->quantity_folio]);
            $request->merge(['available_folio' => $cart->attributes->quantity_folio]);
            $request->merge(['status' => Folio::PER_ACTIVE]);
            $request->merge(['payment_platform' => $request->payment_platform]);
            $request->merge(['payment_platform_request' => null]);
            $request->merge(['payment_platform_response' => null]);

            //Guardar
            //Registro principal
            $folio = Folio::create($request->except(['_token','payment_way_code']));
            $folio->name = 'CFL'.$folio->id; //Actualizamos el folio
            $folio->save();

            //Guardamos en pedidos del sistema
            $company = Helper::firstCompany();
            $sys_customer = \Auth::user()->sysCustomer;
            $sys_sales_order = SysSalesOrder::create([
                'created_uid' => \Auth::user()->id,
                'updated_uid' => \Auth::user()->id,
                'name' => 'XXX' . random_int(1000, 9999),
                'sys_customer_id' => !empty($sys_customer) ? $sys_customer->id : null,
                'email_created' => \Auth::user()->email,
                'currency_id' => $folio->currency_id,
                'currency_value' => $folio->currency_value,
                'product' => $folio->product,
                'product_code' => $folio->product_code,
                'quantity' => $folio->quantity,
                'price_unit' => $folio->price_unit,
                'discount' => $folio->discount,
                'price_reduce' => $folio->price_reduce,
                'amount_discount' => $folio->amount_discount,
                'amount_untaxed' => $folio->amount_untaxed,
                'amount_tax' => $folio->amount_tax,
                'amount_tax_ret' => $folio->amount_tax_ret,
                'amount_total' => $folio->amount_total,
                'payment_way_id' => $folio->payment_way_id,
                'folio_id' => $folio->id,
                'sys_customer_folio' => $folio->name,
                'quantity_folio' => $folio->quantity_folio,
                'status' => SysSalesOrder::PER_ACTIVE,
                'payment_platform' => $request->payment_platform,
                'payment_platform_request' => null,
                'payment_platform_response' => null
            ]);
            $sys_sales_order->name = 'PV'.$sys_sales_order->id; //Actualizamos el folio
            $sys_sales_order->save();

            //Genera intencion de pago
            $payment_platform_request = [];
            if($request->payment_platform == 'paypal' || $request->payment_platform == 'stripe'){
                if(!empty($sys_customer->distributor)){
                    $distributor = $sys_customer->distributor;
                    if($distributor->available_folio < $folio->quantity_folio){
                        throw new \Exception(__('auth/user.error_available_folio_2'));
                    }
                }
            }
            if($request->payment_platform == 'paypal'){
                $payment_platform_request['folio_id'] = $folio->id;
                $payment_platform = PaymentPlatformResolver::resolveService($request->payment_platform);
                $payment_platform_request = $payment_platform->handlePayment($total,(!empty($currency) ? $currency->code : null), $payment_platform_request);
                $folio->payment_platform_request = serialize($payment_platform_request);
                $folio->save();
            }
            if($request->payment_platform == 'stripe'){
                $payment_platform_request['folio_id'] = $folio->id;
                $payment_platform_request['payment_method'] = $request->stripe_payment_method;
                $payment_platform = PaymentPlatformResolver::resolveService($request->payment_platform);
                $payment_platform_request = $payment_platform->createIntent($sys_sales_order->name, $total,(!empty($currency) ? $currency->code : null), $payment_platform_request);

                //Validar si la peticion es correcta
                if($payment_platform_request['status'] != 'requires_confirmation'){
                    throw new \Exception(__('shop/cart.error_stripe_create_intent'));
                }

                $folio->payment_platform_request = serialize($payment_platform_request);
                $folio->save();

                //Confirma el pago
                $payment_platform_response = $payment_platform->confirmPayment($payment_platform_request['paymentIntentId']);
                $folio->payment_platform_response = serialize($payment_platform_response);
                if($payment_platform_response->status == 'succeeded') {
                    $folio->activation_date = \Date::now();
                    $folio->status = Folio::ACTIVE; //Se activan los folios
                    $folio->save();

                    //Activa pedido
                    $sys_sales_order->updated_uid = \Auth::user()->id;
                    $sys_sales_order->activation_date = \Date::now();
                    $sys_sales_order->status = SysSalesOrder::ACTIVE;
                    $sys_sales_order->save();

                    //Descuenta de los folios del distribuidor
                    if(!empty($sys_customer->distributor)) {
                        $distributor = $sys_customer->distributor;
                        $distributor->available_folio -= $folio->quantity_folio;
                        $distributor->save();
                    }

                    //Limpiamos el carrito
                    \Cart::session(\Auth::user()->id)->clear();
                }else{
                    throw new \Exception(__('shop/cart.error_stripe_confirm_payment'));
                }
            }

            //Estas acciones se realizan hasta comprobar el pago
            if($request->payment_platform != 'paypal' && $request->payment_platform != 'stripe') {
                //Creamos PDF
                $pdf = $this->print($folio);

                //Enviamos correo de notificacion uno al cliente y otro al administrador
                \Mail::to($folio->createdUser)->send(new SendCartSuccess($folio, $pdf));
                if (!empty(config('app.mail_cart_alert'))) {
                    $tmp[] = config('app.mail_cart_alert');
                    if (!empty($sys_customer) && !empty($sys_customer->distributor->email)) {
                        $tmp[] = $sys_customer->distributor->email;
                    }
                    \Mail::to($tmp)->send(new SendCartSuccess($folio, $pdf));
                }

                //Limpiamos el carrito
                \Cart::session(\Auth::user()->id)->clear();
            }

            \DB::commit();
            \DB::connection('tenant')->commit();

            if($request->payment_platform == 'stripe'){
                if($payment_platform_response->status == 'succeeded') {
                    //Creamos PDF
                    $pdf = $this->print($folio);

                    //Enviamos correo de notificacion uno al cliente y otro al administrador
                    \Mail::to($folio->createdUser)->send(new SendCartSuccess($folio, $pdf));
                    if (!empty(config('app.mail_cart_alert'))) {
                        $tmp[] = config('app.mail_cart_alert');
                        if (!empty($sys_customer) && !empty($sys_customer->distributor->email)) {
                            $tmp[] = $sys_customer->distributor->email;
                        }
                        \Mail::to($tmp)->send(new SendCartSuccess($folio, $pdf));
                    }

                    //Correo de notificacion de activacion de folios
                    \Mail::to($folio->createdUser)->send(new SendActiveFolios($folio));
                }
            }

            //Redirecciona a paypal para pago
            if($request->payment_platform == 'paypal'){
                return redirect($payment_platform_request['approve_href']);
            }

            if($request->payment_platform == 'stripe'){
                return view('shop.cart.success', compact('folio'));
            }

            //Redirecciona
            return redirect(route('cart/success', [$folio->id]));

        } catch (\Exception $e) {
            \DB::rollback();
            \DB::connection('tenant')->rollback();
            flash($e->getMessage())->error();
            return back()->withInput();
        }
    }

    /**
     * Cofirmacion de procesado
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function success(Folio $folio,Request $request)
    {
        //Validaciones de PayPal
        if($folio->payment_platform == 'paypal'){
            $payment_platform_response = [];
            $payment_platform_request = unserialize($folio->payment_platform_request);
            $payment_platform = PaymentPlatformResolver::resolveService($folio->payment_platform);
            $payment_platform_response = $payment_platform->handleApproval($payment_platform_request, $payment_platform_response);
            $folio->payment_platform_response = serialize($payment_platform_response);
            if($payment_platform_response['status'] == 'COMPLETED') {

                \DB::beginTransaction();
                \DB::connection('tenant')->beginTransaction();
                try{
                    $folio->activation_date = \Date::now();
                    $folio->status = Folio::ACTIVE; //Se activan los folios
                    $folio->save();

                    //Activa pedido
                    $sys_customer = \Auth::user()->sysCustomer;
                    if (!empty($sys_customer)) {
                        $sys_sales_order = SysSalesOrder::where('sys_customer_id', '=', $sys_customer->id)
                            ->where('folio_id', '=', $folio->id)
                            ->update([
                                'updated_uid' => \Auth::user()->id,
                                'activation_date' => \Date::now(),
                                'status' => SysSalesOrder::ACTIVE,
                            ]);
                    }

                    //Descuenta de los folios del distribuidor
                    if(!empty($sys_customer->distributor)) {
                        $distributor = $sys_customer->distributor;
                        $distributor->available_folio -= $folio->quantity_folio;
                        $distributor->save();
                    }

                    \DB::commit();
                    \DB::connection('tenant')->commit();

                } catch (\Exception $e) {
                    \DB::rollback();
                    \DB::connection('tenant')->rollback();
                    flash($e->getMessage())->error();
                    return redirect()->route('cart/checkout');
                }

                //Limpiamos el carrito
                \Cart::session(\Auth::user()->id)->clear();

                //Creamos PDF
                $pdf = $this->print($folio);

                //Enviamos correo de notificacion uno al cliente y otro al administrador
                \Mail::to($folio->createdUser)->send(new SendCartSuccess($folio, $pdf));
                if (!empty(config('app.mail_cart_alert'))) {
                    $tmp[] = config('app.mail_cart_alert');
                    if (!empty($sys_customer) && !empty($sys_customer->distributor->email)) {
                        $tmp[] = $sys_customer->distributor->email;
                    }
                    \Mail::to($tmp)->send(new SendCartSuccess($folio, $pdf));
                }

                //Correo de notificacion de activacion de folios
                \Mail::to($folio->createdUser)->send(new SendActiveFolios($folio));

            }else{
                flash(__('shop/cart.text_success_help6'))->error();
                return redirect()->route('cart/checkout');
            }
        }

        return view('shop.cart.success', compact('folio'));
    }

    /**
     * Impresion de comprobante
     *
     * @param Folio $folio
     * @return mixed
     */
    public function print(Folio $folio)
    {
        //PDF
        $pdf = \PDF::loadView('shop.cart.pdf', compact('folio'));

        //Redireccion
        return $pdf->stream($folio->name . '.pdf');
    }

}
