<?php

namespace App\Http\Controllers\System;

use App\Helpers\Helper;
use App\Models\Auth\User;
use App\Helpers\BaseHelper;
use Hyn\Tenancy\Environment;
use Illuminate\Http\Request;
use App\Jobs\CreateTenantJob;
use GuzzleHttp\TransferStats;
use Hyn\Tenancy\Models\Website;
use Illuminate\Validation\Rule;
use App\Models\Shop\ShopProduct;
use Hyn\Tenancy\Models\Hostname;
use App\Models\System\SysCustomer;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Crypt;
use App\Http\Middleware\CheckSuperadmin;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\ServerException;
use GuzzleHttp\Exception\ConnectException;
use App\Mail\SendSysCustomerAccessFactura16;
use Illuminate\Validation\ValidationException;
use App\Http\Middleware\CheckSuperadminOrDistributor;
use Hyn\Tenancy\Contracts\Repositories\WebsiteRepository;
use Hyn\Tenancy\Contracts\Repositories\HostnameRepository;

class SysCustomerController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware(CheckSuperadminOrDistributor::class);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        //Filtro solo los clientes del distribuidor
        if(\Auth::user()->distributor){
            $results = SysCustomer::filter($request->all())->userDistributor()->sortable(['created_at' => 'desc'])->paginate($limit);
        }else{
            $results = SysCustomer::filter($request->all())->sortable(['created_at' => 'desc'])->paginate($limit);
        }

        //
        $results->map(function ($result, $key) {
            if(!empty($result->website->uuid ?? '')){
                $folios_available = 'N/A';
                app(Environment::class)->tenant($result->website);
                if(empty(setting('disabled_folio_cart'))){
                    $folios_available = Helper::numberFormat(BaseHelper::getAvailableFolios());
                }
                $result->folios_available = $folios_available;
            }

            return $result;
        });

        //Retornamos a entorno default
        app(Environment::class)->tenant(\Auth::user()->sysCustomer->website ?? null);

        //Vista
        return view('system.sys_customers.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $distributors = User::distributor()->active()->pluck('name', 'id');
        $shop_products = ShopProduct::populateSelect()->pluck('description', 'id');
        return view('system.sys_customers.create',compact('distributors','shop_products'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //
        $request->merge(['taxid' => strtoupper(trim($request->taxid))]);
        $request->merge(['email' => trim($request->email)]);

        //Genera subdominio para validarlo
        //$request->merge(['fqdn' => $request->record . '.' . $request->domain]);
        /*$this->validate($request, [
            //'fqdn' => 'unique:system.sys_customers',
        ], [
            //'fqdn.unique' => __('system/sys_customer.error_fqdn_unique'),
        ]);*/

        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'email' => 'unique:system.sys_customers|unique:system.users',
            'amount_total' => 'nullable|numeric|required_with:product',
            'quantity_folio' => 'nullable|integer|required_with:product',
            'expiry_date' => 'nullable|date|date_format:"' . setting('date_format', 'd-m-Y') . '"|required_with:product',
        ], [
            //'taxid.unique' => __('system/sys_customer.error_taxid_unique'),
            'email.unique' => __('system/sys_customer.error_email_unique'),
            'amount_total.*' => __('base/folio.error_amount_total'),
            'quantity_folio.required_with' => __('base/folio.error_quantity_folio'),
            'quantity_folio.integer' => __('base/folio.error_quantity_folio_integer'),
            'expiry_date.date_format' => __('base/folio.error_expiry_date_format'),
        ]);

        try{

            //Validaciones manuales
            $validator = \Validator::make([], []);
            $distributor_id = !empty(\Auth::user()->distributor) ? \Auth::user()->id : $request->distributor_id;
            if(!empty($distributor_id)){
                $distributor = User::findOrFail($distributor_id);
                //Validacion para folios manuales
                $available_folio = $distributor->available_folio;
                if(!empty($request->quantity_folio)){
                    if($available_folio < $request->quantity_folio){
                        $validator->after(function ($validator) use ($distributor) {
                            $validator->errors()->add('quantity_folio', sprintf(__('auth/user.error_available_folio'),$distributor->name,$distributor->available_folio));
                        });
                    }else{
                        $available_folio -= $request->quantity_folio;
                    }
                }
                //Validacion para paquetes de folios
                if(!empty($request->shop_product_id)){
                    $shop_product = ShopProduct::findOrFail($request->shop_product_id);
                    if($available_folio < $shop_product->quantity_folio){
                        $validator->after(function ($validator) use ($distributor) {
                            $validator->errors()->add('shop_product_id', sprintf(__('auth/user.error_available_folio'),$distributor->name,$distributor->available_folio));
                        });
                    }
                }

            }
            //Valida que no exista un directorio con el mismo RFC
            /*if(\Storage::exists($request->taxid)) {
                $validator->after(function ($validator) {
                    $validator->errors()->add('taxid', __('system/sys_customer.error_taxid_directory_exists'));
                });
            }
            if ($validator->fails()) {
                throw new ValidationException($validator);
            }*/

            //Logica
            $password = random_int(1000, 9999) . mb_strtolower($request->name[0]); //Contraseña aleatoria
            $request->merge(['created_uid' => \Auth::user()->id]);
            $request->merge(['updated_uid' => \Auth::user()->id]);
            $request->merge(['password' => Crypt::encryptString($password)]);
            $request->merge(['sort_order' => (int)$request->sort_order]);
            $request->merge(['status' => !empty($request->status) ? 1 : 0]);
            if(\Auth::user()->distributor){
                $request->merge(['distributor_id' => \Auth::user()->id]);
            }

            //Guardar
            //Registro principal
            $sys_customer = SysCustomer::create($request->input());

            //Crea nuevo inquilino
            CreateTenantJob::dispatch($request->input(), $sys_customer);

            //Mensaje
            flash(__('general.text_form_success_add'))->success();

            //Respuesta
            return response()->json([
                'success' => true
            ]);
        }catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 422);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\System\SysCustomer $sys_customer
     * @return \Illuminate\Http\Response
     */
    public function show(SysCustomer $sys_customer)
    {
        return redirect('/system/sys-customers');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\System\SysCustomer $sys_customer
     * @return \Illuminate\Http\Response
     */
    public function edit(SysCustomer $sys_customer)
    {
        $distributors = User::distributor()->active()->pluck('name', 'id');
        return view('system.sys_customers.edit', compact('sys_customer','distributors'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @param  \App\Models\System\SysCustomer $sys_customer
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, SysCustomer $sys_customer)
    {
        //Validacion
        $this->validation($request);
        //$request->merge(['fqdn' => $request->record . '.' . $request->domain]);
        $this->validate($request, [
            //'taxid' => [
                //Rule::unique('system.sys_customers')->ignore($sys_customer->id)
            //],
            //'email' => [
                //Rule::unique('system.sys_customers')->ignore($sys_customer->id)
            //],
            //'fqdn' => [
            //Rule::unique('system.sys_customers')->ignore($sys_customer->id)
            //],
        ], [
            //'taxid.unique' => __('system/sys_customer.error_taxid_unique'),
            //'email.unique' => __('system/sys_customer.error_email_unique'),
            //'fqdn.unique' => __('system/sys_customer.error_fqdn_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => 0]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        //Activa/Desactiva todos los usuarios relacionados con el inquilino
        User::where('sys_customer_id','=',$sys_customer->id)->where('superadmin','=','0')->where('distributor','=','0')->update(['status'=> $request->status]);
        if(empty($request->status)){
            User::where('sys_customer_id','=',$sys_customer->id)->where(function($q) {
                $q->where('superadmin','=','1')->orWhere('distributor','=','1');
            })->update(['sys_customer_id'=> null]);
        }

        if(\Auth::user()->distributor){
            $request->merge(['distributor_id' => \Auth::user()->id]);
        }

        if(empty($request->distributor_id)){
            User::where('sys_customer_id','=',$sys_customer->id)->where('distributor','=','1')->update(['sys_customer_id'=> null]);
        }else if($sys_customer->distributor_id != $request->distributor_id){
            User::where('sys_customer_id','=',$sys_customer->id)->where('id', $sys_customer->distributor_id)->update(['sys_customer_id'=> null]);
        }


        $sys_customer->fill($request->only([
            'updated_uid',
            'name',
            'taxid',
            'email',
            'phone',
            'sort_order',
            'status',
            'name_user',
            'distributor_id',
        ]));

        //Guardar
        //Registro principal
        $sys_customer->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/system/sys-customers');

    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\System\SysCustomer $sys_customer
     * @return \Illuminate\Http\Response
     */
    public function destroy(SysCustomer $sys_customer)
    {
        //En caso de intentar borrar un cliente que no es del distribuidor
        if(\Auth::user()->distributor && \Auth::user()->id != $sys_customer->distributor_id){
            //Redireccion
            return redirect('/system/sys-customers');
        }

        //Logica
        $sys_customer->updated_uid = \Auth::user()->id;
        $sys_customer->status = 0;
        //\Storage::move($sys_customer->taxid, $sys_customer->taxid . '_' . date('dmYHis') . random_int(1, 9)); //--Cambia RFC y renombra Directorio
        $sys_customer->save();

        //Cancela todos los usuarios relacionados con cliente
        $tmp = User::where('sys_customer_id','=',$sys_customer->id)->where('superadmin','=','0')->where('distributor','=','0')->update(['status'=> 0]);
        User::where('sys_customer_id','=',$sys_customer->id)->where(function($q) {
            $q->where('superadmin','=','1')->orWhere('distributor','=','1');
        })->update(['sys_customer_id'=> null]);

        //Elimina el hostname y website
        /*if(!empty($sys_customer->fqdn)) {
            if ($hostname = Hostname::where('fqdn', $sys_customer->fqdn)->firstOrFail()) {
                app(HostnameRepository::class)->delete($hostname, false);
            }
        }
        if(!empty($sys_customer->website)) {
            app(WebsiteRepository::class)->delete($sys_customer->website, false);
        }*/

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/system/sys-customers');

    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
            'record' => 'nullable|string',
            'domain' => 'nullable|string',
            'taxid' => [
                'required',
                'regex:/^[A-Z&Ñ]{3,4}[0-9]{2}(0[1-9]|1[012])(0[1-9]|[12][0-9]|3[01])[A-Z0-9]{2}[0-9A]$/i'
            ],
            'name_user' => 'required',
            'email' => 'required|email',
        ], [
            'name.*' => __('system/sys_customer.error_name'),
            'record.*' => __('system/sys_customer.error_record'),
            'domain.*' => __('system/sys_customer.error_domain'),
            'taxid.required' => __('system/sys_customer.error_taxid'),
            'taxid.regex' => __('system/sys_customer.error_taxid_format'),
            'name_user.*' => __('system/sys_customer.error_name_user'),
            'email.required' => __('system/sys_customer.error_email'),
            'email.email' => __('system/sys_customer.error_email_format'),
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Auth\User $user
     * @return \Illuminate\Http\Response
     */
    public function sendSysCustomerAccessLinkEmail(SysCustomer $sys_customer)
    {
        //Logica
        $password = Crypt::decryptString($sys_customer->password);
        \Mail::to($sys_customer->email)->send(new SendSysCustomerAccessFactura16($sys_customer, $password));

        //Mensaje
        flash(__('system/sys_customer.text_success_send_access_link_email'))->success();

        //Redireccion
        return redirect('/system/sys-customers');
    }

    /**
     * Autoacompletar select2 de clientes
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function autocomplete(Request $request)
    {
        //Variables
        $term = $request->term;

        //Logica
        if ($request->ajax() && !empty($term)) {
            //Filtro solo los clientes del distribuidor
            if(\Auth::user()->distributor){
                $tmp = SysCustomer::filter(['filter_search_select2' => $term])->userDistributor()->active()->sortable('name')->limit(16)->get();
            }else{
                $tmp = SysCustomer::filter(['filter_search_select2' => $term])->active()->sortable('name')->limit(16)->get();
            }
            $results = [];
            if ($tmp->isNotEmpty()) {
                foreach ($tmp as $result) {
                    $results[] = [
                        'id' => $result->id,
                        'text' => $result->text_select2,
                        'description' => $result->description_select2
                    ];
                }
            }
            return response()->json($results, 200);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Modal para buscar clientes
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function modalSearch(Request $request)
    {
        //Variables
        $active_btn_select = (!empty($request->active_btn_select) ? $request->active_btn_select : '');
        //En caso de estar vacio le asignamos un datos para mostrar los clientes con "a"
        $request->merge(['filter_search' => !empty($request->filter_search) ? $request->filter_search : '']);

        //Logica
        if ($request->ajax()) {
            //Filtro solo los clientes del distribuidor
            if(\Auth::user()->distributor){
                $results = SysCustomer::filter($request->all())->userDistributor()->active()->sortable('name')->limit(10)->get();
            }else{
                $results = SysCustomer::filter($request->all())->active()->sortable('name')->limit(10)->get();
            }

            //Busca cliente
            $html_sys_customers = view('layouts.partials.sys_customers.search',
                compact('results', 'active_btn_select'))->render();

            //modal de buscar
            $html = view('layouts.partials.sys_customers.modal_search',
                compact('html_sys_customers', 'active_btn_select'))->render();


            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Modal para buscar clientes
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function modalUserSearch(Request $request)
    {
        //Variables
        $active_btn_select = (!empty($request->active_btn_select) ? $request->active_btn_select : '');
        //En caso de estar vacio le asignamos un datos para mostrar los clientes con "a"
        $request->merge(['filter_search' => !empty($request->filter_search) ? $request->filter_search : '']);

        //Logica
        if ($request->ajax()) {
            //Filtro solo los clientes del distribuidor
            if(\Auth::user()->distributor){
                $results = SysCustomer::filter($request->all())->userDistributor()->active()->sortable('name')->limit(10)->get();
            }else{
                $results = SysCustomer::filter($request->all())->active()->sortable('name')->limit(10)->get();
            }

            //Busca cliente
            $html_sys_customers = view('layouts.partials.sys_customers.search',
                compact('results', 'active_btn_select'))->render();

            //modal de buscar
            $html = view('layouts.partials.sys_customers.modal_user_search',
                compact('html_sys_customers', 'active_btn_select'))->render();


            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Modal para buscar clientes
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function search(Request $request)
    {
        //Variables
        $active_btn_select = (!empty($request->active_btn_select) ? $request->active_btn_select : '');
        //En caso de estar vacio le asignamos un datos para mostrar los clientes con "a"
        $request->merge(['filter_search' => !empty($request->filter_search) ? $request->filter_search : '']);

        //Logica
        if ($request->ajax()) {
            //Filtro solo los clientes del distribuidor
            if(\Auth::user()->distributor){
                $results = SysCustomer::filter($request->all())->userDistributor()->active()->sortable('name')->limit(10)->get();
            }else{
                $results = SysCustomer::filter($request->all())->active()->sortable('name')->limit(10)->get();
            }

            //Busca cliente
            $html = view('layouts.partials.sys_customers.search',
                compact('results', 'active_btn_select'))->render();

            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Actualiza datos generales
     *
     * @param Request $request
     * @param SysCustomer $sys_customer
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function storeStep2(Request $request, SysCustomer $sys_customer)
    {
        /*
        Modulo para configurar primer ingreso
            Datos de empresa
            Archivos de certificado
            Datos de matriz
            Nombres de documentos
            Crear folios gratis
            Color de documento
        ---
        */

        //Crea un nuevo sitio
        /*$website = new Website();
        app(WebsiteRepository::class)->create($website);

        $hostname = new Hostname();
        $hostname->fqdn = $request->fqdn . '.' . config('app.tenant_base_url');
        app(HostnameRepository::class)->attach($hostname, $website);*/

        //\Config::set('database.default', 'system');
        //\DB::reconnect('system');

        //make hostname current
        //app(Environment::class)->hostname($hostname);
        //app(Environment::class)->tenant($website);


        //Redireccion
        //return redirect()->route('sys-customers/store-step-2', ['id' => $sys_customer->id]);

        /*$hostname = Hostname::where('fqdn', $sys_customer->fqdn . '.' . config('app.tenant_base_url'))->firstOrFail();
        $website = $hostname->website;

        //Actualiza informacion
        $sys_customer->hostname_id = $hostname->id;
        $sys_customer->website_id = $website->id;
        $sys_customer->save();*/

        //Crea subdominio en Godaddy
        /*try {
            $client = new \GuzzleHttp\Client();
            $record = [
                'type' => 'A',
                'name' => $sys_customer->fqdn,
                'data' => config('app.godaddy_records_a_data'),
                'ttl' => 600,
            ];
            $url = 'https://api.godaddy.com/v1/domains/'.config('app.tenant_base_url').'/records/';
            $response = $client->patch($url, [
                'headers' => [
                    'Accept' => 'application/json',
                    'Authorization' => 'sso-key ' . config('app.godaddy_api_key').':'.config('app.godaddy_api_secret'),
                    'Content-Type' => 'application/json',
                ],
                'body' => '['.json_encode($record).']',
            ]);
            //OK Code: 200
        } catch (ClientException $e) {
            $result = \GuzzleHttp\json_decode($e->getResponse()->getBody()->getContents());
            $error = !empty($result->code) ? $result->code .': ' . $result->message : 'Error';
            flash(__($error))->error();
            return redirect('/system/sys-customers');
        } catch (ServerException $e) {
            flash(__($e->getMessage()))->error();
            return redirect('/system/sys-customers');
        } catch (ConnectException $e) {
            flash(__($e->getMessage()))->error();
            return redirect('/system/sys-customers');
        }*/

        //Crear registros

        //Mensaje
        /*flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/system/sys-customers');*/


        //Crear nuevo subdominio
        /*try {
            $client = new \GuzzleHttp\Client();*/
        //Obtener dominios
        /*$url = 'https://api.godaddy.com/v1/domains/';
        $response = $client->get($url, [
            'headers' => [
                'Accept' => 'application/json',
                'Authorization' => 'sso-key ' . config('app.godaddy_key').':'.config('app.godaddy_secret'),
                'Content-Type' => 'application/json',
            ],
        ]);
        $result = \GuzzleHttp\json_decode($response->getBody()->getContents());
        $success = !empty($result) ? $result : [];
        foreach($success as $domain){
            echo $domain->domain.'<br>';
        }*/
        //Obtener registros
        /*$url = 'https://api.godaddy.com/v1/domains/factura16.com/records';
        $response = $client->get($url, [
            'headers' => [
                'Accept' => 'application/json',
                'Authorization' => 'sso-key ' . config('app.godaddy_key').':'.config('app.godaddy_secret'),
                'Content-Type' => 'application/json',
            ],
        ]);
        $result = \GuzzleHttp\json_decode($response->getBody()->getContents());
        $success = !empty($result) ? $result : [];
        foreach($success as $record){
            echo $record->type.'--'.$record->name.'--'.$record->data.'--'.$record->ttl.'<br>';
        }*/
        //Crear subdominio
        /*$data = [
            'type' => 'A',
            'name' => 'testiv104',
            'data' => config('app.godaddy_subdomain_data'),
            'ttl' => 600,
        ];
        $url = 'https://api.godaddy.com/v1/domains/factura16.com/records/';

        $response = $client->patch($url, [
            'headers' => [
                'Accept' => 'application/json',
                'Authorization' => 'sso-key ' . config('app.godaddy_key').':'.config('app.godaddy_secret'),
                'Content-Type' => 'application/json',
            ],
            'body' => '['.json_encode($data).']',
        ]);
        dd($response);
        $result = \GuzzleHttp\json_decode($response->getBody()->getContents());
        dd($result);
        $success = !empty($result) ? $result : [];
        dd($success);*/

        // see GoDaddy API documentation - https://developer.godaddy.com/doc
        // url to check domain availability
        /*$url = "https://api.godaddy.com/v1/domains/factura16.com/records/";

        // set your key and secret
        $header = array(
            'Authorization: sso-key '.config('app.godaddy_key').':'.config('app.godaddy_secret'),
            'Content-Type: application/json',
            'Accept: application/json',
        );
        $variable = [
            "type" => "A",
            "name" => "testiv3",
            "data" => config("app.godaddy_subdomain_data"),
            "ttl" => 600,
        ];
        //dd((object)$variable);

        //open connection
        $ch = curl_init();
        $timeout=60;

        //set the url and other options for curl
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER,false);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PATCH'); // Values: GET, POST, PUT, DELETE, PATCH, UPDATE
        //curl_setopt($ch, CURLOPT_POSTFIELDS, "[{\"type\": \"A\",\"name\": \"appv1\",\"data\": \"192.1.2.3\",\"ttl\": 3600}]" );
        //curl_setopt($ch, CURLOPT_POSTFIELDS, '[{"type": "A","name": "appv1","data": "192.1.2.3","ttl": 3600}]' );
        curl_setopt($ch, CURLOPT_POSTFIELDS, '[' .json_encode($variable) .']' );
        curl_setopt($ch, CURLOPT_HTTPHEADER, $header);
        curl_setopt($ch, CURLINFO_HEADER_OUT, true);

        //execute call and return response data.
        $result = curl_exec($ch);
        //close curl connection
        curl_close($ch);
        // decode the json response
        $dn = json_decode($result, true);
        dd($dn);*/

        /*} catch (ClientException $e) {
            $result = \GuzzleHttp\json_decode($e->getResponse()->getBody()->getContents());
            $error = !empty($result->code) ? $result->code .': ' . $result->message : 'Error';
            dd($error);
        } catch (ServerException $e) {
            dd('Falla inesperada. Intente mas tarde');
        } catch (ConnectException $e) {
            dd('No se puede contectar al servicio');
        }
        return redirect('/system/sys-customers');*/
    }
}
