<?php

namespace App\Imports;

use App\Helpers\Helper;
use App\Models\Base\BankMoveTemp;
use App\Models\Catalogs\Currency;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;

class BankMoveTemplateImport implements ToCollection, WithHeadingRow
{
    public function __construct(Request $request)
    {
        $this->request = $request;
    }

    /**
     * @param Collection $rows
     * @return mixed
     * @throws \Exception
     */
    public function collection(Collection $rows)
    {
        //Limpieza de datos
        $rows = Helper::clearArrayImport($rows);

        //Validacion
        foreach ($rows as $key => $row) {
            $num_row = $key + 2;

            if (strpos($row['fecha'], "'") || strpos($row['fecha'], "-") || strpos($row['fecha'], "/")) {
                $rows[$key]['fecha'] = $row['fecha'] = str_replace('\'', '', str_replace('/', '-', $row['fecha']));
            } else {
                $rows[$key]['fecha'] = \Date::instance(\PhpOffice\PhpSpreadsheet\Shared\Date::excelToDateTimeObject($row['fecha']))->format(setting('date_format', 'd-m-Y'));
            }

            Validator::make($row->toArray(), [
                'fecha' => 'required|date|date_format:"'.setting('date_format', 'd-m-Y').'"',
                'depositos' => 'nullable|numeric|min:0',
                'retiros' => 'nullable|numeric|min:0',
                'moneda' => 'required',
            ], [
                'fecha.required' => __('base/bank_reconciliation.error_date') . sprintf(__('base/bank_reconciliation.error_row'), $num_row,$row['fecha']),
                'fecha.date' => __('base/bank_reconciliation.error_date_format') . sprintf(__('base/bank_reconciliation.error_row'), $num_row,$row['fecha']),
                'fecha.date_format' => __('base/bank_reconciliation.error_date_format') . sprintf(__('base/bank_reconciliation.error_row'), $num_row,$row['fecha']),
                'depositos.*' => __('base/bank_reconciliation.error_amount') . sprintf(__('base/bank_reconciliation.error_row'), $num_row,$row['depositos']),
                'retiros.*' => __('base/bank_reconciliation.error_amount') . sprintf(__('base/bank_reconciliation.error_row'), $num_row,$row['retiros']),
                'moneda.*' => __('base/bank_reconciliation.error_currency_code') . sprintf(__('base/bank_reconciliation.error_row'), $num_row,$row['moneda']),
            ])->validate();
        }

        $company = Helper::defaultCompany(); //Empresa

        //Importar datos
        foreach ($rows as $key => $row) {
            $num_row = $key + 2;

            //Moneda
            $currency = null;
            if (!empty($row['moneda'])) {
                $currency = Currency::where('code', '=', $row['moneda'])->first();
                if (empty($currency)) {
                    throw new \Exception(__('base/bank_reconciliation.error_currency_code') . sprintf(__('base/bank_reconciliation.error_row'), $num_row,$row['moneda']));
                }
            }

            //Guardar
            //---Registro principal
            $depositos = (double)$row['depositos'];
            $retiros = (double)$row['retiros'];
            if($depositos > 0 || $retiros > 0) {
                $bank_move_temp = BankMoveTemp::create([
                    'created_uid' => \Auth::user()->id,
                    'updated_uid' => \Auth::user()->id,
                    'key' => $this->request->key,
                    'date' => !empty($row['fecha']) ? Helper::convertDateToSql($row['fecha']) : null,
                    'reference' => $row['referencia'],
                    'amount' => ($depositos > 0 ? $depositos : ($retiros>0 ? ($retiros * (-1)) : 0) ),
                    'currency_id' => $currency->id,
                    'company_id' => $company->id,
                    'file_template' => $this->request->file_template,
                    'status' => 1,
                ]);
            }

        }

        return true;
    }
}
