<?php

namespace App\Jobs;

use App\Helpers\BaseHelper;
use App\Helpers\Helper;
use App\Helpers\PacHelper;
use App\Http\Controllers\Sales\CustomerInvoiceController;
use App\Mail\SendCustomerInvoice;
use App\Models\Catalogs\PaymentTerm;
use App\Models\Catalogs\Postcode;
use App\Models\Sales\CustomerInvoice;
use App\Models\Sales\CustomerInvoiceCfdi;
use App\Models\Sales\CustomerInvoiceLine;
use App\Models\Sales\CustomerInvoiceLineComplement;
use App\Models\Sales\CustomerInvoiceTax;
use App\Models\Sales\RecurringCustomerInvoice;
use Hyn\Tenancy\Environment;
use Illuminate\Bus\Queueable;
use Illuminate\Queue\SerializesModels;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;

class ProcessRecurringCustomerInvoice implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $recurring_customer_invoice;
    protected $website;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(RecurringCustomerInvoice $recurring_customer_invoice, $website)
    {
        //
        $this->recurring_customer_invoice = $recurring_customer_invoice;
        $this->website = $website;
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        //
        $recurring_customer_invoice = $this->recurring_customer_invoice;
        $website = $this->website;
        $document_type_code = 'customer.invoice';

        //Cambiamos entorno al del cliente
        app(Environment::class)->tenant($website);

        //Almacenamiento de archivos
        $customer_invoice = null;
        $customer_invoice_controller = new CustomerInvoiceController();

        \DB::connection('tenant')->beginTransaction();
        try {
            //Logica
            $company = $recurring_customer_invoice->company;
            $payment_term = PaymentTerm::findOrFail($recurring_customer_invoice->payment_term_id);
            $date_due = $payment_term->days > 0 ? \Date::now()->addDays($payment_term->days) : \Date::now();
            $document_type = Helper::getNextDocumentTypeByCode($document_type_code,$company->id,false, $recurring_customer_invoice->branch_office_id);

            //Valida que tenga folios disponibles
            if(BaseHelper::getAvailableFolios() <= 0){
                throw new \Exception(__('general.error_available_folios'));
            }

            //Obtiene fecha con el codigo postal
            $postcode = Postcode::where('name','=',$recurring_customer_invoice->branchOffice->postcode)->first();
            $hours = 0;
            if(!empty($postcode)){
                $hours = 6 - abs($postcode->diff2);
            }
            $cfdi_date = \Date::now();
            if($hours > 0){
                $cfdi_date->addHours($hours);
            }elseif($hours < 0){
                $cfdi_date->subHours(abs($hours));
            }
            $cfdi_date->subMinutes(5);

            //Guardar
            //Registro principal
            $customer_invoice = CustomerInvoice::create([
                'created_uid' => null,
                'updated_uid' => null,
                'name' => $document_type['name'],
                'serie' => $document_type['serie'],
                'folio' => $document_type['folio'],
                'date' => Helper::dateTimeToSql($cfdi_date),
                'date_due' => Helper::dateToSql($date_due),
                'reference' => $recurring_customer_invoice->reference,
                'customer_id' => $recurring_customer_invoice->customer_id,
                'branch_office_id' => $recurring_customer_invoice->branch_office_id,
                'payment_term_id' => $recurring_customer_invoice->payment_term_id,
                'payment_way_id' => $recurring_customer_invoice->payment_way_id,
                'payment_method_id' => $recurring_customer_invoice->payment_method_id,
                'cfdi_use_id' => $recurring_customer_invoice->cfdi_use_id,
                'salesperson_id' => $recurring_customer_invoice->salesperson_id,
                'currency_id' => $recurring_customer_invoice->currency_id,
                'currency_value' => $recurring_customer_invoice->currency_value,
                'amount_discount' => $recurring_customer_invoice->amount_discount,
                'amount_untaxed' => $recurring_customer_invoice->amount_untaxed,
                'amount_tax' => $recurring_customer_invoice->amount_tax,
                'amount_tax_ret' => $recurring_customer_invoice->amount_tax_ret,
                'amount_total' => $recurring_customer_invoice->amount_total,
                'balance' => $recurring_customer_invoice->amount_total,
                'document_type_id' => $document_type['id'],
                'cfdi_relation_id' => null,
                'comment' => $recurring_customer_invoice->comment,
                'mail_sent' => false,
                'status' => CustomerInvoice::OPEN,
                'company_id' => $company->id,
                'title' => $recurring_customer_invoice->title,
                'project_id' => $recurring_customer_invoice->project_id,
                'tax_regimen_id' => $recurring_customer_invoice->tax_regimen_id,
                'tax_regimen_customer_id' => $recurring_customer_invoice->tax_regimen_customer_id,
            ]);

            //Lineas
            foreach($recurring_customer_invoice->recurringCustomerActiveInvoiceLines as $item){
                $customer_invoice_line = CustomerInvoiceLine::create([
                    'created_uid' => null,
                    'updated_uid' => null,
                    'customer_invoice_id' => $customer_invoice->id,
                    'name' => $item->name,
                    'product_id' => $item->product_id,
                    'sat_product_id' => $item->sat_product_id,
                    'unit_measure_id' => $item->unit_measure_id,
                    'quantity' => $item->quantity,
                    'price_unit' => $item->price_unit,
                    'discount' => $item->discount,
                    'price_reduce' => $item->price_reduce,
                    'amount_discount' => $item->amount_discount,
                    'amount_untaxed' => $item->amount_untaxed,
                    'amount_tax' => $item->amount_tax,
                    'amount_tax_ret' => $item->amount_tax_ret,
                    'amount_total' => $item->amount_total,
                    'sort_order' => $item->sort_order,
                    'status' => 1,
                    'quota_ieps' => $item->quota_ieps,
                    'tax_object' => !empty($item->tax_object) ? $item->tax_object : ($item->taxes->isEmpty() ? '01' : '02'),
                ]);

                //Guardar impuestos por linea
                $taxes = [];
                if($item->taxes->isNotEmpty()){
                    foreach($item->taxes as $tmp){
                        $taxes[] = $tmp->id;
                    }
                }
                if(!empty($taxes)) {
                    $customer_invoice_line->taxes()->sync($taxes);
                }

                //Datos de complementos IEDU
                $customer_invoice_line_complement = CustomerInvoiceLineComplement::create([
                    'created_uid' => null,
                    'updated_uid' => null,
                    'customer_invoice_line_id' => $customer_invoice_line->id,
                    'name' => $item->recurringCustomerInvoiceLineComplement->name,
                    'customer_iedu_id' => $item->recurringCustomerInvoiceLineComplement->customer_iedu_id,
                    'iedu_nombre_alumno' => $item->recurringCustomerInvoiceLineComplement->iedu_nombre_alumno,
                    'iedu_curp' => $item->recurringCustomerInvoiceLineComplement->iedu_curp,
                    'iedu_nivel_educativo' => $item->recurringCustomerInvoiceLineComplement->iedu_nivel_educativo,
                    'iedu_aut_rvoe' => $item->recurringCustomerInvoiceLineComplement->iedu_aut_rvoe,
                    'iedu_rfc_pago' => $item->recurringCustomerInvoiceLineComplement->iedu_rfc_pago,
                    'sort_order' => $item->recurringCustomerInvoiceLineComplement->sort_order,
                    'status' => 1,
                ]);
            }

            //Resumen de impuestos
            if($recurring_customer_invoice->recurringCustomerInvoiceTaxes->isNotEmpty()){
                foreach($recurring_customer_invoice->recurringCustomerInvoiceTaxes as $result){
                    $customer_invoice_tax = CustomerInvoiceTax::create([
                        'created_uid' => null,
                        'updated_uid' => null,
                        'customer_invoice_id' => $customer_invoice->id,
                        'name' => $result->name,
                        'tax_id' => $result->tax_id,
                        'amount_base' => $result->amount_base,
                        'amount_tax' => $result->amount_tax,
                        'sort_order' => $result->sort_order,
                        'status' => 1,
                    ]);
                }
            }

            //Valida que tenga exista la clase de facturacion
            $class_cfdi = setting('cfdi_version');
            if (empty($class_cfdi)) {
                throw new \Exception(__('general.error_cfdi_version'));
            }
            if (!method_exists($customer_invoice_controller, $class_cfdi)) {
                throw new \Exception(__('general.error_cfdi_class_exists'));
            }

            //Registros de cfdi
            $customer_invoice_cfdi = CustomerInvoiceCfdi::create([
                'created_uid' => null,
                'updated_uid' => null,
                'customer_invoice_id' => $customer_invoice->id,
                'name' => $customer_invoice->name,
                'cfdi_version' => $class_cfdi,
                'status' => 1,
            ]);

            //Valida Empresa y PAC para timbrado
            PacHelper::validateSatActions($company);

            //Actualiza proxima fecha de ejecucion
            //Crear XML y timbra
            $tmp = $customer_invoice_controller->$class_cfdi($customer_invoice);

            //Disminuye folios
            BaseHelper::decrementFolios();

            //Guardar registros de CFDI
            $customer_invoice_cfdi->fill(array_only($tmp, [
                'pac_id',
                'cfdi_version',
                'uuid',
                'date',
                'file_xml',
                'file_xml_pac',
            ]));
            $customer_invoice_cfdi->save();

            //Genera nueva fecha de ejecucion
            $date_next = \Date::today();
            if (preg_match('/D/i', $recurring_customer_invoice->frequency)) { //dias
                $date_next->addDays((int)str_replace('D','',$recurring_customer_invoice->frequency));
            }
            if (preg_match('/W/i', $recurring_customer_invoice->frequency)) { //semanas
                $date_next->addWeeks((int)str_replace('W','',$recurring_customer_invoice->frequency));
            }
            if (preg_match('/M/i', $recurring_customer_invoice->frequency)) { //meses
                $date_next->addMonths((int)str_replace('M','',$recurring_customer_invoice->frequency));
            }
            if (preg_match('/Y/i', $recurring_customer_invoice->frequency)) {
                $date_next->addYears((int)str_replace('Y','',$recurring_customer_invoice->frequency));
            }

            //Actualiza estatus
            $recurring_customer_invoice->date_last = \Date::now();
            $recurring_customer_invoice->date_next = $date_next;
            $recurring_customer_invoice->status_process = sprintf(__('sales/recurring_customer_invoice.text_success_process_recurring_customer_invoice'), $customer_invoice->name);
            $recurring_customer_invoice->date_process = \Date::now();
            $recurring_customer_invoice->save();

            //
            \DB::connection('tenant')->commit();
        }catch (\Exception $e) {
            \DB::connection('tenant')->rollback();

            //Actualiza estatus de error
            $recurring_customer_invoice->status_process = sprintf(__('sales/recurring_customer_invoice.text_error_process_recurring_customer_invoice'), $e->getMessage());
            $recurring_customer_invoice->date_process = \Date::now();
            $recurring_customer_invoice->save();
        }

        try{
            //Almacenamiento dropbox
            $customer_invoice_controller->dropboxBackup($customer_invoice);

            if(!empty($customer_invoice) && $recurring_customer_invoice->mail_sent && !empty($customer_invoice->customer->email)){
                //Creamos PDF en stream
                $pdf = $customer_invoice_controller->print($customer_invoice);
                //Ruta del XML
                $path_xml = Helper::setDirectory(CustomerInvoice::PATH_XML_FILES_CI, $customer_invoice->company_id) . '/';
                $file_xml_pac = $path_xml . $customer_invoice->customerInvoiceCfdi->file_xml_pac;

                //
                $subject = $customer_invoice->documentType->name .' ('.$customer_invoice->name.') - ' . $customer_invoice->company->name;
                $custom_message = view('layouts.partials.customer_invoices.message_send_mail', compact('customer_invoice'))->render();

                $to = [];
                if (!empty($customer_invoice->customer->email)) {
                    $tmp = explode(';',$customer_invoice->customer->email);
                    if(!empty($tmp)) {
                        foreach($tmp as $email) {
                            $email = trim($email);
                            $to[$email] = $email;
                        }
                    }
                }

                $reply = [];
                if(!empty($customer_invoice->company->email)){
                    $reply[$customer_invoice->company->email] = $customer_invoice->company->email;
                }

                //Envio de correo
                if(!empty($to))
                    \Mail::to($to)->send(new SendCustomerInvoice($customer_invoice, $subject, $custom_message, $pdf, $file_xml_pac, $reply));

                //Actualiza campo de enviado
                $customer_invoice->updated_uid = null;
                $customer_invoice->mail_sent = 1;
                $customer_invoice->save();

                //
                sleep(3);
            }
        }catch (\Exception $e) {
            \Log::error('(' . $customer_invoice->company->taxid . ') ' . $e->getMessage());
        }
    }
}
