<?php

namespace App\Models\Base;

use App\Models\Catalogs\CfdiType;
use Illuminate\Database\Eloquent\Model;
use Kyslik\ColumnSortable\Sortable;
use Hyn\Tenancy\Traits\UsesTenantConnection;

/**
 * App\Models\Base\DocumentType
 *
 * @property int $id
 * @property \Illuminate\Support\Carbon|null $created_at
 * @property \Illuminate\Support\Carbon|null $updated_at
 * @property int|null $created_uid
 * @property int|null $updated_uid
 * @property string $name
 * @property string $code
 * @property string|null $prefix
 * @property int $current_number
 * @property int $increment_number
 * @property bool $nature
 * @property int|null $cfdi_type_id
 * @property int $sort_order
 * @property bool $status
 * @property int|null $company_id
 * @property string|null $draft_prefix
 * @property int $draft_current_number
 * @property-read \App\Models\Catalogs\CfdiType|null $cfdiType
 * @property-read \App\Models\Base\Company|null $company
 * @property-read mixed $name_sat
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\DocumentType active()
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\DocumentType filter($input = array())
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\DocumentType newModelQuery()
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\DocumentType newQuery()
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\DocumentType populateSelect()
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\DocumentType query()
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\DocumentType sortable($defaultParameters = null)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\DocumentType whereCfdiTypeId($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\DocumentType whereCode($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\DocumentType whereCompanyId($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\DocumentType whereCreatedAt($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\DocumentType whereCreatedUid($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\DocumentType whereCurrentNumber($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\DocumentType whereDraftCurrentNumber($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\DocumentType whereDraftPrefix($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\DocumentType whereId($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\DocumentType whereIncrementNumber($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\DocumentType whereName($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\DocumentType whereNature($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\DocumentType wherePrefix($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\DocumentType whereSortOrder($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\DocumentType whereStatus($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\DocumentType whereUpdatedAt($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\DocumentType whereUpdatedUid($value)
 * @mixin \Eloquent
 * @property int|null $branch_office_id
 * @property-read \App\Models\Base\BranchOffice|null $branchOffice
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\DocumentType whereBranchOfficeId($value)
 */
class DocumentType extends Model
{
    use Sortable,UsesTenantConnection;

    protected $table = 'document_types';

    const NO_NATURE = 1;
    const DEBIT = 2;
    const CREDIT = 3;

    /**
     * Attributes that should be mass-assignable.
     *
     * @var array
     */
    protected $fillable = [
        'created_uid',
        'updated_uid',
        'name',
        'code',
        'prefix',
        'current_number',
        'increment_number',
        'nature',
        'cfdi_type_id',
        'sort_order',
        'status',
        'company_id',
        'draft_prefix',
        'draft_current_number',
        'branch_office_id'
    ];

    /**
     * Sortable columns.
     *
     * @var array
     */
    public $sortable = ['name', 'code', 'prefix', 'current_number', 'increment_number','nature','cfdi_type_id', 'sort_order', 'status'];

    public function scopeFilter($query, array $input = [])
    {
        $query->where(function ($query) {
            $query->where('document_types.company_id', '=', \App\Helpers\Helper::defaultCompany()->id);
        })->where(function($query) use ($input) {
            if (!empty($input['filter_search'])) {
                $search = $input['filter_search'];
                $query->orWhere('name', 'like', '%' . str_replace(' ', '%%', $search) . '%');
                $query->orWhere('code', 'like', '%' . str_replace(' ', '%%', $search) . '%');
                $query->orWhere('prefix', 'like', '%' . str_replace(' ', '%%', $search) . '%');
                $query->orWhereHas('cfdiType', function ($q) use ($search) {
                    $q->where('cfdi_types.name', 'like', '%' . str_replace(' ', '%%', $search) . '%');
                });
                $query->orWhereHas('branchOffice', function ($q) use ($search) {
                    $q->where('branch_offices.name', 'like', '%' . str_replace(' ', '%%', $search) . '%');
                });
            }
        });

        return $query;
    }

    /**
     * Solo registros activos
     *
     * @param $query
     * @return mixed
     */
    public function scopeActive($query)
    {
        $query->where('document_types.company_id', '=', \App\Helpers\Helper::defaultCompany()->id);
        $query->where('status','=','1');

        return $query;
    }

    /**
     * Funcion para llenado de datos en un select
     *
     * @param $query
     * @return mixed
     */
    public function scopePopulateSelect($query)
    {
        $query->where('document_types.company_id', '=', \App\Helpers\Helper::defaultCompany()->id);
        $query->where('status','=','1');
        $query->orderBy('sort_order');
        $query->orderBy('name');

        return $query;
    }

    public function natureHuman($nature){
        $str = __('base/document_type.text_no_nature');
        if($nature == self::DEBIT){
            $str = __('base/document_type.text_debit');
        }elseif($nature == self::CREDIT){
            $str = __('base/document_type.text_credit');
        }
        return $str;
    }

    public function getNameSatAttribute()
    {
        return '[' . $this->code . '] ' . $this->name;
    }

    public function cfdiType()
    {
        return $this->belongsTo(CfdiType::class);
    }

    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    public function branchOffice()
    {
        return $this->belongsTo(BranchOffice::class);
    }
}
