<?php

namespace App\Models\Base;

use App\Helpers\Helper;
use App\Models\Catalogs\Currency;
use App\Models\Catalogs\TaxRegimen;
use Kyslik\ColumnSortable\Sortable;
use App\Models\Catalogs\PayrollType;
use Illuminate\Database\Eloquent\Model;
use Hyn\Tenancy\Traits\UsesTenantConnection;

/**
 * App\Models\Base\Payroll
 *
 * @property int $id
 * @property \Illuminate\Support\Carbon|null $created_at
 * @property \Illuminate\Support\Carbon|null $updated_at
 * @property int|null $created_uid
 * @property int|null $updated_uid
 * @property string $name
 * @property string $generated_type
 * @property string $file
 * @property int|null $payroll_type_id
 * @property int|null $branch_office_id
 * @property string|null $date_payment
 * @property string|null $date_start_payment
 * @property string|null $date_end_payment
 * @property int $payment_days
 * @property int|null $currency_id
 * @property float $currency_value
 * @property float $amount_discount
 * @property float $amount_untaxed
 * @property float $amount_isr
 * @property float $amount_tax
 * @property float $amount_tax_ret
 * @property float $amount_total
 * @property float $amount_perceptions
 * @property float $amount_deductions
 * @property float $amount_other_payment_types
 * @property string|null $comment
 * @property bool $mail_sent
 * @property int|null $company_id
 * @property int $sort_order
 * @property bool $status
 * @property-read \App\Models\Base\BranchOffice|null $branchOffice
 * @property-read \App\Models\Base\Company|null $company
 * @property-read \App\Models\Catalogs\Currency|null $currency
 * @property-read \Illuminate\Database\Eloquent\Collection|\App\Models\Base\EmployeePayroll[] $employeePayrolls
 * @property-read \App\Models\Catalogs\PayrollType|null $payrollType
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll filter($input = array())
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll newModelQuery()
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll newQuery()
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll query()
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll sortable($defaultParameters = null)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereAmountDeductions($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereAmountDiscount($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereAmountIsr($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereAmountOtherPaymentTypes($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereAmountPerceptions($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereAmountTax($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereAmountTaxRet($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereAmountTotal($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereAmountUntaxed($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereBranchOfficeId($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereComment($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereCompanyId($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereCreatedAt($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereCreatedUid($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereCurrencyId($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereCurrencyValue($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereDateEndPayment($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereDatePayment($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereDateStartPayment($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereFile($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereGeneratedType($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereId($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereMailSent($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereName($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll wherePaymentDays($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll wherePayrollTypeId($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereSortOrder($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereStatus($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereUpdatedAt($value)
 * @method static \Illuminate\Database\Eloquent\Builder|\App\Models\Base\Payroll whereUpdatedUid($value)
 * @mixin \Eloquent
 */
class Payroll extends Model
{
    use Sortable,UsesTenantConnection;

    protected $table = 'payrolls';

    const PATH_FILES = 'files/payrolls';

    //Por timbrar
    const DRAFT = 1;
    const OPEN = 2; //Timbrado
    const CANCEL = 3; //Cancelado

    /**
     * Attributes that should be mass-assignable.
     *
     * @var array
     */
    protected $fillable = [
        'created_at',
        'updated_at',
        'name',
        'generated_type',
        'file',
        'payroll_type_id',
        'branch_office_id',
        'date_payment',
        'date_start_payment',
        'date_end_payment',
        'payment_days',
        'currency_id',
        'currency_value',
        'amount_discount',
        'amount_untaxed',
        'amount_isr',
        'amount_tax',
        'amount_tax_ret',
        'amount_total',
        'amount_perceptions',
        'amount_deductions',
        'amount_other_payment_types',
        'comment',
        'mail_sent',
        'company_id',
        'sort_order',
        'status',
        'tax_regimen_id',
    ];

    /**
     * Sortable columns.
     *
     * @var array
     */
    public $sortable = [
        'name',
        'generated_type',
        'created_at',
        'date_payment',
        'amount_discount',
        'amount_untaxed',
        'amount_isr',
        'amount_tax',
        'amount_tax_ret',
        'amount_total',
        'amount_perceptions',
        'amount_deductions',
        'amount_other_payment_types',
        'mail_sent',
        'sort_order',
        'status'
    ];

    public function getNameDatePaymentAttribute()
    {
        return '[FP: ' . Helper::convertSqlToDate($this->date_payment) . '] ' . $this->name;
    }

    public function scopeFilter($query, array $input = [])
    {
        $query->where(function ($query) {
            $query->where('company_id', '=', \App\Helpers\Helper::defaultCompany()->id);
        })->where(function($query) use ($input) {
            if (!empty($input['filter_search'])) {
                $search = $input['filter_search'];
                $query->orWhere('name', 'like', '%' . str_replace(' ', '%%', $search) . '%');
            }
            if (!empty($input['filter_name'])) {
                $name = $input['filter_name'];
                $query->where('name', 'like', '%' . str_replace(' ', '%%', $name) . '%');
            }
            if (!empty($input['filter_date_payment_from'])) {
                $query->whereDate('date_payment', '>=', Helper::convertDateToSql($input['filter_date_payment_from']));
            }
            if (!empty($input['filter_date_payment_to'])) {
                $query->whereDate('date_payment', '<=', Helper::convertDateToSql($input['filter_date_payment_to']));
            }
            if (!empty($input['filter_payroll_type_id'])) {
                $payroll_type_id = $input['filter_payroll_type_id'];
                $query->where('payroll_type_id', '=', $payroll_type_id);
            }
            if (!empty($input['filter_currency_id'])) {
                $currency_id = $input['filter_currency_id'];
                $query->where('currency_id', '=', $currency_id);
            }
            if (!empty($input['filter_branch_office_id'])) {
                $branch_office_id = $input['filter_branch_office_id'];
                $query->where('branch_office_id', '=', $branch_office_id);
            }
            if (!empty($input['filter_status'])) {
                $status = $input['filter_status'];
                $query->where('status', '=', $status);
            }
        });

        return $query;
    }

    /**
     * Solo registros activos
     *
     * @param $query
     * @return mixed
     */
    public function scopePopulateSelect($query)
    {
        $query->where('company_id', '=', \App\Helpers\Helper::defaultCompany()->id);
        $query->whereIn('status', [self::DRAFT, self::OPEN]);
        $query->orderBy('date_payment', 'desc');

        return $query;
    }

    public function payrollType()
    {
        return $this->belongsTo(PayrollType::class);
    }

    public function branchOffice()
    {
        return $this->belongsTo(BranchOffice::class);
    }

    public function currency()
    {
        return $this->belongsTo(Currency::class);
    }

    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    public function employeePayrolls()
    {
        return $this->hasMany(EmployeePayroll::class);
    }

    public function taxRegimen()
    {
        return $this->belongsTo(TaxRegimen::class);
    }
}
