<?php

namespace App\Services;


use App\Traits\ConsumesExternalServices;
use Illuminate\Http\Request;

class PaypalService
{
    use ConsumesExternalServices;

    protected $baseUri;

    protected $clientId;

    protected $clientSecret;

    public function __construct()
    {
        $this->baseUri = config('services.paypal.base_uri');
        $this->clientId = config('services.paypal.client_id');
        $this->clientSecret = config('services.paypal.client_secret');
    }

    /**
     * Resuelve autorizacion
     *
     * @param $queryParams
     * @param $formParams
     * @param $headers
     */
    public function resolveAuthorization(&$queryParams, &$formParams, &$headers)
    {
        $headers['Authorization'] = $this->resolveAccessToken();
    }

    /**
     * Decodifica la respuesta
     *
     * @param $response
     * @return mixed
     */
    public function decodeResponse($response)
    {
        return json_decode($response);
    }

    /**
     * Resuelve token
     *
     * @return string
     */
    public function resolveAccessToken()
    {
        $credentials = base64_encode($this->clientId . ':' . $this->clientSecret);

        return 'Basic ' . $credentials;
    }

    /**
     * Realiza la intencion de pago
     *
     * @param $value
     * @param $currency
     * @param $payment_platform_request
     * @return mixed
     * @throws \Exception
     */
    public function handlePayment($value, $currency, $payment_platform_request)
    {
        $order = $this->createOrder($value, $currency, $payment_platform_request);

        //Obtenemos los links
        $orderLinks = collect($order->links);
        $approve = $orderLinks->where('rel', 'approve')->first(); //Seleccionamos solo el de aprovacion
        if(empty($approve) || empty($approve->href) || empty($order->id)){
            throw new \Exception(__('shop/cart.error_payment_platform_paypal_link_approve'));
        }

        //Guardamos el Id de aprobacion en la solicitud
        $payment_platform_request['approvalId'] = $order->id;
        //Guardamos url para la aprovacion del pago
        $payment_platform_request['approve_href'] = $approve->href;

        //Retorna respuesta de intencion de pago
        return $payment_platform_request;
    }

    /**
     * Realiza la aprovacion del pago
     *
     * @param $payment_platform_request
     * @param $payment_platform_response
     * @return mixed
     */
    public function handleApproval($payment_platform_request, $payment_platform_response)
    {
        if (!empty($payment_platform_request['approvalId'])) {
            $approvalId = $payment_platform_request['approvalId'];

            //Obtenemos los datos del pago realizado
            $payment = $this->capturePayment($approvalId);
            $payment_platform_response['status'] = $payment->status ?? '';
            $payment_platform_response['payer'] = ($payment->payer->name->given_name ?? '') . ' ' . ($payment->payer->name->surname ?? '');
            $payment_platform_response['email_address'] = $payment->payer->email_address ?? '';
            $tmp_payment = $payment->purchase_units[0]->payments->captures[0]->amount ?? '';
            $payment_platform_response['amount'] = $tmp_payment->value ?? '';
            $payment_platform_response['currency'] = $tmp_payment->currency_code ?? '';

            return $payment_platform_response;
        }

        flash(__('shop/cart.error_payment_platform_paypal_approvalId'))->error();
    }

    /**
     * Crea intencion de pago
     *
     * @param $value
     * @param $currency
     * @param $payment_platform_request
     * @return mixed|\Psr\Http\Message\ResponseInterface|string
     */
    public function createOrder($value, $currency, &$payment_platform_request)
    {
        $purchase_units = [
            0 => [
                'amount' => [
                    'currency_code' => strtoupper($currency),
                    'value' => round($value * $factor = $this->resolveFactor($currency)) / $factor,
                ]
            ]
        ];
        $application_context = [
            'brand_name' => config('app.name'),
            'shipping_preference' => 'NO_SHIPPING',
            'user_action' => 'PAY_NOW',
            'return_url' => route('cart/success', ['id' => $payment_platform_request['folio_id']]),
            'cancel_url' => route('cart/checkout', ['folio_id' => $payment_platform_request['folio_id'],'pp_cancel' => true]),
        ];
        $payment_platform_request ['purchase_units'] = $purchase_units;
        $payment_platform_request ['application_context'] = $application_context;

        //Realiza la creacion de una orden de pago
        return $this->makeRequest(
            'POST',
            '/v2/checkout/orders',
            [],
            [
                'intent' => 'CAPTURE',
                'purchase_units' => $purchase_units,
                'application_context' => $application_context
            ],
            [],
            $isJsonRequest = true
        );
    }

    /**
     * Captura datos de la orden
     *
     * @param $approvalId
     * @return mixed|\Psr\Http\Message\ResponseInterface|string
     */
    public function capturePayment($approvalId)
    {
        return $this->makeRequest(
            'POST',
            "/v2/checkout/orders/{$approvalId}/capture",
            [],
            [],
            [
                'Content-Type' => 'application/json',
            ]
        );
    }

    public function resolveFactor($currency)
    {
        return 100;
    }
}