<?php

namespace App\Actions\Fortify;

use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Laravel\Fortify\Contracts\CreatesNewUsers;
use App\Models\Clientes;
use Spatie\Permission\Models\Role;

class CreateNewUser implements CreatesNewUsers
{
    use PasswordValidationRules;
    /**
     * Validate and create a newly registered user.
     *
     * @param  array<string, string>  $input
     */
    public function create(array $input): User
{
        // Validación de datos
        $validatedData = Validator::make($input, [
            'name' => ['required', 'string', 'max:255'],
            'email' => [
                'required',
                'string',
                'email',
                'max:255',
                Rule::unique(User::class),
            ],
            'password' => $this->passwordRules(),
            'terms' => ['required'],
            ],[
                'terms.required' => 'Debes aceptar los términos y condiciones.',
                'name.required' => 'El nombre es obligatorio para el registro.',
                'email.required' => 'El correo electrónico es obligatorio para el registro.',
                'password.required' => 'Ingrese su contraseña ya que obligatorio para el registro.',
            ])->validate();
        // Creación del usuario
        $user = User::create([
            'name' => htmlspecialchars($validatedData['name']),
            'email' => htmlspecialchars($validatedData['email']),
            'password' => htmlspecialchars(Hash::make($validatedData['password'])),
            'status' => 'Activo', // Usar constante o variable de entorno
        ]);
        // Generación de token y creación del cliente
        $token = $this->generateRandomToken();
        $cliente = Clientes::create([
            'codigo_clie' => $token,
            'usuario_clie' => $user->id,
            'estatus_clie' => 'Activo', // Usar constante o variable de entorno
        ]);
        // Asignación de rol al usuario (asumiendo que existe el rol "Cliente")
        $role = Role::findByName('Cliente');
        $user->assignRole($role);
        // Retorno del usuario creado
        return $user;
    }

// Función para generar token aleatorio
    private function generateRandomToken(): string
    {
        $permittedChars = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
        return "CLIE" . substr(str_shuffle($permittedChars), 0, 16);
    }
}
